/* Piper - the Swiss Army Knife of SOCKS/WinGate manipulation
 * Written by gcc@i.am (Chris Wilson) 16.1.99
 *
 * RCS ChangeLog:
 * $Log: modules.c,v $
 * Revision 1.4  2001/07/20 18:24:19  chris
 * Added bouncer support (finally!)
 * More generic multiplex code (re-used by bouncer)
 *
 * Revision 1.3  2001/07/14 21:16:08  chris
 * Removed limited distribution notice
 * Shortened timeout for socks_test to 5 seconds
 *
 * Revision 1.2  2001/07/14 15:10:03  chris
 * Removed limited distribution messages.
 * Licensed under GPL.
 *
 * Revision 1.1  1999/01/18 23:01:34  chris
 * Initial revision
 *
 */

#include "piper.h"

// Test SOCKS servers, and time connection to a target host.
void socks_test(int argc, char** argv) {
  char host[1024];
  char **servers = NULL, **pserver = NULL;
  char  *server = NULL;
  int   *proclist = NULL;
  int port, forking = 0;
  int verbose = V_ERR;
  int i, numservers, numprocs = 0, pid;

  if (strcmp(argv[0], "-v") == 0) {
    verbose = V_ALL;
    argv++;
    argc--;
  }

  if (strcmp(argv[0], "-f") == 0) {
    forking = 1;
    argv++;
    argc--;
  }

  if (argc != 2) {
    puts("Wrong number of arguments. Use 'piper -h' for help");
    exit(1);
  }

  if (argv[0][0] == '@') {
    servers = readhostlist(argv[0] + 1);
  } else {
    servers = makehostlist(argv[0]);
  }
  pserver = servers;

  split_host_port(argv[1], host, &port, "", 23);
  if (strcmp(host, "") == 0) {
    puts("Must specify remote host for 'piper -t'");
    exit(1);
  }

  if (forking) {
    /* avoid zombie children, by catching their return values */
    signal(SIGCHLD, SIG_IGN);
    /* setup some memory for the list of pids */
    numservers = counthostlist(servers);
    proclist = calloc(numservers+1, sizeof(pid_t));
    if (proclist == NULL) {
      perror("calloc()");
      exit(3);
    }
    /* setup the signal handler which kills our children when we die */
    signal(SIGTERM, sig_die);
    // signal(SIGABRT, sig_abort);
  }

  i = 0;

  do {
    server = *(pserver++);
    if (forking) {
      usleep(250000);
      pid = fork();

      if (pid < 0) {
	/* fork failed, so die quietly */
	perror("fork()");
	exit(2);
      } else if (pid == 0) {
	/* fork returns 0 in the child process, so that's what we are */
	testhost(server, host, port, verbose, -20);
	exit(0);
      } else {
	/* fork returned a positive value, so we're the parent */
	/* store the pid in our process list */
	proclist[numprocs++] = pid;
      }
    } else { /* not forking */
      testhost(server, host, port, verbose, +5);
    }
  } while (*pserver != NULL);

  for (i = 0; i < numprocs; i++) {
    if (proclist[i] > 0) {
      waitpid(proclist[i], NULL, 0);
    } else {
      printf("invalid value in proclist!\n");
      exit(3);
    }
  }

  signal(SIGCHLD, SIG_DFL);
  freehostlist(servers);
}



// Investigate Hosts - connects to each host in a list of hosts (on the
// specified port), read the first line from the daemon (if any), print it out
// and disconnect. Useful for identifying OSes and daemon versions
void socks_investigate(int argc, char** argv) {
  char **servers = NULL, **hosts = NULL;
  char **pserver = NULL, **phost = NULL;
  char  *server  = NULL,  *host  = NULL;
  int port, forking = 0, pid = 0, i = 0;
  int verbose = V_ERR;
  pid_t *proclist = NULL;
  int numhosts, numprocs = 0, status, forkdelay = 0;

  if (strcmp(argv[0], "-v") == 0) {
    verbose = V_ALL;
    argv++;
    argc--;
  }

  if (strcmp(argv[0], "-f") == 0) {
    forking = 1;
    argv++;
    argc--;
  }

  if (argc != 3) {
    puts("Wrong number of arguments. Use 'piper -h' for help");
    exit(1);
  }

  port = atoi(argv[2]);
  if (port < 0 || port > 65535) {
    printf("Invalid port specified [%s]: must be between 0 and 65535\n", 
	   argv[2]);
    exit(1);
  }

  if (argv[0][0] == '@') {
    servers = readhostlist(argv[0] + 1);
  } else {
    servers = makehostlist(argv[0]);
  }
  
  if (verbose >= V_ALL)
    printf("Loaded %i SOCKS servers\n", counthostlist(servers));
  pserver = servers;

  if (argv[1][0] == '@') {
    hosts = readhostlist(argv[1] + 1);
  } else {
    hosts = makehostlist(argv[1]);
  }

  numhosts = counthostlist(hosts);
  if (forking) {
    proclist = calloc(numhosts, sizeof(pid_t));
    if (proclist == NULL) {
      perror("calloc()");
      exit(3);
    }
    signal(SIGTERM, sig_die);
    signal(SIGCHLD, SIG_IGN);
    forkdelay = 10000000 / counthostlist(servers);
    /* allow 1 seconds per connection per SOCKS server */
    /* if the server refuses us (overloaded) we will auto retry */
  }
  if (verbose >= V_ALL)
    printf("Loaded %i hosts to investigate\n", numhosts);

  phost = hosts;

  do {
    if (*pserver == NULL) {
      // we ran out of servers, so start again with the first one
      pserver = servers;
    }

    host = *(phost++);
    server = *(pserver++);

    if (forking) {
      pid = fork();

      if (pid < 0) {
	/* fork() failed */
	perror("fork()");
	exit(2);
      } else if (pid == 0) {
	/* we are the child */
	do {
	  status = investigate_host(server, host, port, V_TEST);

	  server = *(pserver++);
	  if (*pserver == NULL) {
	    pserver = servers;
	  }
	} while (status == ST_UNKNOWN);
	exit(0);
      } else {
	/* sleep a while to avoid forkbombing - unless we're finished */
	if (*phost != NULL) usleep(forkdelay);
	proclist[numprocs++] = pid;
      }
    } else { /* not forking */
      do {
	if (numhosts > 1) {
	  status = investigate_host(server, host, port, V_TEST);
	} else { /* just one host, use long output format */
	  status = investigate_host(server, host, port, verbose);
	}

	server = *(pserver++);
	if (*pserver == NULL) {
	  // we ran out of servers, so start again with the first one
	  pserver = servers;
	}
      } while (status == ST_UNKNOWN);
    }
  } while (*phost != NULL);

  for (i = 0; i < numprocs; i++) {
    waitpid(proclist[i], NULL, 0);
  }

  freehostlist(servers);
  freehostlist(hosts);
  if (forking) free(proclist);
}



/*
 * socks_portscan
 * uses socks host(s) to connect to ports on remote system(s)
 */
void socks_portscan(int argc, char** argv) {
  char **servers = NULL, **hosts = NULL;
  char **pserver = NULL, **phost = NULL;
  char  *server  = NULL,  *host  = NULL;
  int   *ports   = NULL;
  int port = 0, numports = 0, forking = 0, pid = 0, i = 0;
  int verbose = V_ERR;
  pid_t *proclist = NULL; /* list of processes we forked off */
  int numprocs = 0;       /* number of processes we forked off */
  int forkdelay = 0, numhosts, status;

  if (strcmp(argv[0], "-v") == 0) {
    verbose = V_ALL;
    argv++;
    argc--;
  }

  if (strcmp(argv[0], "-f") == 0) {
    forking = 1;
    argv++;
    argc--;
  }

  if (argc < 2 || argc > 3) {
    puts("Wrong number of arguments. Use 'piper -h' for help");
    exit(1);
  }

  if (argv[0][0] == '@') {
    servers = readhostlist(argv[0] + 1);
  } else {
    servers = makehostlist(argv[0]);
  }

  if (verbose >= V_ALL)
    printf("Loaded %i SOCKS servers\n", counthostlist(servers));
  pserver = servers;

  if (argv[1][0] == '@') {
    hosts = readhostlist(argv[1] + 1);
  } else {
    hosts = makehostlist(argv[1]);
  }
  numhosts = counthostlist(hosts);

  if (forking) {
    proclist = calloc(counthostlist(hosts), numports*sizeof(pid_t));
    if (proclist == NULL) {
      perror("calloc()");
      exit(3);
    }
    signal(SIGCHLD, SIG_IGN);
    signal(SIGTERM, sig_die);
    forkdelay = 1000000 / counthostlist(servers);
    /* allow 1 second per SOCKS server per connection */
    /* if the server is too busy we will retry indefinitely */
  }

  if (verbose >= V_ALL)
    printf("Loaded %i hosts to portscan\n", numhosts);
  phost = hosts;
  host = *phost++;

  if (argc == 3) {
    /* if we were given a list of ports, comma-separated, then use it */
    numports = makeintlist(argv[2], &ports);
  } else {
    /* we weren't given any ports, so read from /etc/services */
    numports = readservices(&ports);
  }

  do {
    server = nexthost(&pserver, servers);

    if (i < numports) {
      port = ports[i++];
    } else {
      /* done all ports for this host, so start the next one */
      i = 0;
      host = *(phost++);
      if (host == NULL) continue; /* ran out of hosts, we're done */
    }

    if (forking) {
      pid = fork();

      if (pid < 0) {
	/* fork error */
	perror("fork()");
	exit(2);
      } else if (pid == 0) {
	/* we are the child */
	do {
	  status = portscan(server, host, port, V_TEST);
	  server = nexthost(&pserver, servers);
	} while (status == ST_UNKNOWN);
	exit(0);
      } else {
	/* we are the parent */
	/* sleep a while to avoid forkbombing - unless we're finished */
	if (*phost != NULL) usleep(forkdelay);
	proclist[numprocs++] = pid;
      }
    } else { /* not forking */
      do {
	if (numhosts > 1) 
	  status = portscan(server, host, port, V_TEST); /* short format */
	else
	  status = portscan(server, host, port, verbose); /* long format */
	server = nexthost(&pserver, servers);
      } while (status == ST_UNKNOWN);
    }
  } while (host != NULL);

  if (forking) {
    for (i = 0; i < numprocs; i++) {
      if (proclist[i] > 0) {
	waitpid(proclist[i], NULL, 0);
      } else {
	printf("invalid pid in proclist!\n");
	exit(2);
      }
    }
  }

  freehostlist(servers);
  freehostlist(hosts);
  free(ports);
  free(proclist);
}



void conn(int argc, char** argv) {
  char host[1024];
  int port, sock;
  int verbose = V_ERR;

  if (argc < 2 || argc > 3) {
    puts("Wrong number of arguments. Use 'piper -h' for help");
    exit(1);
  }

  if (argc == 3) {
    if (strcmp(argv[0], "-v") == 0) {
      verbose = V_ALL;
      argv++;
      argc--;
    } else {
      puts("Invalid command line, second parameter must be '-v' or a wingate");
      puts("Use 'piper -h' for help");
      exit(1);
    }
  }

  split_host_port(argv[1], host, &port, "", 23);
  if (strcmp(host, "") == 0) {
    puts("Must specify remote host for 'piper -c'");
    exit(1);
  }
  if (verbose >= V_ALL) {
    printf("Connecting to %s:%i (via %s:1080)...\n",
	   host, port, argv[0]);
  }

  sock = socks_connect(argv[0], host, port, verbose, -20);

  if (sock < 0) {
    exit(-sock);
  } else {
    printf("You are connected to %s.\n", host);
    multiplex_con(sock);
    exit(0);
  }
}



void socks_bouncer(int argc, char** argv) {
  // char localhost[1024], 
  char sockshost[1024], desthost[1024];
  int localport, socksport, destport, 
    listensock, insock, outsock, addrlen;
  int verbose = V_ERR;
  struct sockaddr_in localaddr, clientaddr;

  if (argc < 3 || argc > 4) {
    puts("Wrong number of arguments. Use 'piper -h' for help");
    exit(1);
  }

  if (argc == 4) {
    if (strcmp(argv[0], "-v") == 0) {
      verbose = V_ALL;
      argv++;
      argc--;
    } else {
      puts("Invalid command line, second parameter must be '-v' or a wingate");
      puts("Use 'piper -h' for help");
      exit(1);
    }
  }

  strncpy(sockshost, argv[0], 1024);
  socksport = 1080;

  split_host_port(argv[1], desthost, &destport, "", 23);
  if (strcmp(desthost, "") == 0) {
    puts("Must specify remote host for 'piper -b'");
    exit(1);
  }

  localport = atoi(argv[2]);
  if (localport < 1 || localport > 65535) {
    puts("Local port must be between 1 and 65535!");
    exit(1);
  }

  listensock = socket(PF_INET, SOCK_STREAM, 6);
  if (listensock < 0) {
    perror("Getting a socket");
    exit(1);
  }

  bzero(&localaddr, sizeof(localaddr));
  localaddr.sin_family = AF_INET;
  localaddr.sin_port = htons(localport);
  localaddr.sin_addr.s_addr = 0;

  if (bind(listensock, (struct sockaddr *)&localaddr, 
	   sizeof(localaddr))) {
    perror("Binding to local port");
    exit(1);
  }

  if (listen(listensock, 0) < 0) {
    perror("Listening on port");
    exit(1);
  }

  while (1) {
    printf("Listening for connection on port %i...\n", localport);

    addrlen = sizeof(clientaddr);
    insock = accept(listensock, (struct sockaddr *)&clientaddr, 
		    &addrlen);
    printf("Accepted connection from %s.\n", 
	   inet_ntoa( clientaddr.sin_addr ));
    
    if (verbose >= V_ALL) {
      printf("Connecting to %s:%i (via %s:%i)...\n",
	     desthost, destport, sockshost, socksport);
    }
  
    outsock = socks_connect(sockshost, desthost, destport, verbose, -20);

    if (outsock < 0) {
      printf("Connection failed!\n");
    } else {
      printf("Connected to %s.\n", desthost);
      multiplex_con_2(insock, outsock);
      puts("Connection closed.");
      close(insock);
      close(outsock);
    }
  }
}
