# `window-resized`

{{since('20210314-114017-04b7cedd')}}

The `window-resized` event is emitted when the window is resized and when
transitioning between full-screen and regular windowed mode.

The event is triggered asynchronously with respect to the potentially-ongoing
live resize operation.  `wezterm` will coalesce the stream of multiple events
generated by a live resize such that there can be a maximum of 1 event
executing and 1 event buffered.

The first event parameter is a [`window` object](../window/index.md) that
represents the gui window.

The second event parameter is a [`pane` object](../pane/index.md) that
represents the active pane in that window.

This example demonstrates adjusting the window padding when going fullscreen
so that the terminal content fits in the middle third of the display.

Note how care is taken to avoid calling `window:set_config_overrides` unless
something has changed.  Also note how both the `window-resized` and
[window-config-reloaded](window-config-reloaded.md) events are connected to the
same `recompute_padding` function; that causes the padding changes to take
effect both when toggling fullscreen and when editing the config file to adjust
the event handling code:

```lua
local wezterm = require 'wezterm'

function recompute_padding(window)
  local window_dims = window:get_dimensions()
  local overrides = window:get_config_overrides() or {}

  if not window_dims.is_full_screen then
    if not overrides.window_padding then
      -- not changing anything
      return
    end
    overrides.window_padding = nil
  else
    -- Use only the middle 33%
    local third = math.floor(window_dims.pixel_width / 3)
    local new_padding = {
      left = third,
      right = third,
      top = 0,
      bottom = 0,
    }
    if
      overrides.window_padding
      and new_padding.left == overrides.window_padding.left
    then
      -- padding is same, avoid triggering further changes
      return
    end
    overrides.window_padding = new_padding
  end
  window:set_config_overrides(overrides)
end

wezterm.on('window-resized', function(window, pane)
  recompute_padding(window)
end)

wezterm.on('window-config-reloaded', function(window)
  recompute_padding(window)
end)
```
