/*
 * Copyright (C) 2002 Paul Martin <pm@debian.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include "lcd.h"

#define MIN(a,b) ((a)>(b) ? (b) : (a))

/* 1.5 seconds ish */
#define DEBOUNCE_SELECT 150
#define DEBOUNCE_OTHER  5

#define LCDDEV "/dev/lcd"

void lcd_op(const int operation,
	    struct lcd_display *disp, const char *error)
{
    int fd, rc;
    
    fd = open(LCDDEV, O_RDWR);
    if (fd == -1) {
	perror(error);
	exit(1);
    }
    rc = ioctl(fd, operation, disp);
    close(fd);
    if (rc) {
	perror(error);
	exit(1);
    }

#if __i386__
    /* Don't apply this to the Qube2 or Raq2 */
    if (operation == BUTTON_Read) {
	disp->buttons &= BUTTON_MASK;
    }
#endif

}

void wait_for_select(void (*idle_task)(void))
{
    struct lcd_display display;
    int bouncecount = 0;
    struct timespec tv;

    tv.tv_sec = 0;
    tv.tv_nsec = 10*1000*1000; /* 10ms ish */
    
    while (1) {
	lcd_op(BUTTON_Read, &display, "Reading Buttons");

	if ((display.buttons != BUTTON_Next)
	    && (display.buttons != BUTTON_Next_B)) {
	    if (idle_task != NULL) idle_task();
	    sleep(1);
	    bouncecount = 0;
	} else {
	    bouncecount ++;
	    if (bouncecount >= DEBOUNCE_SELECT) break;
	    nanosleep(&tv, NULL);
	}
    }
}

unsigned long wait_for_button(void)
{
    struct lcd_display display;
    int bouncecount = 0;
    time_t tim;
    struct timespec tv;

    tv.tv_sec = 0;
    tv.tv_nsec = 10*1000*1000; /* 10ms ish */
    
    tim = time(NULL) + 60;
    while (tim > time(NULL)) {
	lcd_op(BUTTON_Read, &display, "Reading Buttons");

	if ((display.buttons == BUTTON_NONE)
	    || (display.buttons == BUTTON_NONE_B)) {
	    bouncecount = 0;
	} else {
	    bouncecount ++;
	    if (bouncecount >= DEBOUNCE_OTHER)
	      return(display.buttons);
	}
	nanosleep(&tv, NULL);
    }
    return(0); /* for timeout */
}

void wait_for_no_button(void)
{
    struct lcd_display display;
    int bouncecount = 0;
    struct timespec tv;

    tv.tv_sec = 0;
    tv.tv_nsec = 10*1000*1000; /* 10ms ish */
    
    while (1) {
	lcd_op(BUTTON_Read, &display, "Reading Buttons");

	if ((display.buttons == BUTTON_NONE)
	    || (display.buttons == BUTTON_NONE_B)) {
	    bouncecount ++;
	    if (bouncecount >= DEBOUNCE_OTHER)
	      break;
	} else {
	    bouncecount = 0;
	}
	nanosleep(&tv, NULL);
    }
}

void put_lcd(const char *line1, const char *line2)
{
    struct lcd_display display;
    
    memset(&display, 0, sizeof(display));
    memset(display.line1, ' ', sizeof(display.line1));
    memset(display.line2, ' ', sizeof(display.line2));
    
    if (line1 != NULL) {
	memcpy(display.line1, line1,
	       MIN(sizeof(display.line1), strlen(line1)));
	display.size1 = sizeof(display.line1);
    }

    if (line2 != NULL) {
	memcpy(display.line2, line2,
	       MIN(sizeof(display.line2), strlen(line2)));
	display.size2 = sizeof(display.line2);
    }
    
    lcd_op(LCD_Write, &display, "Writing LCD");
}

void save_lcd(struct lcd_display *display)
{
    memset(display, 0, sizeof(*display));
    lcd_op(LCD_Read, display, "Saving LCD");
    display->size1 = sizeof(display->line1);
    display->size2 = sizeof(display->line2);
}

void restore_lcd(struct lcd_display *display)
{
    lcd_op(LCD_Write, display, "Restoring LCD");
}

void set_cursor(int address)
{
    struct lcd_display display;
    
    memset(&display, 0, sizeof(display));
    display.cursor_address = address;
    lcd_op(LCD_Set_Cursor_Pos, &display,"Set Cursor Pos");
    lcd_op(LCD_Cursor_On, &display,"Cursor On");
}

void unset_cursor(void)
{
    struct lcd_display display;
        
    memset(&display, 0, sizeof(display));
    lcd_op(LCD_Cursor_Off, &display,"Cursor Off");
}

const char yesno[] =  "  [Y]es  [N]o";
const unsigned char
  cursors[] = { 0x40 + 10, 0x40 + 3 };


int ask_yes_no(const char *question) 
{
    int state = 0; /* No */
    int selected = 0;
    unsigned long button;
    
    put_lcd(question,yesno);
    while (!selected) {
	set_cursor(cursors[state]);
	wait_for_no_button();
	button=wait_for_button();
	switch (button) {
	 case BUTTON_Enter:
	 case BUTTON_Enter_B:
	    selected = 1;
	    break;
	 case BUTTON_Left:
	 case BUTTON_Left_B:
	 case BUTTON_Right:
	 case BUTTON_Right_B:
	    state = !state;
	    break;
	 case 0: /* timeout */
	    state = 0; /* no */
	    selected = 1;
	    break;
	}
    }
    unset_cursor();
    return (state);
}

