










package com.adacore.gpr_parser;

import java.util.Arrays;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.Objects;

import java.lang.StringBuilder;
import java.lang.Iterable;
import java.lang.reflect.Method;

import java.math.BigInteger;

import java.io.File;
import java.nio.ByteOrder;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;

import com.oracle.truffle.api.CompilerDirectives;
import com.oracle.truffle.api.strings.TruffleString;

import org.graalvm.nativeimage.CurrentIsolate;
import org.graalvm.nativeimage.ImageInfo;
import org.graalvm.nativeimage.IsolateThread;
import org.graalvm.nativeimage.StackValue;
import org.graalvm.nativeimage.UnmanagedMemory;
import org.graalvm.nativeimage.c.CContext;
import org.graalvm.nativeimage.c.function.CEntryPoint;
import org.graalvm.nativeimage.c.function.CEntryPointLiteral;
import org.graalvm.nativeimage.c.function.CFunction;
import org.graalvm.nativeimage.c.function.CFunctionPointer;
import org.graalvm.nativeimage.c.function.InvokeCFunctionPointer;
import org.graalvm.nativeimage.c.struct.*;
import org.graalvm.nativeimage.c.type.*;
import org.graalvm.word.PointerBase;
import org.graalvm.word.Pointer;
import org.graalvm.word.UnsignedWord;
import org.graalvm.word.WordFactory;


/*
====================

This is the Java bindings of the gpr_parser API.
You can use all functionalities of the library in a Java environment.
Those bindings call the native library using JNI and Native-Image C API.

====================
*/
public final class GprParser {

    // ==========
    // Native entry points
    // ==========

    /**
     * This method is the only valid callback to pass to a native event
     * handler for unit requests events.
     * This method will dispatch the execution according to the passed
     * analysis context.
     */
    @CEntryPoint
    static void unitRequested(
        final IsolateThread thread,
        final AnalysisContextNative contextNative,
        final TextNative nameNative,
        final AnalysisUnitNative fromNative,
        final byte foundNative,
        final byte isNotFoundErrorNative
    ) {
        try(
            final AnalysisContext context = AnalysisContext.wrap(
                contextNative
            );
            final Text text = Text.wrap(nameNative)
        ) {
            // Get the callback from the context event handler
            final EventHandler.UnitRequestedCallback callback = context
                .getEventHandler()
                .getUnitRequestCallback();

            // Call the callback
            if(callback != null) {
                callback.invoke(
                    context,
                    text.getContent(),
                    AnalysisUnit.wrap(fromNative),
                    foundNative != 0,
                    isNotFoundErrorNative != 0
                );
            }
        }
    }

    /**
     * This method is the only valid callback to pass to a native event
     * handler for unit parsing events.
     * This method will dispatch the execution according to the passed
     * analysis context.
     */
    @CEntryPoint
    static void unitParsed(
        final IsolateThread thread,
        final AnalysisContextNative contextNative,
        final AnalysisUnitNative unitNative,
        final byte reparsedNative
    ) {
        try(
            final AnalysisContext context = AnalysisContext.wrap(
                contextNative
            )
        ) {
            // Get the callback from the context event handler
            final EventHandler.UnitParsedCallback callback = context
                .getEventHandler()
                .getUnitParsedCallback();

            // Call the callback
            if(callback != null) {
                callback.invoke(
                    context,
                    AnalysisUnit.wrap(unitNative),
                    reparsedNative != 0
                );
            }
        }
    }

    // ==========
    // Static values
    // ==========

    /** The default grammar rule to parse the inputs. */
    private static final GrammarRule DEFAULT_GRAMMAR_RULE =
        GrammarRule.COMPILATION_UNIT_RULE;

    /** The os name in lower case. */
    private static final String OS =
            System.getProperty("os.name").toLowerCase();

    /** The byte order of the system. */
    private static final ByteOrder BYTE_ORDER = ByteOrder.nativeOrder();

    /** The node to convert a Java string to a truffle string */
    private static final TruffleString.FromJavaStringNode fromJavaStringNode =
        TruffleString.FromJavaStringNode.create();

    /** The node to convert a truffle string to a Java string. */
    private static final TruffleString.ToJavaStringNode toJavaStringNode =
        TruffleString.ToJavaStringNode.create();

    /** The node to convert a byte array to a truffle string. */
    private static final TruffleString.FromByteArrayNode fromByteArrayNode =
        TruffleString.FromByteArrayNode.create();

    /** The node to convert a truffle string to a byte array. */
    private static final TruffleString.CopyToByteArrayNode toByteArrayNode =
        TruffleString.CopyToByteArrayNode.create();

    /** A map to store the nodes classes from their name. */
    public static final Map<String, Class<? extends GprNode>>
        NODE_CLASS_MAP = new HashMap<>();

    static {
        // Populate the node class map
        NODE_CLASS_MAP.put(
            "GprNode",
            GprNode.class
        );
        NODE_CLASS_MAP.put(
            "AllQualifier",
            AllQualifier.class
        );
        NODE_CLASS_MAP.put(
            "AllQualifierAbsent",
            AllQualifierAbsent.class
        );
        NODE_CLASS_MAP.put(
            "AllQualifierPresent",
            AllQualifierPresent.class
        );
        NODE_CLASS_MAP.put(
            "AttributeDecl",
            AttributeDecl.class
        );
        NODE_CLASS_MAP.put(
            "AttributeReference",
            AttributeReference.class
        );
        NODE_CLASS_MAP.put(
            "BaseList",
            BaseList.class
        );
        NODE_CLASS_MAP.put(
            "CaseItemList",
            CaseItemList.class
        );
        NODE_CLASS_MAP.put(
            "GprNodeList",
            GprNodeList.class
        );
        NODE_CLASS_MAP.put(
            "Choices",
            Choices.class
        );
        NODE_CLASS_MAP.put(
            "TermList",
            TermList.class
        );
        NODE_CLASS_MAP.put(
            "IdentifierList",
            IdentifierList.class
        );
        NODE_CLASS_MAP.put(
            "StringLiteralList",
            StringLiteralList.class
        );
        NODE_CLASS_MAP.put(
            "TermListList",
            TermListList.class
        );
        NODE_CLASS_MAP.put(
            "WithDeclList",
            WithDeclList.class
        );
        NODE_CLASS_MAP.put(
            "BuiltinFunctionCall",
            BuiltinFunctionCall.class
        );
        NODE_CLASS_MAP.put(
            "CaseConstruction",
            CaseConstruction.class
        );
        NODE_CLASS_MAP.put(
            "CaseItem",
            CaseItem.class
        );
        NODE_CLASS_MAP.put(
            "CompilationUnit",
            CompilationUnit.class
        );
        NODE_CLASS_MAP.put(
            "EmptyDecl",
            EmptyDecl.class
        );
        NODE_CLASS_MAP.put(
            "Expr",
            Expr.class
        );
        NODE_CLASS_MAP.put(
            "Prefix",
            Prefix.class
        );
        NODE_CLASS_MAP.put(
            "SingleTokNode",
            SingleTokNode.class
        );
        NODE_CLASS_MAP.put(
            "Identifier",
            Identifier.class
        );
        NODE_CLASS_MAP.put(
            "NumLiteral",
            NumLiteral.class
        );
        NODE_CLASS_MAP.put(
            "StringLiteral",
            StringLiteral.class
        );
        NODE_CLASS_MAP.put(
            "LimitedNode",
            LimitedNode.class
        );
        NODE_CLASS_MAP.put(
            "LimitedAbsent",
            LimitedAbsent.class
        );
        NODE_CLASS_MAP.put(
            "LimitedPresent",
            LimitedPresent.class
        );
        NODE_CLASS_MAP.put(
            "OthersDesignator",
            OthersDesignator.class
        );
        NODE_CLASS_MAP.put(
            "PackageDecl",
            PackageDecl.class
        );
        NODE_CLASS_MAP.put(
            "PackageExtension",
            PackageExtension.class
        );
        NODE_CLASS_MAP.put(
            "PackageRenaming",
            PackageRenaming.class
        );
        NODE_CLASS_MAP.put(
            "PackageSpec",
            PackageSpec.class
        );
        NODE_CLASS_MAP.put(
            "Project",
            Project.class
        );
        NODE_CLASS_MAP.put(
            "ProjectDeclaration",
            ProjectDeclaration.class
        );
        NODE_CLASS_MAP.put(
            "ProjectExtension",
            ProjectExtension.class
        );
        NODE_CLASS_MAP.put(
            "ProjectQualifier",
            ProjectQualifier.class
        );
        NODE_CLASS_MAP.put(
            "ProjectQualifierAbstract",
            ProjectQualifierAbstract.class
        );
        NODE_CLASS_MAP.put(
            "ProjectQualifierAggregate",
            ProjectQualifierAggregate.class
        );
        NODE_CLASS_MAP.put(
            "ProjectQualifierAggregateLibrary",
            ProjectQualifierAggregateLibrary.class
        );
        NODE_CLASS_MAP.put(
            "ProjectQualifierConfiguration",
            ProjectQualifierConfiguration.class
        );
        NODE_CLASS_MAP.put(
            "ProjectQualifierLibrary",
            ProjectQualifierLibrary.class
        );
        NODE_CLASS_MAP.put(
            "ProjectQualifierStandard",
            ProjectQualifierStandard.class
        );
        NODE_CLASS_MAP.put(
            "StringLiteralAt",
            StringLiteralAt.class
        );
        NODE_CLASS_MAP.put(
            "Terms",
            Terms.class
        );
        NODE_CLASS_MAP.put(
            "TypeReference",
            TypeReference.class
        );
        NODE_CLASS_MAP.put(
            "TypedStringDecl",
            TypedStringDecl.class
        );
        NODE_CLASS_MAP.put(
            "VariableDecl",
            VariableDecl.class
        );
        NODE_CLASS_MAP.put(
            "VariableReference",
            VariableReference.class
        );
        NODE_CLASS_MAP.put(
            "WithDecl",
            WithDecl.class
        );
    }

    // ==========
    // Util functions
    // ==========

    /**
     * Get the string representing the memory.
     *
     * @param pointer The pointer to start displaying the memory from.
     * @param count The number of bytes to display from the pointer.
     * @return The string representing the memory as hex bytes.
     */
    private static String dumpMemory(
        final Pointer pointer,
        final long count
    ) {
        final StringBuilder res = new StringBuilder();
        for(int i = 0 ; i < count ; i++) {
            final byte toDump = pointer.readByte(i);
            res.append(String.format("%02x ", toDump));
        }
        return res.toString();
    }

    /**
     * Convert a Java string to a C string by allocating memory.
     *
     * @param jString The Java string to convert.
     * @return The native C char pointer. This pointer MUST be freed.
     */
    @CompilerDirectives.TruffleBoundary
    private static CCharPointer toCString(
        final String jString
    ) {
        final UnsignedWord size = WordFactory.unsigned(jString.length() + 1);
        final CCharPointer res = UnmanagedMemory.calloc(size);
        if(jString.length() > 0) {
            CTypeConversion.toCString(
                jString,
                StandardCharsets.UTF_8,
                res,
                size
            );
        }

        return res;
    }

    /**
     * Convert a Java byte array in a native byte array.
     *
     * @param bytes The Java bytes.
     * @return The native C char pointer. This pointer MUST be freed.
     */
    private static CCharPointer toCBytes(
        final byte[] bytes
    ) {
        final UnsignedWord size = WordFactory.unsigned(bytes.length);
        final CCharPointer res = UnmanagedMemory.malloc(size);
        for(int i = 0 ; i < bytes.length ; i++) {
            res.write(i, bytes[i]);
        }
        return res;
    }

    /**
     * Convert a native-image C string to a Java string.
     *
     * @param pointer The char pointer to convert to a Java string.
     * @return The Java string.
     */
    @CompilerDirectives.TruffleBoundary
    private static String toJString(
        final CCharPointer pointer
    ) {
        return CTypeConversion.toJavaString(pointer);
    }

    /**
     * This function decode a utf 32 int array in a
     * string without calling Java charset.
     *
     * @param chars The int array to decode.
     * @return The resulting string.
     */
    @CompilerDirectives.TruffleBoundary
    private static String decodeUTF32(
        final byte[] toDecode
    ) {
        return toJavaStringNode.execute(
            fromByteArrayNode.execute(toDecode, TruffleString.Encoding.UTF_32)
        );
    }

    /**
     * This function encode a given string to a int array
     * according to the utf 32 standard.
     *
     * @param toEncode The string to encode.
     * @return The encoded string in an int array.
     */
    @CompilerDirectives.TruffleBoundary
    private static byte[] encodeUTF32(
        final String toEncode
    ) {
        return toByteArrayNode.execute(
            fromJavaStringNode.execute(
                toEncode,
                TruffleString.Encoding.UTF_32
            ),
            TruffleString.Encoding.UTF_32
        );
    }

    /**
     * Get the string representation of the given string.
     * This function escaped needed chars and format the string.
     *
     * @param source The source string to get the representation for.
     * @return The representation of the string.
     */
    private static String stringRepresentation(
        final String source
    ) {
        return source
            .replace("\"", "\\\"")
            .replace("\n", "\\x0a");
    }

    /**
     * Convert a C Langkit exception to the LangkitException class.

     * @param
     */
    private static LangkitException wrapException(
        final LangkitExceptionNative exc
    ) {
        return new LangkitException(
            exc.get_kind(),
            toJString(exc.get_information())
        );
    }

    /**
      * Return the exception raised by the last C API call, or null if the last
      * call was successful.
      */
    private static LangkitException getLastException() {
        LangkitException result = null;

        if(ImageInfo.inImageCode()) {
            final LangkitExceptionNative exceptionNative =
                NI_LIB.gpr_get_last_exception();
            if(exceptionNative.isNonNull()) {
                result = wrapException(exceptionNative);
            }
        } else {
            result = JNI_LIB.gpr_get_last_exception();
        }
        return result;
    }

    /**
     * Check the last exception raised by langkit and throw it.
     *
     * @throws The last langkit exception if there is one.
     */
    private static void checkException() throws LangkitException {
        LangkitException exc = getLastException();
        if(exc != null)
            throw exc;
    }

    // ==========
    // Util interfaces
    // ==========

    /**
     * Interface to visit the AST.
     */
    public static interface BasicVisitor<T> {
        T visit(GprNode node);
        T visit(AllQualifierAbsent node);
        T visit(AllQualifierPresent node);
        T visit(AttributeDecl node);
        T visit(AttributeReference node);
        T visit(CaseItemList node);
        T visit(GprNodeList node);
        T visit(Choices node);
        T visit(TermList node);
        T visit(IdentifierList node);
        T visit(StringLiteralList node);
        T visit(TermListList node);
        T visit(WithDeclList node);
        T visit(BuiltinFunctionCall node);
        T visit(CaseConstruction node);
        T visit(CaseItem node);
        T visit(CompilationUnit node);
        T visit(EmptyDecl node);
        T visit(Prefix node);
        T visit(Identifier node);
        T visit(NumLiteral node);
        T visit(StringLiteral node);
        T visit(LimitedAbsent node);
        T visit(LimitedPresent node);
        T visit(OthersDesignator node);
        T visit(PackageDecl node);
        T visit(PackageExtension node);
        T visit(PackageRenaming node);
        T visit(PackageSpec node);
        T visit(Project node);
        T visit(ProjectDeclaration node);
        T visit(ProjectExtension node);
        T visit(ProjectQualifierAbstract node);
        T visit(ProjectQualifierAggregate node);
        T visit(ProjectQualifierAggregateLibrary node);
        T visit(ProjectQualifierConfiguration node);
        T visit(ProjectQualifierLibrary node);
        T visit(ProjectQualifierStandard node);
        T visit(StringLiteralAt node);
        T visit(Terms node);
        T visit(TypeReference node);
        T visit(TypedStringDecl node);
        T visit(VariableDecl node);
        T visit(VariableReference node);
        T visit(WithDecl node);
    }

    /**
     * Interface to visit the AST with a parameter.
     */
    public static interface ParamVisitor<T, P> {
        T visit(GprNode node, P param);
        T visit(AllQualifierAbsent node, P param);
        T visit(AllQualifierPresent node, P param);
        T visit(AttributeDecl node, P param);
        T visit(AttributeReference node, P param);
        T visit(CaseItemList node, P param);
        T visit(GprNodeList node, P param);
        T visit(Choices node, P param);
        T visit(TermList node, P param);
        T visit(IdentifierList node, P param);
        T visit(StringLiteralList node, P param);
        T visit(TermListList node, P param);
        T visit(WithDeclList node, P param);
        T visit(BuiltinFunctionCall node, P param);
        T visit(CaseConstruction node, P param);
        T visit(CaseItem node, P param);
        T visit(CompilationUnit node, P param);
        T visit(EmptyDecl node, P param);
        T visit(Prefix node, P param);
        T visit(Identifier node, P param);
        T visit(NumLiteral node, P param);
        T visit(StringLiteral node, P param);
        T visit(LimitedAbsent node, P param);
        T visit(LimitedPresent node, P param);
        T visit(OthersDesignator node, P param);
        T visit(PackageDecl node, P param);
        T visit(PackageExtension node, P param);
        T visit(PackageRenaming node, P param);
        T visit(PackageSpec node, P param);
        T visit(Project node, P param);
        T visit(ProjectDeclaration node, P param);
        T visit(ProjectExtension node, P param);
        T visit(ProjectQualifierAbstract node, P param);
        T visit(ProjectQualifierAggregate node, P param);
        T visit(ProjectQualifierAggregateLibrary node, P param);
        T visit(ProjectQualifierConfiguration node, P param);
        T visit(ProjectQualifierLibrary node, P param);
        T visit(ProjectQualifierStandard node, P param);
        T visit(StringLiteralAt node, P param);
        T visit(Terms node, P param);
        T visit(TypeReference node, P param);
        T visit(TypedStringDecl node, P param);
        T visit(VariableDecl node, P param);
        T visit(VariableReference node, P param);
        T visit(WithDecl node, P param);
    }

    // ==========
    // Util classes
    // ==========

    /**
     * This class represents a pointer and can hold NI and JNI addresses.
     */
    public static final class PointerWrapper {

        // ----- Instance attributes -----

        /** The pointer NI value. */
        private PointerBase ni;

        /** The pointer JNI value. */
        private final long jni;

        // ----- Constructors -----

        /**
         * Create a new custom pointer from a NI pointer based value.
         *
         * @param niPointer The pointer based value.
         */
        PointerWrapper(
            final PointerBase niPointer
        ) {
            this.ni = niPointer;
            this.jni = -1;
        }

        /**
         * Create a new custom pointer from a long value.
         *
         * @param jniPointer The pointer in a long value.
         */
        PointerWrapper(
            final long jniPointer
        ) {
            this.jni = jniPointer;
        }

        /**
         * Wrap the given NI pointer in the Java class.
         *
         * @param niPointer The NI pointer to wrap.
         * @return The wrapped pointer.
         */
        static PointerWrapper wrap(
            final PointerBase niPointer
        ) {
            return new PointerWrapper(niPointer);
        }

        /**
         * Get the null pointer according to the execution mode.
         *
         * @return The null custom pointer.
         */
        public static PointerWrapper nullPointer() {

            if(ImageInfo.inImageCode()) {
                return new PointerWrapper(WordFactory.nullPointer());
            } else {
                return new PointerWrapper(0L);
            }

        }

        // ----- Instance methods -----

        /**
         * Get the pointer as an NI pointer based value.
         *
         * @return The pointer based value for NI.
         */
        public <T extends PointerBase> T ni() {
            return (T) this.ni;
        }

        /**
         * Get the pointer as a long Java value.
         *
         * @return The pointer as a long value for JNI.
         */
        public long jni() {
            return this.jni;
        }

        /**
         * Get if the pointer is null.
         *
         * @return True if the pointer is null, false else.
         */
        public boolean isNull() {

            if(ImageInfo.inImageCode()) {
                return this.ni.isNull();
            } else {
                return this.jni == 0;
            }

        }

        // ----- Override methods -----

        @Override
        public String toString() {

            if(ImageInfo.inImageCode()) {
                return "PointerWrapper{"
                    + this.ni.rawValue()
                    + "}";
            } else {
                return "PointerWrapper{"
                    + this.jni
                    + "}";
            }

        }

        @Override
        public boolean equals(Object o) {
            if(o == this) return true;
            if(!(o instanceof PointerWrapper)) return false;
            final PointerWrapper other = (PointerWrapper) o;
            if(ImageInfo.inImageCode()) {
                return this.ni.equal(other.ni);
            } else {
                return this.jni == other.jni;
            }
        }

        @Override
        public int hashCode() {

            if(ImageInfo.inImageCode()) {
                return (int) this.ni.rawValue();
            } else {
                return (int) this.jni;
            }

        }

    }

    /**
     * This class represents the description of a node field.
     */
    public static final class GprParserField {

        // ----- Instance attributes -----

        /** The Java method for the field */
        public final Method javaMethod;

        /** The parameters of the method */
        public final List<Param> params;

        // ----- Constructors -----

        /**
         * Create a new field description.
         *
         * @param method The Java method to access the field.
         * @param params The parameters of the field call.
         */
        public GprParserField(
            final Method javaMethod,
            final List<Param> params
        ) {
            this.javaMethod = javaMethod;
            this.params = params;
        }

    }

    /**
     * This class represents a parameter description.
     */
    public static class Param {

        // ----- Instance attributes -----

        /** The type of the argument */
        public final Class<?> type;

        /** The name of the parameter */
        public final String name;

        // ----- Constructors -----

        /**
         * Create a new langkit parameter.
         *
         * @param type The type of the parameter.
         * @param name The name of the parameter.
         */
        public Param(
            final Class<?> type,
            final String name
        ) {
            this.type = type;
            this.name = name;
        }

    }

    /**
     * This class represents an parameter in a langkit function description
     * which has a default value.
     */
    public static final class ParamWithDefaultValue extends Param {

        // ----- Instance attributes -----

        /** The default value of the parameter */
        public final Object defaultValue;

        // ----- Constructors -----

        /**
         * Create a new langkit parameter.
         *
         * @param type The type of the parameter.
         * @param name The name of the parameter.
         * @param defaultValue The default value of the parameter.
         */
        public ParamWithDefaultValue(
            final Class<?> type,
            final String name,
            final Object defaultValue
        ) {
            super(type, name);
            this.defaultValue = defaultValue;
        }

    }

    // ==========
    // Language specific extensions
    // ==========

    


    // ==========
    // Defining the JNI bindings library
    // ==========

    
    
    
    
    
    
    

    /** This class contains all native function definitions for JNI */
    public static final class JNI_LIB {

        // ----- Static initializer -----

        static {
            if(!ImageInfo.inImageCode()) {
                // Load the needed libraries
                if(OS.indexOf("win") < 0) {
                    System.loadLibrary("langkit_sigsegv_handler");
                }
                System.loadLibrary("gprlang_jni");

                // Initialize the JNI library
                gpr_initialize();

                // Register the library finalizer
                Runtime.getRuntime().addShutdownHook(
                    new Thread(JNI_LIB::gpr_finalize)
                );
            }
        }

        // ----- Language specific functions -----

        


        // ----- Lifecycle functions ------

        /** Function to initialize the JNI library */
        public static native void gpr_initialize();

        /** Function to finalize the JNI library */
        public static native void gpr_finalize();

        // ----- Exception functions ------

        /** Get the last langkit exception */
        @CompilerDirectives.TruffleBoundary
        public static native LangkitException gpr_get_last_exception();

        // ----- Text functions -----

        /** Create a new text from its content */
        @CompilerDirectives.TruffleBoundary
        public static native Text gpr_create_text(
            byte[] utf32Content
        );

        /** Destroy the given text */
        @CompilerDirectives.TruffleBoundary
        public static native void gpr_destroy_text(
            Text text
        );

        // ----- File reader functions -----

        /** Decrease the reference counter of the given file reader */
        @CompilerDirectives.TruffleBoundary
        public static native void gpr_dec_ref_file_reader(
            FileReader fileReader
        );

        // ----- Unit provider functions -----

        /** Decrease the ref counter of the unit provider */
        @CompilerDirectives.TruffleBoundary
        public static native void gpr_dec_ref_unit_provider(
            UnitProvider unitProvider
        );

        // ----- Event handler functions -----

        /** Create a new event handler */
        @CompilerDirectives.TruffleBoundary
        public static native PointerWrapper gpr_create_event_handler(
            EventHandler.UnitRequestedCallback unitRequestedCallback,
            EventHandler.UnitParsedCallback unitParsedCallback
        );

        /** Decrease the ref counter of the event handler */
        @CompilerDirectives.TruffleBoundary
        public static native void gpr_dec_ref_event_handler(
            EventHandler eventHandler
        );

        // ----- Token functions -----

        /** Get the next token */
        @CompilerDirectives.TruffleBoundary
        public static native Token gpr_token_next(
            Token token
        );

        /** Get the previous token */
        @CompilerDirectives.TruffleBoundary
        public static native Token gpr_token_previous(
            Token token
        );

        /** Get if the tokens are equivalent */
        @CompilerDirectives.TruffleBoundary
        public static native boolean gpr_token_is_equivalent(
            Token left,
            Token right
        );

        /** Get text between the two tokens */
        @CompilerDirectives.TruffleBoundary
        public static native Text gpr_token_range_text(
            Token start,
            Token end
        );

        // ----- Analysis context functions -----

        /** Create a new analysis context */
        @CompilerDirectives.TruffleBoundary
        public static native PointerWrapper gpr_create_analysis_context(
            String charset,
            FileReader fileReader,
            UnitProvider unitProvider,
            EventHandler eventHandler,
            boolean withTrivia,
            int tabstop
        );

        /** Increase the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        public static native void gpr_context_incref(
            long context
        );

        /** Decrease the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        public static native void gpr_context_decref(
            long context
        );

        // ----- Analysis unit functions -----

        /** Get the analysis unit from a file */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit
        gpr_get_analysis_unit_from_file(
            AnalysisContext context,
            String fileName,
            String charset,
            boolean reparse,
            int grammarRule
        );

        /** Get the analysis unit from a buffer */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit
        gpr_get_analysis_unit_from_buffer(
            AnalysisContext context,
            String fileName,
            String charset,
            String buffer,
            long bufferSize,
            int grammarRule
        );


        /** Get the root of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_unit_root(
            AnalysisUnit unit
        );

        /** Get the file name of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native String gpr_unit_filename(
            AnalysisUnit unit
        );

        /** Get the token count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native int gpr_unit_token_count(
            AnalysisUnit unit
        );

        /** Get the trivia count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native int gpr_unit_trivia_count(
            AnalysisUnit unit
        );

        /** Get the first token of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Token gpr_unit_first_token(
            AnalysisUnit unit
        );

        /** Get the last token of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Token gpr_unit_last_token(
            AnalysisUnit unit
        );

        /** Get the context of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisContext gpr_unit_context(
            AnalysisUnit unit
        );

        /** Get the number of diagnostic in the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native int gpr_unit_diagnostic_count(
            AnalysisUnit unit
        );

        /** Get the nth diagnostic of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        public static native Diagnostic gpr_unit_diagnostic(
            AnalysisUnit unit,
            int n
        );

        // ----- Node functions -----

        /** Return whether the two given entities are equal */
        @CompilerDirectives.TruffleBoundary
        public static native int gpr_node_is_equivalent(
            Entity entity_left,
            Entity entity_right
        );

        /** Get the hash of a node */
        @CompilerDirectives.TruffleBoundary
        public static native int gpr_node_hash(
            Entity entity
        );

        /** Get the node kind */
        @CompilerDirectives.TruffleBoundary
        public static native int gpr_node_kind(
            Entity entity
        );

        /** Get the node text */
        @CompilerDirectives.TruffleBoundary
        public static native Text gpr_node_text(
            Entity entity
        );

        /** Get the node source location range */
        @CompilerDirectives.TruffleBoundary
        public static native SourceLocationRange gpr_node_sloc_range(
            Entity entity
        );

        /** Get the node children count */
        @CompilerDirectives.TruffleBoundary
        public static native int gpr_node_children_count(
            Entity entity
        );

        /** Get the node nth child */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_node_child(
            Entity entity,
            int n
        );

        /** Get if the node is a token node */
        @CompilerDirectives.TruffleBoundary
        public static native boolean gpr_node_is_token_node(
            Entity entity
        );

        /** Get the unit of the node */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit gpr_node_unit(
            Entity entity
        );

        /** Get the entity image of the node */
        @CompilerDirectives.TruffleBoundary
        public static native Text gpr_node_image(
            Entity entity
        );

        // ----- Node fields accessors and properties -----

        
    

            

        /** Isomethod of gpr_gpr_node_parent langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_gpr_node_parent(
            Entity node
        );
            

        /** Isomethod of gpr_gpr_node_parents langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native GprNodeArray gpr_gpr_node_parents(
            boolean with_self,
            Entity node
        );
            

        /** Isomethod of gpr_gpr_node_children langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native GprNodeArray gpr_gpr_node_children(
            Entity node
        );
            

        /** Isomethod of gpr_gpr_node_token_start langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Token gpr_gpr_node_token_start(
            Entity node
        );
            

        /** Isomethod of gpr_gpr_node_token_end langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Token gpr_gpr_node_token_end(
            Entity node
        );
            

        /** Isomethod of gpr_gpr_node_child_index langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native int gpr_gpr_node_child_index(
            Entity node
        );
            

        /** Isomethod of gpr_gpr_node_previous_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_gpr_node_previous_sibling(
            Entity node
        );
            

        /** Isomethod of gpr_gpr_node_next_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_gpr_node_next_sibling(
            Entity node
        );
            

        /** Isomethod of gpr_gpr_node_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native AnalysisUnit gpr_gpr_node_unit(
            Entity node
        );
            

        /** Isomethod of gpr_gpr_node_is_ghost langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean gpr_gpr_node_is_ghost(
            Entity node
        );
            

        /** Isomethod of gpr_gpr_node_full_sloc_image langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native String gpr_gpr_node_full_sloc_image(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_all_qualifier_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean gpr_all_qualifier_p_as_bool(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of gpr_attribute_decl_f_attr_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_attribute_decl_f_attr_name(
            Entity node
        );
            

        /** Isomethod of gpr_attribute_decl_f_attr_index langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_attribute_decl_f_attr_index(
            Entity node
        );
            

        /** Isomethod of gpr_attribute_decl_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_attribute_decl_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_attribute_reference_f_attribute_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_attribute_reference_f_attribute_name(
            Entity node
        );
            

        /** Isomethod of gpr_attribute_reference_f_attribute_index langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_attribute_reference_f_attribute_index(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of gpr_builtin_function_call_f_function_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_builtin_function_call_f_function_name(
            Entity node
        );
            

        /** Isomethod of gpr_builtin_function_call_f_parameters langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_builtin_function_call_f_parameters(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_case_construction_f_var_ref langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_case_construction_f_var_ref(
            Entity node
        );
            

        /** Isomethod of gpr_case_construction_f_items langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_case_construction_f_items(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_case_item_f_choice langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_case_item_f_choice(
            Entity node
        );
            

        /** Isomethod of gpr_case_item_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_case_item_f_decls(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_compilation_unit_f_project langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_compilation_unit_f_project(
            Entity node
        );

        
    


        
    


        
    

            

        /** Isomethod of gpr_prefix_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_prefix_f_prefix(
            Entity node
        );
            

        /** Isomethod of gpr_prefix_f_suffix langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_prefix_f_suffix(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of gpr_limited_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native boolean gpr_limited_node_p_as_bool(
            Entity node
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of gpr_package_decl_f_pkg_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_package_decl_f_pkg_name(
            Entity node
        );
            

        /** Isomethod of gpr_package_decl_f_pkg_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_package_decl_f_pkg_spec(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_package_extension_f_extended_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_package_extension_f_extended_name(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_package_renaming_f_renamed_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_package_renaming_f_renamed_name(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_package_spec_f_extension langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_package_spec_f_extension(
            Entity node
        );
            

        /** Isomethod of gpr_package_spec_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_package_spec_f_decls(
            Entity node
        );
            

        /** Isomethod of gpr_package_spec_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_package_spec_f_end_name(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_project_f_context_clauses langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_project_f_context_clauses(
            Entity node
        );
            

        /** Isomethod of gpr_project_f_project_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_project_f_project_decl(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_project_declaration_f_qualifier langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_project_declaration_f_qualifier(
            Entity node
        );
            

        /** Isomethod of gpr_project_declaration_f_project_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_project_declaration_f_project_name(
            Entity node
        );
            

        /** Isomethod of gpr_project_declaration_f_extension langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_project_declaration_f_extension(
            Entity node
        );
            

        /** Isomethod of gpr_project_declaration_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_project_declaration_f_decls(
            Entity node
        );
            

        /** Isomethod of gpr_project_declaration_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_project_declaration_f_end_name(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_project_extension_f_is_all langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_project_extension_f_is_all(
            Entity node
        );
            

        /** Isomethod of gpr_project_extension_f_path_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_project_extension_f_path_name(
            Entity node
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of gpr_string_literal_at_f_str_lit langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_string_literal_at_f_str_lit(
            Entity node
        );
            

        /** Isomethod of gpr_string_literal_at_f_at_lit langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_string_literal_at_f_at_lit(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_terms_f_terms langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_terms_f_terms(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_type_reference_f_var_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_type_reference_f_var_type_name(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_typed_string_decl_f_type_id langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_typed_string_decl_f_type_id(
            Entity node
        );
            

        /** Isomethod of gpr_typed_string_decl_f_string_literals langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_typed_string_decl_f_string_literals(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_variable_decl_f_var_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_variable_decl_f_var_name(
            Entity node
        );
            

        /** Isomethod of gpr_variable_decl_f_var_type langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_variable_decl_f_var_type(
            Entity node
        );
            

        /** Isomethod of gpr_variable_decl_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_variable_decl_f_expr(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_variable_reference_f_variable_name langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_variable_reference_f_variable_name(
            Entity node
        );
            

        /** Isomethod of gpr_variable_reference_f_attribute_ref langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_variable_reference_f_attribute_ref(
            Entity node
        );

        
    

            

        /** Isomethod of gpr_with_decl_f_is_limited langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_with_decl_f_is_limited(
            Entity node
        );
            

        /** Isomethod of gpr_with_decl_f_path_names langkit function */
        @CompilerDirectives.TruffleBoundary
        public static native Entity gpr_with_decl_f_path_names(
            Entity node
        );


    }


    // ==========
    // Defining the Native-Image bindings library
    // ==========

    
    
    
    
    
    
    

    /** This class contains the directives for the shared lib loading */
    public static final class LibDirectives implements CContext.Directives {
        @Override
        public List<String> getHeaderFiles() {
            List<String> res = new ArrayList<>();
            res.add("<gpr_parser.h>");
            res.add("<stdlib.h>");
            return res;
        }

        @Override
        public List<String> getLibraries() {
            List<String> res = new ArrayList<>();
            res.add("gprlang");
            return res;
        }
    }

    // ===== Language specific structures =====

    


    // ===== Constant structures =====

    /** The structure for the langkit exceptions */
    @CContext(LibDirectives.class)
    @CStruct("gpr_exception")
    public interface LangkitExceptionNative extends PointerBase {
        @CField("kind") public int get_kind();
        @CField("kind") public void set_kind(
            int kind
        );

        @CField("information") public CCharPointer get_information();
        @CField("information") public void set_information(
            CCharPointer information
        );
    }

    /** The big integers are just pointers */
    public interface BigIntegerNative extends Pointer {}

    /** The structure for the symbols */
    @CContext(LibDirectives.class)
    @CStruct("gpr_symbol_type")
    public interface SymbolNative extends PointerBase {
        @CField("data") public VoidPointer get_data();
        @CField("data") public void set_data(
            VoidPointer data
        );

        @CField("bounds") public VoidPointer get_bounds();
        @CField("bounds") public void set_bounds(
            VoidPointer bounds
        );
    }

    /** The string wrappers are just pointers */
    public interface StringNative extends Pointer {}

    /** The structure for the text */
    @CContext(LibDirectives.class)
    @CStruct("gpr_text")
    public interface TextNative extends PointerBase {
        @CField("chars") public CIntPointer get_chars();
        @CField("chars") public void set_chars(
            CIntPointer chars
        );

        @CField("length") public long get_length();
        @CField("length") public void set_length(
            long length
        );

        @CField("is_allocated") public int get_is_allocated();
        @CField("is_allocated") public void set_is_allocated(
            int is_allocated
        );
    }

    /** The structure for the source locations */
    @CContext(LibDirectives.class)
    @CStruct("gpr_source_location")
    public interface SourceLocationNative extends PointerBase {
        @CField("line") public int get_line();
        @CField("line") public void set_line(
            int line
        );

        @CField("column") public short get_column();
        @CField("column") public void set_column(
            short column
        );
    }

    /** The structure for the source location ranges */
    @CContext(LibDirectives.class)
    @CStruct("gpr_source_location_range")
    public interface SourceLocationRangeNative extends PointerBase {
        @CField("start.line") public int get_start_line();
        @CField("start.line") public void set_start_line(
            int start_line
        );

        @CField("start.column") public short get_start_column();
        @CField("start.column") public void set_start_column(
            short start_column
        );

        @CField("end.line") public int get_end_line();
        @CField("end.line") public void set_end_line(
            int end_line
        );

        @CField("end.column") public short get_end_column();
        @CField("end.column") public void set_end_column(
            short end_column
        );
    }

    /** The structure for the diagnostic */
    @CContext(LibDirectives.class)
    @CStruct("gpr_diagnostic")
    public interface DiagnosticNative extends PointerBase {
        @CField("sloc_range.start.line") public int get_start_line();
        @CField("sloc_range.start.line") public void set_start_line(
            int start_line
        );

        @CField("sloc_range.start.column") public short get_start_column();
        @CField("sloc_range.start.column") public void set_start_column(
            short start_column
        );

        @CField("sloc_range.end.line") public int get_end_line();
        @CField("sloc_range.end.line") public void set_end_line(
            int end_line
        );

        @CField("sloc_range.end.column") public short get_end_column();
        @CField("sloc_range.end.column") public void set_end_column(
            short end_column
        );

        @CField("message.chars") public CIntPointer get_message_chars();
        @CField("message.chars") public void set_message_chars(
            CIntPointer chars
        );

        @CField("message.length") public long get_message_length();
        @CField("message.length") public void set_message_length(
            long length
        );

        @CField("message.is_allocated") public int get_message_is_allocated();
        @CField("message.is_allocated") public void set_message_is_allocated(
            int is_allocated
        );
    }


    /** The file reader is just a pointer */
    public interface FileReaderNative extends Pointer {}

    /** The unit provider is just a pointer */
    public interface UnitProviderNative extends Pointer {}

    /** The event handler is just a pointer */
    public interface EventHandlerNative extends Pointer {}

    /** The event handler unit requested callback type */
    public interface UnitRequestedFunctionPointer extends CFunctionPointer {
        @InvokeCFunctionPointer
        void invoke(
            VoidPointer data,
            AnalysisContextNative context,
            TextNative name,
            AnalysisUnitNative from,
            boolean found,
            boolean is_not_found_error
        );
    }

    /** The event handler unit parsed callback type */
    public interface UnitParsedFunctionPointer extends CFunctionPointer {
        @InvokeCFunctionPointer
        void invoke(
            VoidPointer data,
            AnalysisContextNative context,
            AnalysisUnitNative unit,
            boolean reparsed
        );
    }

    /** Anonymous structure for the token data handler */
    @RawStructure
    public interface TokenDataHandlerNative extends PointerBase {
        @RawField public long version();
    }

    /** The structure representing a token */
    @CContext(LibDirectives.class)
    @CStruct("gpr_token")
    public interface TokenNative extends PointerBase {
        @CField("context") public AnalysisContextNative get_context();
        @CField("context") public void set_context(
            AnalysisContextNative context
        );

        @CField("token_data") public TokenDataHandlerNative get_data();
        @CField("token_data") public void set_data(
            TokenDataHandlerNative data
        );

        @CField("token_index") public int get_token_index();
        @CField("token_index") public void set_token_index(
            int token_index
        );

        @CField("trivia_index") public int get_trivia_index();
        @CField("trivia_index") public void set_trivia_index(
            int trivia_index
        );
    }

    /** Anonymous strucutre for analysis context */
    @RawStructure
    public interface AnalysisContextNative extends PointerBase {
        @RawField public long serial_number();
    }

    /** Anonymous strucutre for analysis unit */
    @RawStructure
    public interface AnalysisUnitNative extends PointerBase {
        @RawField public long version_number();
    }

    // ===== Generated structures =====

        
        
    
    

    @CContext(LibDirectives.class)
    @CStruct("gpr_internal_metadata")
    public interface MetadataNative extends PointerBase {
        @CField("dummy") public byte get_dummy();
        @CField("dummy") public void set_dummy(byte dummy);
    }

        
    
    

    /** The structure for the langkit gpr_internal_entity_info */
    @CContext(LibDirectives.class)
    @CStruct("gpr_internal_entity_info")
    public interface EntityInfoNative extends PointerBase {
        @CField("md.dummy")
        public byte
        get_md_dummy();

        @CField("md.dummy")
        public void
        set_md_dummy(
            byte val
        );

        @CFieldAddress("md.dummy")
        public <T extends PointerBase> T address_md_dummy();
        @CField("rebindings")
        public Pointer
        get_rebindings();

        @CField("rebindings")
        public void
        set_rebindings(
            Pointer val
        );

        @CFieldAddress("rebindings")
        public <T extends PointerBase> T address_rebindings();
        @CField("from_rebound")
        public byte
        get_from_rebound();

        @CField("from_rebound")
        public void
        set_from_rebound(
            byte val
        );

        @CFieldAddress("from_rebound")
        public <T extends PointerBase> T address_from_rebound();
    }

    
    

    /** The structure for the langkit gpr_base_entity */
    @CContext(LibDirectives.class)
    @CStruct("gpr_base_entity")
    public interface EntityNative extends PointerBase {
        @CField("node")
        public Pointer
        get_node();

        @CField("node")
        public void
        set_node(
            Pointer val
        );

        @CFieldAddress("node")
        public <T extends PointerBase> T address_node();
        @CField("info.md.dummy")
        public byte
        get_info_md_dummy();

        @CField("info.md.dummy")
        public void
        set_info_md_dummy(
            byte val
        );

        @CFieldAddress("info.md.dummy")
        public <T extends PointerBase> T address_info_md_dummy();
        @CField("info.rebindings")
        public Pointer
        get_info_rebindings();

        @CField("info.rebindings")
        public void
        set_info_rebindings(
            Pointer val
        );

        @CFieldAddress("info.rebindings")
        public <T extends PointerBase> T address_info_rebindings();
        @CField("info.from_rebound")
        public byte
        get_info_from_rebound();

        @CField("info.from_rebound")
        public void
        set_info_from_rebound(
            byte val
        );

        @CFieldAddress("info.from_rebound")
        public <T extends PointerBase> T address_info_from_rebound();
    }

        
        

    // ===== Generated arrays =====

    
    

    /**
     * The native structure of the gpr_gpr_node_array langkit array.
     */
    @CContext(LibDirectives.class)
    @CStruct(
        value = "gpr_gpr_node_array_record",
        addStructKeyword = true,
        isIncomplete = true
    )
    public interface GprNodeArrayNative extends PointerBase {
        @CField("n") public int get_n();
        @CField("ref_count") public int get_ref_count();
        @CFieldAddress("items")
        public <T extends PointerBase> T address_items();
    }


    // ===== Generated iterators =====


    // ===== Native function definitions =====

    /** This class contains all native function definitions for NI */
    @CContext(LibDirectives.class)
    public static final class NI_LIB {

        // ----- Language specific functions -----

        


        // ----- Entry point literals -----

        /**
         * This entry point literal provide a pointer to the unit requested
         * callback.
         */
        public static final CEntryPointLiteral<UnitRequestedFunctionPointer>
            unitRequestedFunction = CEntryPointLiteral.create(
                GprParser.class,
                "unitRequested",
                IsolateThread.class,
                AnalysisContextNative.class,
                TextNative.class,
                AnalysisUnitNative.class,
                byte.class,
                byte.class
            );

        /**
         * This entry point literal provide a pointer to the unit parsed
         * callback.
         */
        public static final CEntryPointLiteral<UnitParsedFunctionPointer>
            unitParsedFunction = CEntryPointLiteral.create(
                GprParser.class,
                "unitParsed",
                IsolateThread.class,
                AnalysisContextNative.class,
                AnalysisUnitNative.class,
                byte.class
            );

        // ----- Util functions -----

        /** Util function to free langkit side allocated memory */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_free(
            PointerBase pointer
        );

        // ----- Exception functions -----

        /** Get the last exception raised by langkit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native LangkitExceptionNative
        gpr_get_last_exception();

        // ----- Big integer functions -----

        /** Create a big integer from a text */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native BigIntegerNative gpr_create_big_integer(
            TextNative text
        );

        /** Get the text representation of a big integer */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_big_integer_text(
            BigIntegerNative big_integer,
            TextNative text
        );

        /** Decrease the reference counter of the big integer */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_big_integer_decref(
            BigIntegerNative big_integer
        );

        // ----- Symbol functions -----

        /** Create a new symbol in the given context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_context_symbol(
            AnalysisContextNative context,
            TextNative text,
            SymbolNative res
        );

        /** Get the text of a given symbol */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_symbol_text(
            SymbolNative symbol,
            TextNative text
        );

        // ----- String functions -----

        /** Create a new string wrapper in langkit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native StringNative gpr_create_string(
            CIntPointer content,
            int length
        );

        /** Decrease the reference counter of a string */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_string_dec_ref(
            StringNative string
        );

        // ----- Text functions -----

        /** Destroy a text in the memory */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_destroy_text(
            TextNative text
        );

        // ----- File reader functions -----

        /** Decrease the reference counter of the given file reader */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_dec_ref_file_reader(
            FileReaderNative fileReader
        );

        // ----- Unit provider functions -----

        /** Decrease the ref counter of the unit provider */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_dec_ref_unit_provider(
            UnitProviderNative unitProvider
        );

        // ----- Event handler functions -----

        /** Create a new event handler */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native EventHandlerNative gpr_create_event_handler(
            VoidPointer data,
            VoidPointer destroy_callback,
            UnitRequestedFunctionPointer unit_requested_func,
            UnitParsedFunctionPointer unit_parsed_func
        );

        /** Decrease the ref counter of the event handler */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_dec_ref_event_handler(
            EventHandlerNative eventHandler
        );

        // ----- Token functions -----

        /**
         * Kind for this token.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_token_get_kind(
            TokenNative token
        );

        /**
         * Return the source location range of the given token.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_token_sloc_range(
            TokenNative token,
            SourceLocationRangeNative result
        );

        /** Get the next token */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_token_next(
            TokenNative token,
            TokenNative res
        );

        /** Get the previous token */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_token_previous(
            TokenNative token,
            TokenNative res
        );

        /** Get if two tokens are equivalent */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native byte gpr_token_is_equivalent(
            TokenNative left,
            TokenNative right
        );

        /** Get the text in a token range */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_token_range_text(
            TokenNative start,
            TokenNative end,
            TextNative res
        );

        // ----- Analysis context functions -----

        /** Allocate a new analysis context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisContextNative
        gpr_allocate_analysis_context();

        /** Create a new analysis context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void
        gpr_initialize_analysis_context(
            AnalysisContextNative context,
            CCharPointer charset,
            FileReaderNative file_reader,
            UnitProviderNative unit_provider,
            EventHandlerNative event_handler,
            int with_trivia,
            int tab_stop
        );

        /** Increase the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_context_incref(
            AnalysisContextNative context
        );

        /** Decrease the reference counter of a context */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_context_decref(
            AnalysisContextNative context
        );

        // ----- Analysis unit functions -----

        /** Get a unit from a file */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative
        gpr_get_analysis_unit_from_file(
            AnalysisContextNative context,
            CCharPointer file_name,
            CCharPointer charset,
            int reparse,
            int rule
        );

        /** Get a unit from a buffer */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative
        gpr_get_analysis_unit_from_buffer(
            AnalysisContextNative context,
            CCharPointer file_name,
            CCharPointer charset,
            CCharPointer buffer,
            long buffer_size,
            int rule
        );


        /** Get the root of an analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_unit_root(
            AnalysisUnitNative unit,
            EntityNative res
        );

        /** Get the file name for a given unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native CCharPointer gpr_unit_filename(
            AnalysisUnitNative unit
        );

        /** Get the token count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_unit_token_count(
            AnalysisUnitNative unit
        );

        /** Get the trivia count of the analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_unit_trivia_count(
            AnalysisUnitNative unit
        );

        /** Get the first token of an analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_unit_first_token(
            AnalysisUnitNative unit,
            TokenNative res
        );

        /** Get the last token of an analysis unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_unit_last_token (
            AnalysisUnitNative unit,
            TokenNative res
        );

        /** Get the context for a given unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisContextNative gpr_unit_context(
            AnalysisUnitNative unit
        );

        /** Get the diagnostic count of the unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_unit_diagnostic_count(
            AnalysisUnitNative unit
        );

        /** Get the nth diagnostic for the unit */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_unit_diagnostic(
            AnalysisUnitNative unit,
            int n,
            DiagnosticNative diagnostic
        );

        // ----- Array functions -----

        
    

        /**
         * Create a new sized array.
         *
         * @param size The size of the array to create.
         * @return The native pointer to the created array.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native GprNodeArrayNative gpr_gpr_node_array_create(int size);

        /**
         * Decrease reference counter of the given array
         *
         * @param array The array to decrease the reference counter.
         */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_gpr_node_array_dec_ref(GprNodeArrayNative array);


        // ----- Structure functions -----

            
            
        
    


            
        
    


        
    


            
            

        // ----- Node functions -----

        /** Return whether the two given entities are equal */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_node_is_equivalent(
            EntityNative entity_left,
            EntityNative entity_right
        );

        /** Get the hash of a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_node_hash(
            EntityNative entity
        );

        /** Get the type of a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_node_kind(
            EntityNative entity
        );

        /** Get the text from a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_node_text(
            EntityNative entity,
            TextNative text
        );

        /** Get the source location range for a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_node_sloc_range(
            EntityNative entity,
            SourceLocationRangeNative slocr
        );

        /** Get the number of children for a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_node_children_count(
            EntityNative entity
        );

        /** Get the nth child for the node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_node_child(
            EntityNative entity,
            int n,
            EntityNative res
        );

        /** Get if the node is a token node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_node_is_token_node(
            EntityNative entity
        );

        /** Get the unit of the node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native AnalysisUnitNative gpr_node_unit(
            EntityNative entity
        );

        /** Get the image of a node */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native void gpr_node_image(
            EntityNative entity,
            TextNative text
        );

        // ----- Node fields accessors and properties -----

        
    

            

        /** Isomethod of gpr_gpr_node_parent langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_gpr_node_parent(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_gpr_node_parents langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_gpr_node_parents(
            EntityNative node,
            byte with_self,
            Pointer result
        );
            

        /** Isomethod of gpr_gpr_node_children langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_gpr_node_children(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of gpr_gpr_node_token_start langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_gpr_node_token_start(
            EntityNative node,
            TokenNative result
        );
            

        /** Isomethod of gpr_gpr_node_token_end langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_gpr_node_token_end(
            EntityNative node,
            TokenNative result
        );
            

        /** Isomethod of gpr_gpr_node_child_index langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_gpr_node_child_index(
            EntityNative node,
            CIntPointer result
        );
            

        /** Isomethod of gpr_gpr_node_previous_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_gpr_node_previous_sibling(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_gpr_node_next_sibling langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_gpr_node_next_sibling(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_gpr_node_unit langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_gpr_node_unit(
            EntityNative node,
            Pointer result
        );
            

        /** Isomethod of gpr_gpr_node_is_ghost langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_gpr_node_is_ghost(
            EntityNative node,
            CCharPointer result
        );
            

        /** Isomethod of gpr_gpr_node_full_sloc_image langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_gpr_node_full_sloc_image(
            EntityNative node,
            Pointer result
        );

        
    

            

        /** Isomethod of gpr_all_qualifier_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_all_qualifier_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    

            

        /** Isomethod of gpr_attribute_decl_f_attr_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_attribute_decl_f_attr_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_attribute_decl_f_attr_index langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_attribute_decl_f_attr_index(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_attribute_decl_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_attribute_decl_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_attribute_reference_f_attribute_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_attribute_reference_f_attribute_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_attribute_reference_f_attribute_index langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_attribute_reference_f_attribute_index(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of gpr_builtin_function_call_f_function_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_builtin_function_call_f_function_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_builtin_function_call_f_parameters langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_builtin_function_call_f_parameters(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_case_construction_f_var_ref langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_case_construction_f_var_ref(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_case_construction_f_items langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_case_construction_f_items(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_case_item_f_choice langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_case_item_f_choice(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_case_item_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_case_item_f_decls(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_compilation_unit_f_project langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_compilation_unit_f_project(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    

            

        /** Isomethod of gpr_prefix_f_prefix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_prefix_f_prefix(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_prefix_f_suffix langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_prefix_f_suffix(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of gpr_limited_node_p_as_bool langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_limited_node_p_as_bool(
            EntityNative node,
            CCharPointer result
        );

        
    


        
    


        
    


        
    

            

        /** Isomethod of gpr_package_decl_f_pkg_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_package_decl_f_pkg_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_package_decl_f_pkg_spec langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_package_decl_f_pkg_spec(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_package_extension_f_extended_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_package_extension_f_extended_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_package_renaming_f_renamed_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_package_renaming_f_renamed_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_package_spec_f_extension langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_package_spec_f_extension(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_package_spec_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_package_spec_f_decls(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_package_spec_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_package_spec_f_end_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_project_f_context_clauses langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_project_f_context_clauses(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_project_f_project_decl langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_project_f_project_decl(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_project_declaration_f_qualifier langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_project_declaration_f_qualifier(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_project_declaration_f_project_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_project_declaration_f_project_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_project_declaration_f_extension langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_project_declaration_f_extension(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_project_declaration_f_decls langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_project_declaration_f_decls(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_project_declaration_f_end_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_project_declaration_f_end_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_project_extension_f_is_all langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_project_extension_f_is_all(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_project_extension_f_path_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_project_extension_f_path_name(
            EntityNative node,
            EntityNative result
        );

        
    


        
    


        
    


        
    


        
    


        
    


        
    


        
    

            

        /** Isomethod of gpr_string_literal_at_f_str_lit langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_string_literal_at_f_str_lit(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_string_literal_at_f_at_lit langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_string_literal_at_f_at_lit(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_terms_f_terms langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_terms_f_terms(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_type_reference_f_var_type_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_type_reference_f_var_type_name(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_typed_string_decl_f_type_id langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_typed_string_decl_f_type_id(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_typed_string_decl_f_string_literals langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_typed_string_decl_f_string_literals(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_variable_decl_f_var_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_variable_decl_f_var_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_variable_decl_f_var_type langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_variable_decl_f_var_type(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_variable_decl_f_expr langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_variable_decl_f_expr(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_variable_reference_f_variable_name langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_variable_reference_f_variable_name(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_variable_reference_f_attribute_ref langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_variable_reference_f_attribute_ref(
            EntityNative node,
            EntityNative result
        );

        
    

            

        /** Isomethod of gpr_with_decl_f_is_limited langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_with_decl_f_is_limited(
            EntityNative node,
            EntityNative result
        );
            

        /** Isomethod of gpr_with_decl_f_path_names langkit function */
        @CompilerDirectives.TruffleBoundary
        @CFunction
        public static native int gpr_with_decl_f_path_names(
            EntityNative node,
            EntityNative result
        );


    }


    // ==========
    // Exceptions
    // ==========

    /**
     * This class represents exception during symbol manipulation.
     */
    public static final class SymbolException extends RuntimeException {
        public SymbolException(
            final String symbol
        ) {
            super("Invalid symbol : '" + symbol + "'");
        }
    }

    /**
     * This class reprsents exception during enum manipulation.
     */
    public static final class EnumException extends RuntimeException {
        public EnumException(
            final String msg
        ) {
            super(msg);
        }
    }

    /**
     * This class represents an exception in the references manipulation.
     */
    public static final class ReferenceException extends RuntimeException {
        public ReferenceException(
            final String msg
        ) {
            super(msg);
        }
    }

    /**
     * This class wraps the exceptions from the langkit native library.
     */
    public static class LangkitException extends RuntimeException {

        // ----- Instance attributes -----

        /** The kind of the langkit exception. */
        public final ExceptionKind kind;

        // ----- Constructors -----

        /**
         * Create a new langkit exception.
         *
         * @param kind The kind of the exception represented by an integer
         * which will be mapped to an enum value.
         * @param message The message of the exception.
         */
        public LangkitException(
            final int kind,
            final String message
        ) {
            super(message);
            this.kind = ExceptionKind.fromC(kind);
        }

    }

    // ==========
    // Enum definitions
    // ==========

    // ===== Constants enumeration =====

    /**
     * Kind for this token.
     */
    public enum TokenKind {

        // ----- Enum values -----

        NO_TOKEN(-1, "No_Token"),
        GPR_TERMINATION(0, "Termination"),
        GPR_LEXING_FAILURE(1, "Lexing_Failure"),
        GPR_IDENTIFIER(2, "Identifier"),
        GPR_ALL(3, "All"),
        GPR_ABSTRACT(4, "Abstract"),
        GPR_AT(5, "At"),
        GPR_CASE(6, "Case"),
        GPR_END(7, "End"),
        GPR_FOR(8, "For"),
        GPR_IS(9, "Is"),
        GPR_LIMITED(10, "Limited"),
        GPR_PRIVATE(11, "Private"),
        GPR_NULL(12, "Null"),
        GPR_OTHERS(13, "Others"),
        GPR_PACKAGE(14, "Package"),
        GPR_RENAMES(15, "Renames"),
        GPR_TYPE(16, "Type"),
        GPR_USE(17, "Use"),
        GPR_PRAGMA(18, "Pragma"),
        GPR_WHEN(19, "When"),
        GPR_WITH(20, "With"),
        GPR_EXTENDS(21, "Extends"),
        GPR_PAR_OPEN(22, "Par_Open"),
        GPR_PAR_CLOSE(23, "Par_Close"),
        GPR_SEMICOLON(24, "Semicolon"),
        GPR_COLON(25, "Colon"),
        GPR_COMMA(26, "Comma"),
        GPR_DOT(27, "Dot"),
        GPR_AMP(28, "Amp"),
        GPR_TICK(29, "Tick"),
        GPR_PIPE(30, "Pipe"),
        GPR_ASSIGN(31, "Assign"),
        GPR_ARROW(32, "Arrow"),
        GPR_STRING(33, "String"),
        GPR_NUMBER(34, "Number"),
        GPR_LABEL(35, "Label"),
        GPR_CHAR(36, "Char"),
        GPR_COMMENT(37, "Comment"),
        GPR_WHITESPACE(38, "Whitespace"),
        ;

        // ----- Class attributes -----

        /** Singleton that represents the none token kind. */
        public static final TokenKind NONE = NO_TOKEN;

        /** The map from int to enum values. */
        private static final Map<Integer, TokenKind> map = new HashMap<>();

        // ----- Instance attributes -----

        /** The value of the enum instance. */
        private final int value;

        /** The name of the enum instance in the Langkit DSL. */
        public final String name;

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(TokenKind elem : TokenKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private TokenKind(
            final int value,
            final String name
        ) {
            this.value = value;
            this.name = name;
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static TokenKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get TokenKind from " + cValue
                );
            return (TokenKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    /**
     * Enumerated type describing all possible exceptions that need to be
     * handled in the C bindings.
     */
    public enum ExceptionKind {

        // ----- Enum values -----

        FILE_READ_ERROR(0),
        BAD_TYPE_ERROR(1),
        OUT_OF_BOUNDS_ERROR(2),
        INVALID_INPUT(3),
        INVALID_SYMBOL_ERROR(4),
        INVALID_UNIT_NAME_ERROR(5),
        NATIVE_EXCEPTION(6),
        PRECONDITION_FAILURE(7),
        PROPERTY_ERROR(8),
        TEMPLATE_ARGS_ERROR(9),
        TEMPLATE_FORMAT_ERROR(10),
        TEMPLATE_INSTANTIATION_ERROR(11),
        STALE_REFERENCE_ERROR(12),
        SYNTAX_ERROR(13),
        UNKNOWN_CHARSET(14),
        MALFORMED_TREE_ERROR(15),
        ;

        // ----- Class attributes

        /** Singleton that represents the none expcetion kind. */
        public static final ExceptionKind NONE =
            FILE_READ_ERROR;

        /** The map from int to enum values. */
        private static final Map<Integer, ExceptionKind> map =
            new HashMap<>();

        // ----- Instance ttributes -----

        /** The value of the enum instance. */
        private final int value;

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(ExceptionKind elem : ExceptionKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private ExceptionKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static ExceptionKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get ExceptionKind from " + cValue
                );
            return (ExceptionKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    // ===== Generated enums =====

    
    

    /**
     * Specify a kind of analysis unit. Specification units provide an
     * interface to the outer world while body units provide an implementation
     * for the corresponding interface.
     */
    public enum AnalysisUnitKind {

        // ----- Enum values -----

        UNIT_SPECIFICATION(0),
        UNIT_BODY(1),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final AnalysisUnitKind NONE =
            UNIT_SPECIFICATION;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, AnalysisUnitKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(AnalysisUnitKind elem : AnalysisUnitKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private AnalysisUnitKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static AnalysisUnitKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get AnalysisUnitKind from " + cValue
                );
            return (AnalysisUnitKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**

     */
    public enum LookupKind {

        // ----- Enum values -----

        RECURSIVE(0),
        FLAT(1),
        MINIMAL(2),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final LookupKind NONE =
            RECURSIVE;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, LookupKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(LookupKind elem : LookupKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private LookupKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static LookupKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get LookupKind from " + cValue
                );
            return (LookupKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**
     * Discriminant for DesignatedEnv structures.
     */
    public enum DesignatedEnvKind {

        // ----- Enum values -----

        NONE_ENUM(0),
        CURRENT_ENV(1),
        NAMED_ENV(2),
        DIRECT_ENV(3),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final DesignatedEnvKind NONE =
            NONE_ENUM;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, DesignatedEnvKind> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(DesignatedEnvKind elem : DesignatedEnvKind.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private DesignatedEnvKind(
            final int value
        ) {
            this.value = value;
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static DesignatedEnvKind fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get DesignatedEnvKind from " + cValue
                );
            return (DesignatedEnvKind) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }

    
    

    /**
     * Gramar rule to use for parsing.
     */
    public enum GrammarRule {

        // ----- Enum values -----

        PROJECT_QUALIFIER_RULE(0),
        PROJECT_EXTENSION_RULE(1),
        PROJECT_DECLARATION_RULE(2),
        PROJECT_RULE(3),
        DECLARATIVE_ITEMS_RULE(4),
        DECLARATIVE_ITEM_RULE(5),
        SIMPLE_DECLARATIVE_ITEMS_RULE(6),
        SIMPLE_DECLARATIVE_ITEM_RULE(7),
        VARIABLE_DECL_RULE(8),
        ATTRIBUTE_DECL_RULE(9),
        ASSOCIATIVE_ARRAY_INDEX_RULE(10),
        PACKAGE_DECL_RULE(11),
        PACKAGE_RENAMING_RULE(12),
        PACKAGE_EXTENSION_RULE(13),
        PACKAGE_SPEC_RULE(14),
        EMPTY_DECLARATION_RULE(15),
        CASE_CONSTRUCTION_RULE(16),
        CASE_ITEM_RULE(17),
        OTHERS_DESIGNATOR_RULE(18),
        CHOICE_RULE(19),
        DISCRETE_CHOICE_LIST_RULE(20),
        WITH_DECL_RULE(21),
        CONTEXT_CLAUSES_RULE(22),
        TYPED_STRING_DECL_RULE(23),
        IDENTIFIER_RULE(24),
        STRING_LITERAL_RULE(25),
        NUM_LITERAL_RULE(26),
        STATIC_NAME_RULE(27),
        ATTRIBUTE_REFERENCE_RULE(28),
        VARIABLE_REFERENCE_RULE(29),
        TYPE_REFERENCE_RULE(30),
        BUILTIN_FUNCTION_CALL_RULE(31),
        EXPRESSION_RULE(32),
        EXPRESSION_LIST_RULE(33),
        STRING_LITERAL_AT_RULE(34),
        TERM_RULE(35),
        COMPILATION_UNIT_RULE(36),
        ;

        // ----- Attributes -----

        /** Singleton that represents the none enum value. */
        public static final GrammarRule NONE =
            PROJECT_QUALIFIER_RULE;

        /** The value of the enum instance. */
        private final int value;

        /** The map from int to enum values. */
        private static final Map<Integer, GrammarRule> map = new HashMap<>();

        // ----- Constructors -----

        static {
            // Initialise the lookup map
            for(GrammarRule elem : GrammarRule.values()) {
                map.put(elem.value, elem);
            }
        }

        /** Private constructor. */
        private GrammarRule(
            final int value
        ) {
            this.value = value;
        }

        // ----- Enum methods -----

        /**
         * Get the enum instance for the given C value.
         *
         * @param cValue The C int value to get the enum instance from.
         * @return The enum instance which correspond to the int value.
         * @throws EnumException When the int value doesn't map to any enum
         * instance.
         */
        public static GrammarRule fromC(
            final int cValue
        ) throws EnumException {
            if(!map.containsKey(cValue))
                throw new EnumException(
                    "Cannot get GrammarRule from " + cValue
                );
            return (GrammarRule) map.get(cValue);
        }

        /**
         * Get the C value from the enum instance.
         *
         * @return The int C value of the enum instance.
         */
        public int toC() {
            return this.value;
        }

    }


    // ==========
    // Java wrapping classes
    // ==========

    // ===== Constant structure wrapping classes =====

    /**
     * This class wraps the langkit characters which are 32 bit wide.
     */
    public static final class Char {

        // ----- Class attributes -----

        /** Singleton that represents the none char. */
        public static final Char NONE = new Char(0);

        // ----- Instance attributes -----

        /** The value of the character. */
        public final int value;

        // ----- Constructors -----

        /**
         * Create a new character from its value. In langkit characters are
         * 32 bit wide so represented by Java integer.
         *
         * @param value The value of the character.
         */
        Char(
            final int value
        ) {
            this.value = value;
        }

        /**
         * Create a character from its integer value.
         *
         * @param value The character value.
         * @return The newly created character.
         */
        public static Char create(
            final int value
        ) {
            return new Char(value);
        }

        /**
         * Create a character from a Java character.
         *
         * @param value The source value of the character.
         * @return The newly created character.
         */
        public static Char create(
            final char value
        ) {
            return new Char((int) value);
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given NI pointer in a Java class.
         *
         * @param pointer The NI pointer to wrap.
         * @return The wrapped character.
         */
        static Char wrap(
            final CIntPointer pointer
        ) {
            return wrap(pointer.read());
        }

        /**
         * Wrap an integer to a character.
         *
         * @param value The value of the character in an integer.
         * @return The newly created character.
         */
        static Char wrap(
            final int value
        ) {
            return new Char(value);
        }

        /**
         * Unwrap the character in the given int pointer.
         *
         * @param pointer The pointer to unwrap the character in.
         */
        void unwrap(
            final CIntPointer pointer
        ) {
            pointer.write(this.value);
        }

        /**
         * Unwrap the character in a Java integer.
         *
         * @return The character value in a Java integer.
         */
        int unwrap() {
            return this.value;
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            final ByteBuffer buffer = ByteBuffer.allocate(4);
            buffer.order(BYTE_ORDER);
            buffer.putInt(this.value);
            return decodeUTF32(buffer.array());
        }

    }

    /**
     * Arbitrarily large integer.
     */
    static final class BigIntegerWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none big integer. */
        public static final BigInteger NONE = BigInteger.ZERO;

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer which points to a native big integer.
         *
         * @param pointer The pointer to the native big integer.
         * @return The Java big integer.
         */
        static BigInteger wrap(
            final Pointer pointer
        ) {
            return wrap((BigIntegerNative) pointer.readWord(0));
        }

        /**
         * Wrap the given native big integer in a Java big integer.
         *
         * @param bigIntegerNative The big integer native value.
         * @return The Java big integer.
         */
        static BigInteger wrap(
            final BigIntegerNative bigIntegerNative
        ) {
            final String representation = getRepresentation(bigIntegerNative);
            return new BigInteger(representation);
        }

        /**
         * Unwrap the given big integer in the given pointer as a native
         * big integer.
         *
         * @param bigInteger The big integer to unwrap.
         * @param pointer The pointer to place the big integer in.
         */
        static void unwrap(
            final BigInteger bigInteger,
            final Pointer pointer
        ) {
            pointer.writeWord(0, unwrap(bigInteger));
        }

        /**
         * Unwrap the given big integer.
         *
         * @param bigInteger The big integer to unwrap.
         * @return The native big integer newly allocated.
         */
        static BigIntegerNative unwrap(
            final BigInteger bigInteger
        ) {
            // Create the representation of the big integer
            final String representation = bigInteger.toString();
            try(final Text bigIntegerText = Text.create(representation)) {
                TextNative bigIntegerTextNative = StackValue.get(
                    TextNative.class
                );
                bigIntegerText.unwrap(bigIntegerTextNative);

                // Create the big intger by calling the native function
                return NI_LIB.gpr_create_big_integer(
                    bigIntegerTextNative
                );
            }
        }

        /**
         * Release the big integer pointed by the given pointer.
         *
         * @param pointer The pointer to the big integer to release.
         */
        static void release(
            final Pointer pointer
        ) {
            release((BigIntegerNative) pointer.readWord(0));
        }

        /**
         * Release the given native big integer.
         *
         * @param bigIntegerNative The native big integer to release.
         */
        static void release(
            final BigIntegerNative bigIntegerNative
        ) {
            NI_LIB.gpr_big_integer_decref(bigIntegerNative);
        }

        // ----- Class methods -----

        /**
         * Get the string representation of the given native big integer.
         *
         * @param bigIntegerNative The native big integer to get the
         * representation from.
         */
        private static String getRepresentation(
            final BigIntegerNative bigIntegerNative
        ) {
            // Allocate the stack value for the text
            final TextNative bigIntegerTextNative = StackValue.get(
                TextNative.class
            );
            Text.NONE.unwrap(bigIntegerTextNative);

            // Call the native function
            NI_LIB.gpr_big_integer_text(
                bigIntegerNative,
                bigIntegerTextNative
            );

            // Wrap the text and return the result
            try(final Text bigIntegerText = Text.wrap(bigIntegerTextNative)) {
                return bigIntegerText.getContent();
            }
        }

    }

    /**
     * Reference to a symbol. Symbols are owned by analysis contexts, so they
     * must not outlive them. This type exists only in the C API, and roughly
     * wraps the corresponding Ada type (an array fat pointer).
     */
    public static final class Symbol {

        // ----- Class attributes -----

        /** Singleton that represents the none symbol. */
        public static final Symbol NONE = new Symbol("");

        // ----- Instance attributes

        /** The text of the symbol. */
        public final String text;

        // ----- Constructors -----

        /**
         * Create a new symbol from its text.
         *
         * @param text The symbol text.
         */
        Symbol(
            final String text
        ) {
            this.text = text;
        }

        /**
         * Public access to the symbol creation.
         *
         * @param text The text of the symbol.
         */
        public static Symbol create(
            final String text
        ) {
            return new Symbol(text);
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given pointer to a native symbol.
         *
         * @param pointer The pointer to the native symbol.
         * @return The wrapped symbol.
         */
        static Symbol wrap(
            final Pointer pointer
        ) {
            return wrap((SymbolNative) pointer.readWord(0));
        }

        /**
         * Wrap the given symbol native value in a Java value.
         *
         * @param symbolNative The symbol native value.
         * @return The wrapped symbol.
         */
        static Symbol wrap(
            final SymbolNative symbolNative
        ) {
            // Get the symbol text
            final TextNative textNative = StackValue.get(TextNative.class);
            Text.NONE.unwrap(textNative);
            NI_LIB.gpr_symbol_text(
                symbolNative,
                textNative
            );

            // Return the new symbol
            try(final Text text = Text.wrap(textNative)) {
                return new Symbol(text.getContent());
            }
        }

        /**
         * Unwrap the symbol in the given native structure.
         *
         * @param symbolNative The native structure to unwrap in.
         */
        void unwrap(
            final SymbolNative symbolNative,
            final AnalysisContext context
        ) {
            // Unwrap the symbol text
            try(Text text = Text.create(this.text)) {
                final TextNative textNative = StackValue.get(
                    TextNative.class
                );
                text.unwrap(textNative);

                // Call the symbol creation
                final int resCode = NI_LIB.gpr_context_symbol(
                    context.reference.ni(),
                    textNative,
                    symbolNative
                );

                // Check the symbol creation success
                if(resCode == 0) {
                    throw new SymbolException(this.text);
                }
            }
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.text;
        }

    }

    /**
     * Type to contain Unicode text data.
     */
    static final class StringWrapper {

        // ----- Class attributes -----

        /** Singleton that represents the none string. */
        public static final String NONE = "";

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer that points to a native string.
         *
         * @param pointer The pointer to the native string.
         * @return The Java string wrapped from the native string.
         */
        static String wrap(
            final Pointer pointer
        ) {
            return wrap((StringNative) pointer.readWord(0));
        }

        /**
         * Wrap a native string wrapper in a Java string.
         *
         * @param stringNative The native string to wrap.
         * @return The Java string created from the native one.
         */
        static String wrap(
            final StringNative stringNative
        ) {
            return getNativeStringContent(stringNative);
        }

        /**
         * Unwrap the string in the given pointer.
         *
         * @param string The string to unwrap.
         * @param pointer The pointer to place the native string in.
         */
        static void unwrap(
            final String string,
            final Pointer pointer
        ) {
            pointer.writeWord(0, unwrap(string));
        }

        /**
         * Unwrap the given string in a native one.
         *
         * @param string The string to unwrap.
         * @return The native string unwrapped.
         */
        static StringNative unwrap(
            final String string
        ) {
            return nativeStringFromContent(string);
        }

        /**
         * Release the native string at the given pointer.
         *
         * @param pointer The pointer to the native string.
         */
        static void release(
            final Pointer pointer
        ) {
            release((StringNative) pointer.readWord(0));
        }

        /**
         * Release the given native string.
         *
         * @param stringNative The native string to release.
         */
        static void release(
            final StringNative stringNative
        ) {
            NI_LIB.gpr_string_dec_ref(stringNative);
        }

        // ----- Class methods -----

        /**
         * Get the content of the given native string in a Java one.
         *
         * @param stringNative The native string to get the content from.
         * @return The Java string.
         */
        private static String getNativeStringContent(
            final StringNative stringNative
        ) {
            // Prepare the working variable
            final Pointer pointer = (Pointer) stringNative;
            final int length = pointer.readInt(0);
            final byte[] contentArray = new byte[length * 4];

            // Get the content from the native string
            for(int i = 0 ; i < contentArray.length ; i++) {
                contentArray[i] = pointer.readByte(i + 8);
            }

            // Return the decoded string
            return decodeUTF32(contentArray);
        }

        /**
         * Create a native string from a Java string.
         *
         * @param string The Java string to create the native string from.
         * @return The native string.
         */
        private static StringNative nativeStringFromContent(
            final String string
        ) {
            // Encode the string in UTF-32
            final byte[] contentArray = encodeUTF32(string);
            final int length = string.length();

            // Create the native array
            final Pointer contentArrayNative = UnmanagedMemory.malloc(
                contentArray.length
            );
            for(int i = 0 ; i < contentArray.length ; i++) {
                contentArrayNative.writeByte(i, contentArray[i]);
            }

            // Create the native string
            final StringNative res = NI_LIB.gpr_create_string(
                (CIntPointer) contentArrayNative,
                length
            );

            // Free the memory
            UnmanagedMemory.free(contentArrayNative);

            // Return the result
            return res;
        }

    }

    /**
     * String encoded in UTF-32 (native endianness).
     */
    public static final class Text implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none text. */
        public static final Text NONE = new Text(
            PointerWrapper.nullPointer(),
            0,
            false
        );

        // ----- Instance attributes -----

        /** The pointer to the characters. */
        private final PointerWrapper charPointer;

        /** The size of the text. */
        private final long length;

        /** If the text is allocated. */
        private final boolean isAllocated;

        /** If the text object is the owner of its buffer. */
        private final boolean isOwner;

        /** The content of the text in a Java string. */
        private String content;

        // ----- Constructors -----

        /**
         * Create a new text from its content.
         *
         * @param charPointer The pointer to the characters of the text.
         * @param length The length of the text in character.
         * @param isAllocated If the text is allocated in the memory.
         */
        Text(
            final PointerWrapper charPointer,
            final long length,
            final boolean isAllocated
        ) {
            this(
                charPointer,
                length,
                isAllocated,
                false,
                null
            );
        }

        /**
         * Create a new text from its content and buffer.
         *
         * @param charPointer The pointer to the characters of the text.
         * @param length The length of the text in character.
         * @param isAllocated If the text is allocated in the memory.
         * @param contentArray The characters of the text (for JNI)
         */
        Text(
            final PointerWrapper charPointer,
            final long length,
            final boolean isAllocated,
            final byte[] contentArray
        ) {
            this(
                charPointer,
                length,
                isAllocated,
                false,
                contentArray
            );
        }

        /**
         * Create a new text from its content and buffer.
         *
         * @param charPointer The pointer to the characters of the text.
         * @param length The length of the text in character.
         * @param isAllocated If the text is allocated in the memory.
         * @param isOwner If the Java object owns the text buffer.
         * @param contentArray The characters of the text
         * (as strings, this is only used in JNI mode).
         */
        private Text(
            final PointerWrapper charPointer,
            final long length,
            final boolean isAllocated,
            final boolean isOwner,
            final byte[] contentArray
        ) {
            this.charPointer = charPointer;
            this.length = length;
            this.isAllocated = isAllocated;
            this.isOwner = isOwner;

            if(contentArray != null) {
                this.content = decodeUTF32(contentArray);
            } else {
                this.content = null;
            }
        }

        /**
         * Create a new langkit text from its string content.
         *
         * @param content The content of the text in a Java string.
         * @return The newly created text.
         */
        public static Text create(
            final String content
        ) {
            final byte[] contentArray = encodeUTF32(content);

            if(ImageInfo.inImageCode()) {
                final PointerWrapper charPointer = new PointerWrapper(
                    toCBytes(contentArray)
                );
                return new Text(
                    charPointer,
                    (long) content.length(),
                    false,
                    true,
                    contentArray
                );
            } else {
                return JNI_LIB.gpr_create_text(contentArray);
            }
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the native text pointed by the given pointer.
         *
         * @param pointer The pointer to the native text to wrap.
         * @return The wrapped text.
         */
        static Text wrap(
            final Pointer pointer
        ) {
            return wrap((TextNative) pointer.readWord(0));
        }

        /**
         * Wrap a text native value in the Java class.
         *
         * @param textNative The text native NI value.
         * @return The newly wrapped text.
         */
        static Text wrap(
            final TextNative textNative
        ) {
            return new Text(
                new PointerWrapper(textNative.get_chars()),
                textNative.get_length(),
                textNative.get_is_allocated() != 0
            );
        }

        /**
         * Unwrap the text in the given NI pointer.
         *
         * @param textNative The NI pointer to the native text structure.
         */
        void unwrap(
            final TextNative textNative
        ) {
            textNative.set_chars(this.charPointer.ni());
            textNative.set_length(this.length);
            textNative.set_is_allocated(this.isAllocated ? 1 : 0);
        }

        // ----- Instance methods -----

        /**
         * Get the content of the text in a Java string.
         *
         * @return the content of the text.
         */
        public String getContent() {
            // Content is null only when using the Graal C API.
            if(this.content == null) {
                final byte[] contentArray = new byte[(int) this.length * 4];
                for(int i = 0 ; i < contentArray.length ; i++) {
                    contentArray[i] = (
                        (CCharPointer) this.charPointer.ni()
                    ).read(i);
                }
                this.content = decodeUTF32(contentArray);
            }

            return this.content;
        }

        /** @see java.lang.AutoCloseable#close() */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                if(this.isOwner) {
                    UnmanagedMemory.free(this.charPointer.ni());
                } else {
                    final TextNative textNative = StackValue.get(
                        TextNative.class
                    );
                    this.unwrap(textNative);
                    NI_LIB.gpr_destroy_text(textNative);
                }
            } else {
                JNI_LIB.gpr_destroy_text(this);
            }
            checkException();

        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.getContent();
        }

    }

    /**
     * Location in a source file. Line and column numbers are one-based.
     */
    public static final class SourceLocation {

        // ----- Class attributes -----

        /** Singleton that represents the none source location. */
        public static final SourceLocation NONE = new SourceLocation(
            0,
            (short) 0
        );

        // ----- Instance attributes -----

        /** The line of the source location. */
        public final int line;

        /** The column of the source location. */
        public final short column;

        // ----- Constructors -----

        /**
         * Create a new source location from a line and a column.
         *
         * @param line The line of the source location.
         * @param column The column of the source location.
         */
        SourceLocation(
            final int line,
            final short column
        ) {
            this.line = line;
            this.column = column;
        }

        /**
         * Create a source location from its line and column.
         *
         * @param line The line.
         * @param column The column.
         * @return The newly create source location.
         */
        public static SourceLocation create(
            final int line,
            final short column
        ) {
            return new SourceLocation(
                line,
                column
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given source location in the Java class.
         *
         * @param pointer Pointer to the native source location.
         * @return The newly wrapper source location.
         */
        static SourceLocation wrap(
            final Pointer pointer
        ) {
            return wrap((SourceLocationNative) pointer.readWord(0));
        }

        /**
         * Wrap a source location native value in the Java class.
         *
         * @param sourceLocationNative The source location NI native value.
         * @return The newly wrapped source location.
         */
        static SourceLocation wrap(
            final SourceLocationNative sourceLocationNative
        ) {
            return new SourceLocation(
                sourceLocationNative.get_line(),
                sourceLocationNative.get_column()
            );
        }

        /**
         * Uwrap the source location in the given NI pointer.
         *
         * @param sourceLocationNative The NI pointer to the native structure
         *  to fill.
         */
        public void unwrap(
            final SourceLocationNative sourceLocationNative
        ) {
            sourceLocationNative.set_line(this.line);
            sourceLocationNative.set_column(this.column);
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.line + ":" + this.column;
        }

    }

    /**
     * Location of a span of text in a source file.
     */
    public static final class SourceLocationRange {

        // ----- Class attributes -----

        /** Singleton that represents the none source location range. */
        public static final SourceLocationRange NONE =
            new SourceLocationRange(
                SourceLocation.NONE,
                SourceLocation.NONE
            );

        // ----- Instance attributes -----

        /** The start of the range. */
        public final SourceLocation start;

        /** The end of the range. */
        public final SourceLocation end;

        // ----- Constructors -----

        /**
         * Create a source location range from its bounds.
         *
         * @param start The start of the range.
         * @param end The end of the range.
         */
        SourceLocationRange(
            final SourceLocation start,
            final SourceLocation end
        ) {
            this.start = start;
            this.end = end;
        }

        /**
         * Create a new source location range from its bounds.
         *
         * @param start The starting bound.
         * @param end The ending bound.
         * @return The newly created source location range.
         */
        public static SourceLocationRange create(
            final SourceLocation start,
            final SourceLocation end
        ) {
            return new SourceLocationRange(
                start,
                end
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given source location range in the Java class.*
         *
         * @param pointer The pointer to the native source location range.
         * @return The newly wrapped source location range.
         */
        static SourceLocationRange wrap(
            final Pointer pointer
        ) {
            return wrap((SourceLocationRangeNative) pointer.readWord(0));
        }

        /**
         * Wrap a source location range native value in the Java class.
         *
         * @param sourceLocationRangeNative The source location range NI
         * native value.
         * @return The newly wrapped source location range.
         */
        static SourceLocationRange wrap(
            final SourceLocationRangeNative sourceLocationRangeNative
        ) {
            return new SourceLocationRange(
                new SourceLocation(
                    sourceLocationRangeNative.get_start_line(),
                    sourceLocationRangeNative.get_start_column()
                ),
                new SourceLocation(
                    sourceLocationRangeNative.get_end_line(),
                    sourceLocationRangeNative.get_end_column()
                )
            );
        }

        /**
         * Uwrap the source location range in the given NI pointer.
         *
         * @param sourceLocationRangeNative The NI pointer to the native
         * structure to fill.
         */
        void unwrap(
            final SourceLocationRangeNative sourceLocationRangeNative
        ) {
            sourceLocationRangeNative.set_start_line(this.start.line);
            sourceLocationRangeNative.set_start_column(this.start.column);
            sourceLocationRangeNative.set_end_line(this.end.line);
            sourceLocationRangeNative.set_end_column(this.end.column);
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.start.toString() + "-" + this.end.toString();
        }

    }

    /**
     * Diagnostic for an analysis unit: cannot open the source file, parsing
     * error, ...
     */
    public static final class Diagnostic {

        // ----- Class attributes -----

        /** Singleton that represents the none diagnositc. */
        public static final Diagnostic NONE = new Diagnostic(
            SourceLocationRange.NONE,
            Text.NONE
        );

        // ----- Instance attributes -----

        /** The source location range of the diagnostic. */
        public final SourceLocationRange sourceLocationRange;

        /** The message of the diagnostic. */
        public final Text message;

        // ----- Constructors -----

        /**
         * Create a diagnostic from its content.
         *
         * @param sourceLocationRange The range of the diagnostic.
         * @param message The message of the diagnostic.
         */
        Diagnostic(
            final SourceLocationRange sourceLocationRange,
            final Text message
        ) {
            this.sourceLocationRange = sourceLocationRange;
            this.message = message;
        }

        /**
         * Create a new diagnostic from its content.
         *
         * @param sourceLocationRange The source location range concerned by
         * this diagnostic.
         * @param message The message of the diagnostic.
         * @return The newly created diagnostic
         */
        public static Diagnostic create(
            final SourceLocationRange sourceLocationRange,
            final Text message
        ) {
            return new Diagnostic(
                sourceLocationRange,
                message
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to a native diagnostic.
         *
         * @param pointer The pointer to the native diagnositc.
         * @return The wrapped diagnositc.
         */
        static Diagnostic wrap(
            final Pointer pointer
        ) {
            return wrap((DiagnosticNative) pointer.readWord(0));
        }

        /**
         * Wrap a diagnostic native value in the Java class.
         *
         * @param diagnosticNative The diagnostic NI native value.
         * @return The newly wrapped diagnositc.
         */
        static Diagnostic wrap(
            final DiagnosticNative diagnosticNative
        ) {
            return new Diagnostic(
                new SourceLocationRange(
                    new SourceLocation(
                        diagnosticNative.get_start_line(),
                        diagnosticNative.get_start_column()
                    ),
                    new SourceLocation(
                        diagnosticNative.get_end_line(),
                        diagnosticNative.get_end_column()
                    )
                ),
                new Text(
                    new PointerWrapper(diagnosticNative.get_message_chars()),
                    diagnosticNative.get_message_length(),
                    diagnosticNative.get_message_is_allocated() != 0
                )
            );
        }

        /**
         * Unwrap the diagnostic in the given NI pointer.
         *
         * @param diagnosticNative The pointer to the native structure to
         * fill.
         */
        public void unwrap(
            final DiagnosticNative diagnosticNative
        ) {
            diagnosticNative.set_start_line(
                this.sourceLocationRange.start.line
            );
            diagnosticNative.set_start_column(
                this.sourceLocationRange.start.column
            );
            diagnosticNative.set_end_line(
                this.sourceLocationRange.end.line
            );
            diagnosticNative.set_end_column(
                this.sourceLocationRange.end.column
            );
            diagnosticNative.set_message_chars(
                this.message.charPointer.ni()
            );
            diagnosticNative.set_message_length(
                this.message.length
            );
            diagnosticNative.set_message_is_allocated(
                this.message.isAllocated ? 1 : 0
            );
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return this.message.toString()
                + " at <"
                + this.sourceLocationRange.toString()
                + ">";
        }

    }

    /**
     * Interface to override how source files are fetched and decoded.
     */
    public static final class FileReader implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none file reader. */
        public static final FileReader NONE = new FileReader(
            PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        /** The reference to the file reader */
        private final PointerWrapper reference;

        // ----- Constructors -----

        /**
         * Create a new file reader from its native reference.
         *
         * @param reference The reference to the native file reader.
         */
        FileReader(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to a native file reader.
         *
         * @param pointer The pointer to the native file reader.
         * @return The newly wrapped file reader.
         */
        static FileReader wrap(
            final Pointer pointer
        ) {
            return wrap((FileReaderNative) pointer.readWord(0));
        }

        /**
         * Wrap the given file reader in the Java class.
         *
         * @param fileReaderNative The native file reader to wrap.
         * @return The wrapped file reader.
         */
        static FileReader wrap(
            final FileReaderNative fileReaderNative
        ) {
            return new FileReader(
                new PointerWrapper(fileReaderNative)
            );
        }

        /**
         * Unwrap the file reader in the given pointer.
         *
         * @param pointer The pointer to unwrap the file reader in.
         */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /**
         * Get the unwrapped file reader.
         *
         * @return The unwrapped native file reader.
         */
        FileReaderNative unwrap() {
            return (FileReaderNative) this.reference.ni();
        }

        // ----- Instance methods -----

        /** @see java.lang.AutoCloseable#close() */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.gpr_dec_ref_file_reader(
                    this.reference.ni()
                );
            } else {
                JNI_LIB.gpr_dec_ref_file_reader(this);
            }
            checkException();

        }

    }

    /*
 * Interface to fetch analysis units from a name and a unit kind.
 *
 * The unit provider mechanism provides an abstraction which assumes that to
 * any couple (unit name, unit kind) we can associate at most one source file.
 * This means that several couples can be associated to the same source file,
 * but on the other hand, only one one source file can be associated to a
 * couple.
 *
 * This is used to make the semantic analysis able to switch from one analysis
 * units to another.
 *
 * See the documentation of each unit provider for the exact semantics of the
 * unit name/kind information.
 */
    public static final class UnitProvider implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none unit provider. */
        public static final UnitProvider NONE = new UnitProvider(
            PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        /** The reference to the unit provider */
        private final PointerWrapper reference;

        // ----- Constructors -----

        /**
         * Create a new unit provider with the reference to the native one.
         *
         * @param reference The reference to the native unit provider.
         */
        UnitProvider(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given pointer to a native unit provider.
         *
         * @param pointer The pointer to the native unit provider.
         * @return The wrapped unit provider.
         */
        static UnitProvider wrap(
            final Pointer pointer
        ) {
            return wrap((UnitProviderNative) pointer.readWord(0));
        }

        /**
         * Wrap the given native unit provider.
         *
         * @param unitProviderNative The native unit provider.
         * @return The wrapped unit provider.
         */
        static UnitProvider wrap(
            final UnitProviderNative unitProviderNative
        ) {
            return new UnitProvider(
                new PointerWrapper(unitProviderNative)
            );
        }

        /**
         * Unwrap the unit provider in the given native pointer.
         *
         * @param pointer The pointer to place the native unit provider in.
         */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /**
         * Get the native unit provider.
         *
         * @return The native unit provider.
         */
        UnitProviderNative unwrap() {
            return (UnitProviderNative) this.reference.ni();
        }

        // ----- Instance methods -----

        /** @see java.lang.AutoCloseable#close() */
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.gpr_dec_ref_unit_provider(this.reference.ni());
            } else {
                JNI_LIB.gpr_dec_ref_unit_provider(this);
            }
            checkException();

        }

    }

    /**
 * Interface to handle events sent by the analysis context.
 */
    public static final class EventHandler implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none event handler. */
        public static final EventHandler NONE = new EventHandler(
            PointerWrapper.nullPointer(),
            null,
            null
        );

        /** This map contains all created event handlers. */
        private static final Map<PointerWrapper, EventHandler>
            eventHandlerCache = new ConcurrentHashMap<>();

        // ----- Instance attributes -----

        /** The reference to the native event handler. */
        private final PointerWrapper reference;

        /**
         * The Java callback when an analysis unit is requested in the
         * associated context.
         */
        private final UnitRequestedCallback unitRequestedCallback;

        /**
         * The Java callback when an analysis unit is parsed in the
         * associated context.
         */
        private final UnitParsedCallback unitParsedCallback;

        // ----- Constructors -----

        /**
         * Create a new event handler from its native reference.
         *
         * @param reference The reference to the native event handler.
         * @param unitRequestedCallback The callback for unit requests.
         * @param unitParsedCallback The callback for unit parsing.
         */
        EventHandler(
            final PointerWrapper reference,
            final UnitRequestedCallback unitRequestedCallback,
            final UnitParsedCallback unitParsedCallback
        ) {
            this.reference = reference;
            this.unitRequestedCallback = unitRequestedCallback;
            this.unitParsedCallback = unitParsedCallback;
        }

        /**
         * Create a new even handler with its callbacks. Callbacks can be null.
         *
         * @param unitRequestedCallback The callback for analysis unit requests.
         * @param unitParsedCallback The callback for analysis unit parsing.
         */
        public static EventHandler create(
            final UnitRequestedCallback unitRequestedCallback,
            final UnitParsedCallback unitParsedCallback
        ) {
            // Prepare the reference to the native event handler
            final PointerWrapper reference;

            if(ImageInfo.inImageCode()) {
                // Get the current thread
                final IsolateThread thread = CurrentIsolate.getCurrentThread();

                // Create the native event handler
                final EventHandlerNative resNative =
                    NI_LIB.gpr_create_event_handler(
                        (VoidPointer) thread,
                        WordFactory.nullPointer(),
                        NI_LIB.unitRequestedFunction.getFunctionPointer(),
                        NI_LIB.unitParsedFunction.getFunctionPointer()
                    );

                // Set the result to the created event handler
                reference = new PointerWrapper(resNative);
            } else {
                reference = JNI_LIB.gpr_create_event_handler(
                    unitRequestedCallback,
                    unitParsedCallback
                );
            }

            // Return the new event handler wrapped object
            return new EventHandler(
                reference,
                unitRequestedCallback,
                unitParsedCallback
            );
        }

        /**
         * Get event handler Java object from its native pointer.
         *
         * @param reference The pointer to the native event handler.
         * @return The associated Java event handler.
         */
        static EventHandler fromReference(
            final PointerWrapper reference
        ) {
            if(eventHandlerCache.containsKey(reference)) {
                return eventHandlerCache.get(reference);
            } else {
                throw new ReferenceException(
                    "Cannot get event handler from this reference: " +
                    reference.toString()
                );
            }
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the given pointer to an event handler.
         *
         * @param pointer The pointer to the native event handler.
         * @return The wrapped event handler.
         */
        EventHandler wrap(
            final Pointer pointer
        ) {
            return wrap((EventHandlerNative) pointer.readWord(0));
        }

        /**
         * Wrap the given native event handler.
         *
         * @param eventHandlerNative The native value of the event handler.
         * @return The wrapped event handler.
         */
        EventHandler wrap(
            final EventHandlerNative eventHandlerNative
        ) {
            return fromReference(new PointerWrapper(eventHandlerNative));
        }

        /**
         * Unwrap the event handler in the given native pointer.
         *
         * @param pointer The pointer to place the native event handler.
         */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /**
         * Unwrap the event handler to a native value.
         *
         * @return The native value of the event handler.
         */
        EventHandlerNative unwrap() {
            return (EventHandlerNative) this.reference.ni();
        }

        // ----- Getters -----

        public UnitRequestedCallback getUnitRequestCallback() {
            return this.unitRequestedCallback;
        }

        public UnitParsedCallback getUnitParsedCallback() {
            return this.unitParsedCallback;
        }

        // ----- Instance methods -----

        /** @see java.lang.AutoCloseable#close() */
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.gpr_dec_ref_event_handler(this.reference.ni());
            } else {
                JNI_LIB.gpr_dec_ref_event_handler(this);
            }
            checkException();

        }

        // ----- Inner classes -----

        /**
         * Callback that will be called when a unit is requested from the
         * context ``Context``.
         *
         * ``Name`` is the name of the requested unit.
         *
         * ``From`` is the unit from which the unit was requested.
         *
         * ``Found`` indicates whether the requested unit was found or not.
         *
         * ``Is_Not_Found_Error`` indicates whether the fact that the unit was
         * not found is an error or not.
         *
         * .. warning:: The interface of this callback is probably subject to
         *    change, so should be treated as experimental.
         */
        @FunctionalInterface
        public interface UnitRequestedCallback {
            void invoke(
                AnalysisContext context,
                String name,
                AnalysisUnit from,
                boolean found,
                boolean isNotFoundError
            );
        }

        /**
         * Callback that will be called when any unit is parsed from the
         * context ``Context``.
         *
         * ``Unit`` is the resulting unit.
         *
         * ``Reparsed`` indicates whether the unit was reparsed, or whether it
         * was the first parse.
         */
        @FunctionalInterface
        public interface UnitParsedCallback {
            void invoke(
                AnalysisContext context,
                AnalysisUnit unit,
                boolean reparsed
            );
        }

    }

    /**
     * Reference to a token in an analysis unit.
     */
    public static class Token {

        // ----- Instance attributes -----

        /**
         * We only store the reference to the context to avoid ref-count
         * problems. To access the token context go throught the
         * analysis unit.
         */
        protected final PointerWrapper contextRef;

        /** The unit of the token. */
        public final AnalysisUnit unit;

        /** The pointer to the token data handler. */
        protected final PointerWrapper tokenDataHandler;

        /** The index of the token. */
        public final int tokenIndex;

        /** The trivia index. */
        public final int triviaIndex;

        /** The kind of the token. */
        public final TokenKind kind;

        /** The text of the token. */
        protected final String text;

        /** The source location range of the token. */
        public final SourceLocationRange sourceLocationRange;

        // ----- Constructors -----

        /**
         * Create a new token from its content.
         *
         * @param contextRef The context of the token.
         * @param unit The unit which owns the token.
         * @param tokenDataHandler The pointer to the token data.
         * @param tokenIndex The index of the token.
         * @param triviaIndex The trivia index of the token.
         * @param kind The kind of the token in an integer.
         * @param text The text of the token.
         * @param sourceLocationRange The location of the token.
         */
        Token(
            final PointerWrapper contextRef,
            final AnalysisUnit unit,
            final PointerWrapper tokenDataHandler,
            final int tokenIndex,
            final int triviaIndex,
            final TokenKind kind,
            final String text,
            final SourceLocationRange sourceLocationRange
        ) {
            this.contextRef = contextRef;
            this.unit = unit;
            this.tokenDataHandler = tokenDataHandler;
            this.tokenIndex = tokenIndex;
            this.triviaIndex = triviaIndex;
            this.kind = kind;
            this.text = text;
            this.sourceLocationRange = sourceLocationRange;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap the pointer to a native token.
         *
         * @param pointer The pointer to the native token.
         * @param unit The analysis unit which owns the token.
         * @return The wrapped token or a none value if the token is a none
         * one.
         */
        static Token wrap(
            final Pointer pointer,
            final AnalysisUnit unit
        ) {
            return wrap(
                (TokenNative) pointer.readWord(0),
                unit
            );
        }

        /**
         * Wrap a native token value in the Java class.
         *
         * @param tokenNative The native NI token value.
         * @param unit The analysis unit that owns the token.
         * @return The wrapped token or a none value if the token data
         * handler is null.
         */
        static Token wrap(
            final TokenNative tokenNative,
            final AnalysisUnit unit
        ) {
            if(tokenNative.get_data().isNull())
                return NONE(unit);

            // Fetch the token kind, source location range and text from the
            // tokenNative reference.
            final TokenKind kind = TokenKind.fromC(
                NI_LIB.gpr_token_get_kind(tokenNative)
            );

            final SourceLocationRangeNative slocRangeNative =
                StackValue.get(SourceLocationRangeNative.class);
            NI_LIB.gpr_token_sloc_range(
                tokenNative,
                slocRangeNative
            );
            final SourceLocationRange slocRange =
                SourceLocationRange.wrap(slocRangeNative);

            final TextNative textNative = StackValue.get(TextNative.class);
            NI_LIB.gpr_token_range_text(
                tokenNative,
                tokenNative,
                textNative
            );
            final String text = Text.wrap(textNative).getContent();
            NI_LIB.gpr_destroy_text(textNative);

            // Wrap them in a high-level Token instance
            return new Token(
                new PointerWrapper(tokenNative.get_context()),
                unit,
                new PointerWrapper(tokenNative.get_data()),
                tokenNative.get_token_index(),
                tokenNative.get_trivia_index(),
                kind,
                text,
                slocRange
            );
        }

        /**
         * Unwrap the token in the given NI pointer.
         *
         * @param tokenNative The NI pointer to the native structure to
         * fill.
         */
        public void unwrap(
            final TokenNative tokenNative
        ) {
            tokenNative.set_context(this.contextRef.ni());
            tokenNative.set_data(this.tokenDataHandler.ni());
            tokenNative.set_token_index(this.tokenIndex);
            tokenNative.set_trivia_index(this.triviaIndex);
        }

        // ----- Getters -----

        public String getText() {
            return this.text;
        }

        public boolean isTrivia() {
            return this.triviaIndex != 0;
        }

        public boolean isNone() {
            return false;
        }

        // ----- Class methods -----

        /**
         * Get a none token for the given unit.
         *
         * @param unit The unit to get a none token for.
         * @return The none token for the given unit.
         */
        public static Token NONE(
            final AnalysisUnit unit
        ) {
            return NoToken.getInstance(unit);
        }

        /**
         * Get the text from the start token to the end token.
         *
         * @param start The start token.
         * @param end The end token.
         * @return The text between the two tokens.
         */
        @CompilerDirectives.TruffleBoundary
        public static String textRange(
            final Token start,
            final Token end
        ) {

            if(ImageInfo.inImageCode()) {
                final TokenNative startNative = StackValue.get(
                    TokenNative.class
                );
                start.unwrap(startNative);

                final TokenNative endNative = StackValue.get(
                    TokenNative.class
                );
                end.unwrap(endNative);

                final TextNative textNative = StackValue.get(
                    TextNative.class
                );
                Text.NONE.unwrap(textNative);
                NI_LIB.gpr_token_range_text(
                    startNative,
                    endNative,
                    textNative
                );
                try(final Text resText = Text.wrap(textNative)) {
                    return resText.getContent();
                }
            } else {
                try(
                    final Text resText = JNI_LIB.gpr_token_range_text(
                        start,
                        end
                    )
                ) {
                    return resText.getContent();
                }
            }

        }

        // ----- Instance methods -----

        /**
         * Get the next token.
         *
         * @return The next token in the source.
         */
        public Token next() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                this.unwrap(tokenNative);

                final TokenNative nextNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.gpr_token_next(
                    tokenNative,
                    nextNative
                );
                return wrap(nextNative, this.unit);
            } else {
                return JNI_LIB.gpr_token_next(this);
            }

        }

        /**
         * Get the previous token.
         *
         * @return The previous token in the source.
         */
        public Token previous() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                this.unwrap(tokenNative);

                final TokenNative previousNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.gpr_token_previous(
                    tokenNative,
                    previousNative
                );
                return wrap(previousNative, this.unit);
            } else {
                return JNI_LIB.gpr_token_previous(this);
            }

        }

        /**
         * Check of the token is equivalent to the other one.
         *
         * @param other The other token to compare with.
         */
        public boolean isEquivalent(
            final Token other
        ) {

            if(ImageInfo.inImageCode()) {
                final TokenNative leftNative = StackValue.get(
                    TokenNative.class
                );
                this.unwrap(leftNative);

                final TokenNative rightNative = StackValue.get(
                    TokenNative.class
                );
                other.unwrap(rightNative);

                return NI_LIB.gpr_token_is_equivalent(
                    leftNative,
                    rightNative
                ) != 0;
            } else {
                return JNI_LIB.gpr_token_is_equivalent(this, other);
            }

        }

        // ----- Override methods -----

        @Override
        @CompilerDirectives.TruffleBoundary
        public String toString() {
            return "<Token Kind="
                + this.kind.name
                + " Text=\""
                + stringRepresentation(this.getText())
                + "\">";
        }

        @Override
        public boolean equals(
            Object o
        ) {
            if(o == this) return true;
            if(!(o instanceof Token)) return false;
            final Token other = (Token) o;
            return other.tokenDataHandler.equals(this.tokenDataHandler) &&
                    other.tokenIndex == this.tokenIndex &&
                    other.triviaIndex == this.triviaIndex;
        }

        // ----- Inner classes -----

        /**
        * This class represents the absence of token.
        */
        private static final class NoToken extends Token {

            // ----- Class attributes -----

            /** The map of the no token instances */
            private static final Map<AnalysisUnit, NoToken> instances
                = new HashMap<>();

            // ----- Constructors -----

            /**
            * Create a new no token for the given analysis unit.
            * The constructor is private to avoid too many instances.
            *
            * @param unit The analysis unit to create the no token for.
            */
            private NoToken(
                final PointerWrapper contextRef,
                final AnalysisUnit unit
            ) {
                super(
                    contextRef,
                    unit,
                    PointerWrapper.nullPointer(),
                    0,
                    0,
                    TokenKind.fromC(-1),
                    null,
                    SourceLocationRange.NONE
                );
            }

            /**
            * Get the no token instance for the given analysis unit.
            *
            * @param unit The unit to get the no token instance for.
            * @return The no token instance.
            */
            static NoToken getInstance(
                final AnalysisUnit unit
            ) {
                if(!instances.containsKey(unit)) {
                    try(AnalysisContext context = unit.getContext()) {
                        instances.put(
                            unit,
                            new NoToken(context.reference, unit)
                        );
                    }
                }
                return instances.get(unit);
            }

            // ----- Getters -----

            @Override
            public String getText() {
                return "";
            }

            // ----- Instance methods -----

            @Override
            public Token next() {
                return this;
            }

            @Override
            public Token previous() {
                return this;
            }

            @Override
            public boolean isEquivalent(
                final Token other
            ) {
                return other instanceof NoToken;
            }

            @Override
            public boolean isNone() {
                return true;
            }

            @Override
            public void unwrap(
                final TokenNative tokenNative
            ) {
                tokenNative.set_context(this.contextRef.ni());
                tokenNative.set_data(this.tokenDataHandler.ni());
                tokenNative.set_token_index(this.tokenIndex);
                tokenNative.set_trivia_index(this.triviaIndex);
            }

            // ----- Override methods -----

            @Override
            @CompilerDirectives.TruffleBoundary
            public String toString() {
                return "<Token Kind="
                    + this.kind.name
                    + " Text=\"\">";
            }

            @Override
            public boolean equals(
                Object o
            ) {
                return o == this;
            }

        }

    }

    /**
     * This type represents a context for all source analysis. This is the
     * first type you need to create to use Gpr_Parser. It will contain the
     * results of all analysis, and is the main holder for all the data.
     *
     * You can create several analysis contexts if you need to, which enables
     * you, for example to:
     *
     * * analyze several different projects at the same time;
     *
     * * analyze different parts of the same projects in parallel.
     *
     * In the current design, contexts always keep all of their analysis units
     * allocated. If you need to get this memory released, the only option at
     * your disposal is to destroy your analysis context instance.
     */
    public static final class AnalysisContext implements AutoCloseable {

        // ----- Class attributes -----

        /** Singleton that represents the none analysis context. */
        public static final AnalysisContext NONE = new AnalysisContext(
            PointerWrapper.nullPointer(),
            null
        );

        /** This map contains all created analysis contexts. */
        private static final Map<PointerWrapper, AnalysisContext> contextCache
            = new ConcurrentHashMap<>();

        // ----- Instance attributes -----

        /** The reference to the native analysis context. */
        private final PointerWrapper reference;

        /** The event handler associated with the context. */
        private EventHandler eventHandler;

        // ----- Constructors -----

        /**
         * Create a new analysis unit from its reference.
         *
         * @param reference The native analysis context.
         * @param eventHandler The associated event handler.
         */
        private AnalysisContext(
            final PointerWrapper reference,
            final EventHandler eventHandler
        ) {
            this.reference = reference;
            this.eventHandler = eventHandler;

            increaseRefCounter(this);
        }

        /**
         * Create a new analysis context from scratch with its configuration.
         *
         * @param charset The charset for the analysis context, it can be null.
         * @param fileReader The file reader for the analysis context, it
         * can be null.
         * @param unitProvider The unit provider for the analysis context,
         * it can be null.
         * @param eventHandler The event handler for the analysis context,
         * it can be null.
         * @param withTrivia If the analysis context should include trivias.
         * @param tabStop The effect of the tabulations on the column number.
         */
        private AnalysisContext(
            final String charset,
            final FileReader fileReader,
            final UnitProvider unitProvider,
            final EventHandler eventHandler,
            final boolean withTrivia,
            final int tabStop
        ) {
            // Call the function to allocate the native analysis context
            final PointerWrapper reference;
            if(ImageInfo.inImageCode()) {
                reference = new PointerWrapper(
                    NI_LIB.gpr_allocate_analysis_context()
                );
            } else {
                reference = JNI_LIB.gpr_create_analysis_context(
                    charset,
                    (fileReader == null ?
                        FileReader.NONE :
                        fileReader),
                    (unitProvider == null ?
                        UnitProvider.NONE :
                        unitProvider),
                    (eventHandler == null ?
                        EventHandler.NONE :
                        eventHandler),
                    withTrivia,
                    tabStop
                );
            }

            // Place the context in the cache for potention callbacks during
            // context initialization.
            this.reference = reference;
            this.eventHandler = eventHandler;
            contextCache.put(this.reference, this);

            // Perform the context initialization
            if(ImageInfo.inImageCode()) {
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);

                NI_LIB.gpr_initialize_analysis_context(
                    (AnalysisContextNative) reference.ni(),
                    charsetNative,
                    (fileReader == null ?
                        WordFactory.nullPointer() :
                        fileReader.reference.ni()),
                    (unitProvider == null ?
                        WordFactory.nullPointer() :
                        unitProvider.reference.ni()),
                    (eventHandler == null ?
                        WordFactory.nullPointer() :
                        eventHandler.reference.ni()),
                    (withTrivia ? 1 : 0),
                    tabStop
                );
                if(charset != null) UnmanagedMemory.free(charsetNative);
            }
        }

        /**
         * Create a new analysis context with the default parameters.
         *
         * @return The newly create analysis unit.
         */
        public static AnalysisContext create() {
            return new AnalysisContext(
                null,
                null,
                null,
                null,
                true,
                8
            );
        }

        /**
         * Create an analysis context with its parameters.
         *
         * @param charset The charset for the analysis context, it can be null.
         * @param fileReader The file reader for the analysis context, it
         * can be null.
         * @param unitProvider The unit provider for the analysis context,
         * it can be null.
         * @param eventHandler The event handler for the analysis context,
         * it can be null.
         * @param withTrivia If the analysis context should include trivias.
         * @param tabStop The effect of the tabulations on the column number.
         * @return The newly create analysis unit.
         */
        public static AnalysisContext create(
            final String charset,
            final FileReader fileReader,
            final UnitProvider unitProvider,
            final EventHandler eventHandler,
            final boolean withTrivia,
            final int tabStop
        ) {
            return new AnalysisContext(
                charset,
                fileReader,
                unitProvider,
                eventHandler,
                withTrivia,
                tabStop
            );
        }

        /**
         * Get the analysis context object from its reference.
         *
         * @param reference The native reference to the analysis context.
         * @param eventHandler The corresponding event handler.
         * @param setEventHandler Whether to set the result's eventHandler
         * field to eventHandler when there is already a cached
         * AnalysisContext.
         * @return The Java analysis unit associated with the reference.
         */
        static AnalysisContext fromReference(
            final PointerWrapper reference,
            final EventHandler eventHandler,
            final boolean setEventHandler
        ) {
            if(contextCache.containsKey(reference)) {
                final AnalysisContext res = contextCache.get(reference);
                increaseRefCounter(res);
                if(setEventHandler)
                    res.eventHandler = eventHandler;
                return res;
            } else {
                final AnalysisContext result =
                    new AnalysisContext(reference, eventHandler);
                contextCache.put(reference, result);
                return result;
            }
        }

        /**
         * Get the analysis context object from its reference.
         *
         * @param reference The native reference to the analysis context.
         * @return The Java analysis unit associated with the reference.
         */
        static AnalysisContext fromReference(
            final PointerWrapper reference
        ) {
            return fromReference(reference, null, false);
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a native pointer to the native analysis context in the
         * Java class.
         *
         * @param pointer The pointer to the NI analysis context
         * native value.
         * @return The newly wrapped analysis context.
         */
        static AnalysisContext wrap(
            final Pointer pointer
        ) {
            return wrap((AnalysisContextNative) pointer.readWord(0));
        }

        /**
         * Wrap an analysis context native value in the Java class.
         *
         * @param analysisContextNative The NI analysis context native value.
         * @return The newly wrapped analysis context.
         */
        static AnalysisContext wrap(
            final AnalysisContextNative analysisContextNative
        ) {
            return fromReference(new PointerWrapper(analysisContextNative));
        }

        /**
         * Unwrap the analysis context in the given native pointer.
         *
         * @param pointer The pointer to place the native analysis unit.
         */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /**
         * Get the native value of the analysis context.
         *
         * @return The native analysis context.
         */
        AnalysisContextNative unwrap() {
            return (AnalysisContextNative) this.reference.ni();
        }

        // ----- Getters -----

        public EventHandler getEventHandler() {
            return this.eventHandler;
        }

        // ----- Instance methods -----

        /**
         * Get an analysis unit from the given file in the current context.
         *
         * @param fileName The file to get the analysis unit from.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromFile(
            final String fileName
        ) {
            return this.getUnitFromFile(
                fileName,
                null,
                false,
                DEFAULT_GRAMMAR_RULE
            );
        }

        /**
         * Get an analysis unit from the given file in the current context
         * with additional parameters.
         *
         * @param fileName The file to get the analysis unit from.
         * @param charset The charset of the given file.
         * @param reparse If the file should be reparsed.
         * @param rule The grammar rule to parse the source with.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromFile(
            final String fileName,
            final String charset,
            final boolean reparse,
            final GrammarRule rule
        ) {

            if(ImageInfo.inImageCode()) {
                final CCharPointer fileNameNative = toCString(fileName);
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);

                final AnalysisUnitNative resNative =
                    NI_LIB.gpr_get_analysis_unit_from_file(
                    this.reference.ni(),
                    fileNameNative,
                    charsetNative,
                    (reparse ? 1 : 0),
                    rule.toC()
                );
                UnmanagedMemory.free(fileNameNative);
                if(charset != null) UnmanagedMemory.free(charsetNative);
                return AnalysisUnit.wrap(resNative);
            } else {
                return JNI_LIB.gpr_get_analysis_unit_from_file(
                    this,
                    fileName,
                    charset,
                    reparse,
                    rule.toC()
                );
            }

        }

        /**
         * Get an analysis unit from the given buffer in the current context.
         *
         * @param buffer The buffer to parse.
         * @param name The name of the buffer.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromBuffer(
            final String buffer,
            final String name
        ) {
            return this.getUnitFromBuffer(
                buffer,
                name,
                null,
                DEFAULT_GRAMMAR_RULE
            );
        }

        /**
         * Get an analysis unit from the given buffer in the current context
         * with additional parameters.
         *
         * @param buffer The buffer to parse.
         * @param name The name of the buffer.
         * @param charset The charset of the buffer.
         * @param rule The rule to parse the buffer with.
         * @return The new analysis unit.
         */
        public AnalysisUnit getUnitFromBuffer(
            final String buffer,
            final String name,
            final String charset,
            final GrammarRule rule
        ) {

            if(ImageInfo.inImageCode()) {
                final CCharPointer bufferNative = toCString(buffer);
                final CCharPointer nameNative = toCString(name);
                final CCharPointer charsetNative =
                    charset == null ?
                    WordFactory.nullPointer() :
                    toCString(charset);

                final AnalysisUnitNative resNative =
                    NI_LIB.gpr_get_analysis_unit_from_buffer(
                    this.reference.ni(),
                    nameNative,
                    charsetNative,
                    bufferNative,
                    buffer.length(),
                    rule.toC()
                );
                UnmanagedMemory.free(bufferNative);
                UnmanagedMemory.free(nameNative);
                if(charset != null) UnmanagedMemory.free(charsetNative);
                return AnalysisUnit.wrap(resNative);
            } else {
                return JNI_LIB.gpr_get_analysis_unit_from_buffer(
                    this,
                    name,
                    charset,
                    buffer,
                    buffer.length(),
                    rule.toC()
                );
            }

        }


        /**
         * Increase the reference counter of the given context.
         *
         * @param context The context to increase the reference counter of.
         */
        private static void increaseRefCounter(
            final AnalysisContext context
        ) {
            // Increase the context reference counter of the context if not null
            if(!context.reference.isNull()) {
                if(ImageInfo.inImageCode()) {
                    NI_LIB.gpr_context_incref(context.reference.ni());
                } else {
                    JNI_LIB.gpr_context_incref(context.reference.jni());
                }
            }
        }

        /** @see java.lang.AutoCloseable#close() */
        @Override
        public void close() {

            if(ImageInfo.inImageCode()) {
                NI_LIB.gpr_context_decref(this.reference.ni());
            } else {
                JNI_LIB.gpr_context_decref(this.reference.jni());
            }
            checkException();

        }

        


    }

    /**
     * This type represents the analysis of a single file.
     *
     * This type has strong-reference semantics and is ref-counted.
     * Furthermore, a reference to a unit contains an implicit reference to the
     * context that owns it. This means that keeping a reference to a unit will
     * keep the context and all the unit it contains allocated.
     */
    public static final class AnalysisUnit {

        // ----- Class attributes -----

        /** Singleton that represents the none analysis unit. */
        public static final AnalysisUnit NONE = new AnalysisUnit(
            PointerWrapper.nullPointer()
        );

        // ----- Instance attributes -----

        /** The reference to the native analysis unit. */
        private final PointerWrapper reference;

        /** The cache for the unit root. */
        private GprNode root;

        // ----- Constructors -----

        /**
         * Create a new analysis unit from its value.
         *
         * @param reference The native analysis unit native value in
         * a pointer wrapper.
         */
        AnalysisUnit(
            final PointerWrapper reference
        ) {
            this.reference = reference;
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native analysis unit in the Java class.
         *
         * @param pointer The pointer the native analysis unit value.
         * @return The newly wrapped analysis unit.
         */
        static AnalysisUnit wrap(
            final Pointer pointer
        ) {
            return wrap((AnalysisUnitNative) pointer.readWord(0));
        }

        /**
         * Wrap a NI analysis unit native value in the Java class.
         *
         * @param analysisUnitNative The NI analysis unit native value.
         * @return The newly wrapped analysis unit.
         */
        static AnalysisUnit wrap(
            final AnalysisUnitNative analysisUnitNative
        ) {
            return new AnalysisUnit(new PointerWrapper(analysisUnitNative));
        }

        /**
         * Unwrap the analysis unit in the given pointer.
         *
         * @param pointer The pointer to place the native analysis unit in.
         */
        void unwrap(
            final Pointer pointer
        ) {
            pointer.writeWord(0, this.unwrap());
        }

        /**
         * Unwrap the analysis unit as a native value.
         *
         * @return The native analysis unit.
         */
        AnalysisUnitNative unwrap() {
            return (AnalysisUnitNative) this.reference.ni();
        }

        // ----- Instance methods -----

        /**
         * Get the root node of the analysis unit.
         *
         * @return The root node.
         */
        public GprNode getRoot() {
            if(this.root == null) {

                if(ImageInfo.inImageCode()) {
                    final EntityNative entityNative = StackValue.get(
                        EntityNative.class
                    );
                    NI_LIB.gpr_unit_root(
                        this.reference.ni(),
                        entityNative
                    );
                    this.root = GprNode.fromEntity(
                        Entity.wrap(entityNative)
                    );
                } else {
                    this.root = GprNode.fromEntity(
                        JNI_LIB.gpr_unit_root(this)
                    );
                }

            }
            return this.root;
        }

        /**
         * Get the analysis unit file name with its full path.
         *
         * @return The unit file name.
         */
        public String getFileName() {
            return this.getFileName(true);
        }

        /**
         * Get the unit's file name.
         *
         * @param fullPath If the method should return the
         * file full absolute path.
         * @return The file name.
         */
        @CompilerDirectives.TruffleBoundary
        public String getFileName(
            final boolean fullPath
        ) {
            final String absoluteFile;

            if(ImageInfo.inImageCode()) {
                final CCharPointer resNative = NI_LIB.gpr_unit_filename(
                    this.reference.ni()
                );
                absoluteFile = toJString(resNative);
                NI_LIB.gpr_free(resNative);
            } else {
                absoluteFile = JNI_LIB.gpr_unit_filename(this);
            }

            if(fullPath) {
                return absoluteFile;
            } else {
                return new File(absoluteFile).getName();
            }
        }

        /**
         * Get the number of tokens in the analysis unit.
         *
         * @return The number of token.
         */
        public int getTokenCount() {

            if(ImageInfo.inImageCode()) {
                return NI_LIB.gpr_unit_token_count(this.reference.ni());
            } else {
                return JNI_LIB.gpr_unit_token_count(this);
            }

        }

        /**
         * Get the number of trivia in the analysis unit.
         *
         * @return The number of trivia.
         */
        public int getTriviaCount() {

            if(ImageInfo.inImageCode()) {
                return NI_LIB.gpr_unit_trivia_count(
                    this.reference.ni()
                );
            } else {
                return JNI_LIB.gpr_unit_trivia_count(this);
            }

        }

        /**
         * Return the first token of the analysis unit.
         *
         * @return The first token.
         */
        public Token getFirstToken() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.gpr_unit_first_token(
                    this.reference.ni(),
                    tokenNative
                );
                return Token.wrap(tokenNative, this);
            } else {
                return JNI_LIB.gpr_unit_first_token(this);
            }

        }

        /**
         * Return the last token of the analysis unit.
         *
         * @return The last token.
         */
        public Token getLastToken() {

            if(ImageInfo.inImageCode()) {
                final TokenNative tokenNative = StackValue.get(
                    TokenNative.class
                );
                NI_LIB.gpr_unit_last_token(
                    this.reference.ni(),
                    tokenNative
                );
                return Token.wrap(tokenNative, this);
            } else {
                return JNI_LIB.gpr_unit_last_token(this);
            }

        }

        /**
         * Get the text of the analysis unit in a string.
         *
         * @return The text of the analysis unit source.
         */
        public String getText() {
            return Token.textRange(
                this.getFirstToken(),
                this.getLastToken()
            );
        }

        /**
         * Get the analysis context that owns the unit.
         *
         * @return The owning analysis context.
         */
        public AnalysisContext getContext() {

            if(ImageInfo.inImageCode()) {
                final AnalysisContextNative contextNative =
                    NI_LIB.gpr_unit_context(this.reference.ni());
                return AnalysisContext.wrap(contextNative);
            } else {
                return JNI_LIB.gpr_unit_context(this);
            }

        }

        /**
         * Get the list of assiated diagnositcs. Those are parsing errors.
         *
         * @return The diagnositcs of the unit.
         */
        public List<Diagnostic> getDiagnostics() {
            final int diagnosticCount;

            if(ImageInfo.inImageCode()) {
                diagnosticCount = NI_LIB.gpr_unit_diagnostic_count(
                    this.reference.ni()
                );
            } else {
                diagnosticCount = JNI_LIB.gpr_unit_diagnostic_count(
                    this
                );
            }

            final List<Diagnostic> res = new ArrayList<>(diagnosticCount);

            if(ImageInfo.inImageCode()) {
                final DiagnosticNative diagnosticNative = StackValue.get(
                    DiagnosticNative.class
                );
                for(int i = 0 ; i < diagnosticCount ; i++) {
                    NI_LIB.gpr_unit_diagnostic(
                        this.reference.ni(),
                        i,
                        diagnosticNative
                    );
                    res.add(Diagnostic.wrap(diagnosticNative));
                }
            } else {
                for(int i = 0 ; i < diagnosticCount ; i++) {
                    res.add(
                        JNI_LIB.gpr_unit_diagnostic(this, i)
                    );
                }
            }

            return res;
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            return "<AnalysisUnit \"" + this.getFileName(false) + "\">";
        }

        @Override
        public boolean equals(Object o) {
            if(this == o) return true;
            if(!(o instanceof AnalysisUnit)) return false;
            final AnalysisUnit other = (AnalysisUnit) o;
            return other.reference.equals(other.reference);
        }

    }

    // ===== Generated structure wrapping classes =====

        
        
    
    

    public static final class Metadata {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final Metadata NONE = new Metadata();

        // ----- Instance attributes ------

        /** The dummy field is always false (0) */
        final boolean dummy = false;

        // ----- Constructors -----

        /**
         * An empty constructor because the structure is empty
         */
        Metadata() {}

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The None instance because the structure is empty.
         */
        static Metadata wrap(
            final Pointer niPointer
        ) {
            return Metadata.NONE;
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final MetadataNative structNative
        ) {
            // Do nothing because the dummy field is useless
        }


    }

        
    
    

    /**

     */
    public static final class EntityInfo {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final EntityInfo NONE = new EntityInfo(
            Metadata.NONE,PointerWrapper.nullPointer(),false
        );

        // ----- Instance attributes -----

        public final
        Metadata
        md;
        public final
        PointerWrapper
        rebindings;
        public final
        boolean
        fromRebound;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        EntityInfo(
            final Metadata md,final PointerWrapper rebindings,final boolean fromRebound
        ) {
            this.md = md;
            this.rebindings = rebindings;
            this.fromRebound = fromRebound;
        }

        /**
         * Create a new structure with the field values.
         */
        public static EntityInfo create(
            final Metadata md,final PointerWrapper rebindings,final boolean fromRebound
        ) {
            return new EntityInfo(
                md,rebindings,fromRebound
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static EntityInfo wrap(
            final Pointer niPointer
        ) {
            return wrap((EntityInfoNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static EntityInfo wrap(
            final EntityInfoNative structNative
        ) {
            return new EntityInfo(
                Metadata.NONE,PointerWrapper.wrap(structNative.get_rebindings()),structNative.get_from_rebound() != 0
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final EntityInfoNative structNative
        ) {
            CCharPointer md_dummyNative = structNative.address_md_dummy();md_dummyNative.write(this.md.dummy ? (byte) 1 : (byte) 0);
            Pointer rebindingsNative = structNative.address_rebindings();rebindingsNative.writeWord(0, this.rebindings.ni());
            CCharPointer from_reboundNative = structNative.address_from_rebound();from_reboundNative.write(this.fromRebound ? (byte) 1 : (byte) 0);
        }


    }

    
    

    /**

     */
    public static final class Entity {

        // ----- Class attributes -----

        /** Singleton that represents the none value for the structure. */
        public static final Entity NONE = new Entity(
            PointerWrapper.nullPointer(),EntityInfo.NONE
        );

        // ----- Instance attributes -----

        public final
        PointerWrapper
        node;
        public final
        EntityInfo
        info;

        // ----- Constructors -----

        /**
         * Create a new structure object from the value if its fields.
         */
        Entity(
            final PointerWrapper node,final EntityInfo info
        ) {
            this.node = node;
            this.info = info;
        }

        /**
         * Create a new structure with the field values.
         */
        public static Entity create(
            final PointerWrapper node,final EntityInfo info
        ) {
            return new Entity(
                node,info
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to the native structure value in the Java class.
         *
         * @param niPointer The pointer to the NI structure native value.
         * @return The newly wrapped structure.
         */
        static Entity wrap(
            final Pointer niPointer
        ) {
            return wrap((EntityNative) niPointer.readWord(0));
        }

        /**
         * Wrap the given structure native value in the Java class.
         *
         * @param structNative The NI structure native value.
         * @return The newly wrapped structure.
         */
        static Entity wrap(
            final EntityNative structNative
        ) {
            return new Entity(
                PointerWrapper.wrap(structNative.get_node()),new EntityInfo(Metadata.NONE, PointerWrapper.wrap(structNative.get_info_rebindings()), structNative.get_info_from_rebound() != 0)
            );
        }

        /**
         * Unwrap the structure in the given native value.
         *
         * @param structNative The NI structure native value to fill.
         */
        void unwrap(
            final EntityNative structNative
        ) {
            Pointer nodeNative = structNative.address_node();nodeNative.writeWord(0, this.node.ni());
            CCharPointer info_md_dummyNative = structNative.address_info_md_dummy();info_md_dummyNative.write(this.info.md.dummy ? (byte) 1 : (byte) 0);
            Pointer info_rebindingsNative = structNative.address_info_rebindings();info_rebindingsNative.writeWord(0, this.info.rebindings.ni());
            CCharPointer info_from_reboundNative = structNative.address_info_from_rebound();info_from_reboundNative.write(this.info.fromRebound ? (byte) 1 : (byte) 0);
        }


    }

        
        

    // ===== Generated array wrapping classes =====

    /**
     * This class is the base of all array wrapping class.
     */
    public static abstract class ArrayBase<T> implements Iterable<T> {

        // ----- Attributes -----

        /** The content of the array. */
        protected final T[] content;

        // ----- Constructors -----

        /**
         * Protected constructor.
         */
        protected ArrayBase(
            final T[] content
        ) {
            this.content = content;
        }

        // ----- Instance methods -----

        /**
         * Get the size of the array.
         *
         * @return The size of the native array.
         */
        public int size() {
            return this.content.length;
        }

        /**
         * Get the element at the given place in the array.
         *
         * @param i The index of the element to get.
         * @return The element at the given index.
         * @throws ArrayIndexOutOfBoundsException If the requested index is
         * out of bounds.
         */
        public T get(
            final int i
        ) {
            return this.content[i];
        }

        /**
         * Set the element at the given index.
         *
         * @param i The index of the element to set.
         * @param elem The element to place in the array.
         * @throws ArrayIndexOutOfBoundsException If the requested index is
         * out of bounds.
         */
        public void set(
            final int i,
            final T elem
        ) {
            this.content[i] = elem;
        }

        /** @see java.lang.Iterable#iterator() */
        @Override
        public Iterator<T> iterator() {
            return new LangkitArrayIterator<T>(this);
        }

        // ----- Override methods -----

        @Override
        public String toString() {
            final StringBuilder res = new StringBuilder("[");
            for(int i = 0 ; i < this.size() ; i++) {
                res.append(this.get(i).toString());
                if(i < this.size() - 1) res.append(", ");
            }
            res.append(']');
            return res.toString();
        }

        // ----- Inner classes -----

        /**
         * The iterator class for the langkit arrays
         */
        protected static class LangkitArrayIterator<U>
        implements Iterator<U> {

            // ----- Attributes -----

            /** The array to iterate on. */
            private final ArrayBase<U> array;

            /** The current index. */
            private int index;

            // ----- Constructors -----

            /**
             * Create a new array iterator.
             *
             * @param array The array to iterate on.
             */
            LangkitArrayIterator(
                final ArrayBase<U> array
            ) {
                this.array = array;
                this.index = 0;
            }

            // ----- Instance methods -----

            /** @see java.util.Iterator#hasNext() */
            @Override
            public boolean hasNext() {
                return this.index < this.array.size();
            }

            /** @see java.util.Iterator#next() */
            @Override
            public U next() {
                return this.array.get(this.index++);
            }

        }

    }

    
    

    /**
     * This class represents the gpr_gpr_node_array Java wrapping class
     */
    public static final class
    GprNodeArray extends ArrayBase<GprNode> {

        // ----- Class attributes -----

        /** Singleton that represents the none array. */
        public static final GprNodeArray NONE = new GprNodeArray(
            new GprNode[0]
        );

        // ----- Constructors -----

        /**
         * Create a new array with the given content.
         *
         * @param content The content of the array.
         */
        GprNodeArray(
            final GprNode[] content
        ) {
            super(content);
        }

        /**
         * Create a new array from the JNI stub.
         *
         * @param content The unwrapped JNI content.
         */
        private static GprNodeArray jniCreate(
            final Entity[] jniContent
        ) {
            final GprNode[] content =
                new GprNode[jniContent.length];
            for(int i = 0 ; i < content.length ; i++) {
                content[i] =
                    GprNode.fromEntity(jniContent[i]);
            }
            return new GprNodeArray(content);
        }

        /**
         * Create a sized array.
         *
         * @param size The size of the array you want to create.
         * @return The newly created array.
         */
        public static GprNodeArray create(
            final int size
        ) {
            return new GprNodeArray(
                new GprNode[size]
            );
        }

        // ----- Graal C API methods -----

        /**
         * Wrap a pointer to an array native value in the Java class.
         *
         * @param pointer The pointer to the array NI native value.
         * @return The newly wrapped array.
         */
        static GprNodeArray wrap(
            final Pointer pointer
        ) {
            return wrap((GprNodeArrayNative) pointer.readWord(0));
        }

        /**
         * Wrap an array native value in the Java class.
         *
         * @param nativeArray The NI array native value to wrap.
         * @return The newly wrapped array.
         */
        static GprNodeArray wrap(
            final GprNodeArrayNative nativeArray
        ) {
            // Get the size and prepare the working variables
            final int size = nativeArray.get_n();
            final GprNode[] content = new GprNode[size];
            final Pointer nativeItems = nativeArray.address_items();
            Pointer nativeItem;
            EntityNative toRead;

            // Iterate over all array elements
            for(int i = 0 ; i < size ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(EntityNative.class)
                );
                toRead = WordFactory.unsigned(nativeItem.rawValue());
                content[i] = GprNode.fromEntity(Entity.wrap(toRead));
            }

            // Return the new langkit array
            return new GprNodeArray(content);
        }

        /**
         * Unwrap the array in the given pointer
         *
         * @param pointer The pointer to place the native array pointer
         * in.
         */
        void unwrap(
            final Pointer pointer
            
        ) {
            // Create a new native array with the size
            final GprNodeArrayNative resNative = this.unwrap(
                
            );

            // Place the result in the pointer
            pointer.writeWord(0, resNative);
        }

        /**
         * Allocate a new native array and unwrap inside.
         *
         * @return The newly allocated unwraped array.
         */
        GprNodeArrayNative unwrap(
            
        ) {
            // Create a new native array with the size
            final GprNodeArrayNative res = NI_LIB.gpr_gpr_node_array_create(
                this.content.length
            );

            // Prepare the working vars
            final Pointer nativeItems = res.address_items();
            Pointer nativeItem;
            EntityNative toWrite;

            // Place all elements in the native array
            for(int i = 0 ; i < this.content.length ; i++) {
                nativeItem = nativeItems.add(
                    i * SizeOf.get(EntityNative.class)
                );
                toWrite = WordFactory.unsigned(
                    nativeItem.rawValue()
                );
                this.content[i].entity.unwrap(toWrite);
            }

            // Return the result
            return res;
        }

        /**
         * Release native array pointer by the given pointer.
         *
         * @param The pointer to the array to release.
         */
        static void release(
            final Pointer pointer
        ) {
            release((GprNodeArrayNative) pointer.readWord(0));
        }

        /**
         * Release the given native array.
         *
         * @param arrayNative The native array to release.
         */
        static void release(
            final GprNodeArrayNative arrayNative
        ) {
            NI_LIB.gpr_gpr_node_array_dec_ref(arrayNative);
        }

        // ----- Getters -----

        /**
         * Get the content in an array unwrapped for the JNI stubs.
         *
         * @return The content unwrapped.
         */
        private Entity[] jniContent() {
            final Entity[] res =
                new Entity[this.content.length];
            for(int i = 0 ; i < res.length ; i++) {
                res[i] = (this.content[i] != null ? this.content[i].entity : null);
            }
            return res;
        }

    }


    // ===== Generated iterator wrapping classes =====


    // ===== Node classes =====

    /**
     * Data type for all nodes. Nodes are assembled to make up a tree.  See the
     * node primitives below to inspect such trees.
     *
     * Unlike for contexts and units, this type has weak-reference semantics:
     * keeping a reference to a node has no effect on the decision to keep the
     * unit that it owns allocated. This means that once all references to the
     * context and units related to a node are dropped, the context and its
     * units are deallocated and the node becomes a stale reference: most
     * operations on it will raise a ``Stale_Reference_Error``.
     *
     * Note that since reparsing an analysis unit deallocates all the nodes it
     * contains, this operation makes all reference to these nodes stale as
     * well.
     */
    public static abstract class GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "GprNode";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = GprNode.class.getMethod(
                        "parent",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "parent",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GprNode.class.getMethod(
                        "parents",
                        new Class[]{boolean.class}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();
                    parameters.add(new ParamWithDefaultValue(
                        boolean.class,
                        "withSelf",
                        true
                    ));

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "parents",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GprNode.class.getMethod(
                        "children",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "children",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GprNode.class.getMethod(
                        "tokenStart",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "token_start",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GprNode.class.getMethod(
                        "tokenEnd",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "token_end",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GprNode.class.getMethod(
                        "childIndex",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "child_index",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GprNode.class.getMethod(
                        "previousSibling",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "previous_sibling",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GprNode.class.getMethod(
                        "nextSibling",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "next_sibling",
                        new GprParserField(method, parameters)
                    );
                }
                
                
                {
                    // Get the Java method of the field
                    Method method = GprNode.class.getMethod(
                        "isGhost",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "is_ghost",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = GprNode.class.getMethod(
                        "fullSlocImage",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "full_sloc_image",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GprNode NONE = new NoneNode();

        /** The entity of the node. */
        public final Entity entity;

        /** The analysis unit that owns the node. */
        protected AnalysisUnit unit;

        /** The cache for the image of the node. */
        protected String image;

        // ----- Constructors -----

        /**
         * Create a new node with its entity.
         *
         * @param entity The node's entity.
         */
        protected GprNode(
            final Entity entity
        ) {
            this.entity = entity;
            this.unit = null;
            this.image = null;
        }

        /**
         * Get a node from the given entity.
         *
         * @param entity The entity to get the node from.
         * @return The newly created node.
         */
        public static GprNode fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GprNode.NONE :
                dispatchNodeCreation(entity);
        }

        /**
         * Dispatch the node creation to return the valid Java object
         * according to the node kind.
         *
         * @param entity The entity to create the node from.
         * @return The wrapped node in the correct class.
         */
        protected static GprNode dispatchNodeCreation(
            final Entity entity
        ) {
            int nodeKind = -1;

            if(ImageInfo.inImageCode()) {
                EntityNative entityNative = StackValue.get(
                    EntityNative.class
                );
                entity.unwrap(entityNative);
                nodeKind = NI_LIB.gpr_node_kind(entityNative);
            } else {
                nodeKind = JNI_LIB.gpr_node_kind(entity);
            }

            switch(nodeKind) {
                case 1:
                    return entity.node.isNull() ?
                        AllQualifierAbsent.NONE :
                        new AllQualifierAbsent(entity);
                case 2:
                    return entity.node.isNull() ?
                        AllQualifierPresent.NONE :
                        new AllQualifierPresent(entity);
                case 3:
                    return entity.node.isNull() ?
                        AttributeDecl.NONE :
                        new AttributeDecl(entity);
                case 4:
                    return entity.node.isNull() ?
                        AttributeReference.NONE :
                        new AttributeReference(entity);
                case 5:
                    return entity.node.isNull() ?
                        CaseItemList.NONE :
                        new CaseItemList(entity);
                case 6:
                    return entity.node.isNull() ?
                        GprNodeList.NONE :
                        new GprNodeList(entity);
                case 7:
                    return entity.node.isNull() ?
                        Choices.NONE :
                        new Choices(entity);
                case 8:
                    return entity.node.isNull() ?
                        TermList.NONE :
                        new TermList(entity);
                case 9:
                    return entity.node.isNull() ?
                        IdentifierList.NONE :
                        new IdentifierList(entity);
                case 10:
                    return entity.node.isNull() ?
                        StringLiteralList.NONE :
                        new StringLiteralList(entity);
                case 11:
                    return entity.node.isNull() ?
                        TermListList.NONE :
                        new TermListList(entity);
                case 12:
                    return entity.node.isNull() ?
                        WithDeclList.NONE :
                        new WithDeclList(entity);
                case 13:
                    return entity.node.isNull() ?
                        BuiltinFunctionCall.NONE :
                        new BuiltinFunctionCall(entity);
                case 14:
                    return entity.node.isNull() ?
                        CaseConstruction.NONE :
                        new CaseConstruction(entity);
                case 15:
                    return entity.node.isNull() ?
                        CaseItem.NONE :
                        new CaseItem(entity);
                case 16:
                    return entity.node.isNull() ?
                        CompilationUnit.NONE :
                        new CompilationUnit(entity);
                case 17:
                    return entity.node.isNull() ?
                        EmptyDecl.NONE :
                        new EmptyDecl(entity);
                case 18:
                    return entity.node.isNull() ?
                        Prefix.NONE :
                        new Prefix(entity);
                case 19:
                    return entity.node.isNull() ?
                        Identifier.NONE :
                        new Identifier(entity);
                case 20:
                    return entity.node.isNull() ?
                        NumLiteral.NONE :
                        new NumLiteral(entity);
                case 21:
                    return entity.node.isNull() ?
                        StringLiteral.NONE :
                        new StringLiteral(entity);
                case 22:
                    return entity.node.isNull() ?
                        LimitedAbsent.NONE :
                        new LimitedAbsent(entity);
                case 23:
                    return entity.node.isNull() ?
                        LimitedPresent.NONE :
                        new LimitedPresent(entity);
                case 24:
                    return entity.node.isNull() ?
                        OthersDesignator.NONE :
                        new OthersDesignator(entity);
                case 25:
                    return entity.node.isNull() ?
                        PackageDecl.NONE :
                        new PackageDecl(entity);
                case 26:
                    return entity.node.isNull() ?
                        PackageExtension.NONE :
                        new PackageExtension(entity);
                case 27:
                    return entity.node.isNull() ?
                        PackageRenaming.NONE :
                        new PackageRenaming(entity);
                case 28:
                    return entity.node.isNull() ?
                        PackageSpec.NONE :
                        new PackageSpec(entity);
                case 29:
                    return entity.node.isNull() ?
                        Project.NONE :
                        new Project(entity);
                case 30:
                    return entity.node.isNull() ?
                        ProjectDeclaration.NONE :
                        new ProjectDeclaration(entity);
                case 31:
                    return entity.node.isNull() ?
                        ProjectExtension.NONE :
                        new ProjectExtension(entity);
                case 32:
                    return entity.node.isNull() ?
                        ProjectQualifierAbstract.NONE :
                        new ProjectQualifierAbstract(entity);
                case 33:
                    return entity.node.isNull() ?
                        ProjectQualifierAggregate.NONE :
                        new ProjectQualifierAggregate(entity);
                case 34:
                    return entity.node.isNull() ?
                        ProjectQualifierAggregateLibrary.NONE :
                        new ProjectQualifierAggregateLibrary(entity);
                case 35:
                    return entity.node.isNull() ?
                        ProjectQualifierConfiguration.NONE :
                        new ProjectQualifierConfiguration(entity);
                case 36:
                    return entity.node.isNull() ?
                        ProjectQualifierLibrary.NONE :
                        new ProjectQualifierLibrary(entity);
                case 37:
                    return entity.node.isNull() ?
                        ProjectQualifierStandard.NONE :
                        new ProjectQualifierStandard(entity);
                case 38:
                    return entity.node.isNull() ?
                        StringLiteralAt.NONE :
                        new StringLiteralAt(entity);
                case 39:
                    return entity.node.isNull() ?
                        Terms.NONE :
                        new Terms(entity);
                case 40:
                    return entity.node.isNull() ?
                        TypeReference.NONE :
                        new TypeReference(entity);
                case 41:
                    return entity.node.isNull() ?
                        TypedStringDecl.NONE :
                        new TypedStringDecl(entity);
                case 42:
                    return entity.node.isNull() ?
                        VariableDecl.NONE :
                        new VariableDecl(entity);
                case 43:
                    return entity.node.isNull() ?
                        VariableReference.NONE :
                        new VariableReference(entity);
                case 44:
                    return entity.node.isNull() ?
                        WithDecl.NONE :
                        new WithDecl(entity);
                default:
                    throw new EnumException(
                        "Cannot find the node type from " + nodeKind
                    );
            }
        }

        // ----- Getters -----

        public String getKindName() {
            return GprNode.kindName;
        }

        public String[] getFieldNames() {
            return GprNode.fieldNames;
        }

        public boolean isListType() {
            return GprNode.isListType;
        }

        public boolean isTokenNode() {

            if(ImageInfo.inImageCode()) {
                EntityNative entityNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(entityNative);
                return NI_LIB.gpr_node_is_token_node(
                    entityNative
                ) != 0;
            } else {
                return JNI_LIB.gpr_node_is_token_node(this.entity);
            }

        }

        public boolean isNone() {
            return this.entity.node.isNull();
        }

        // ----- Instance methods -----

        /**
         * Get the analysis unit of the node.
         *
         * @return The unit which owns the node.
         */
        public AnalysisUnit getUnit() {
            if(this.unit == null) {

                if(ImageInfo.inImageCode()) {
                    final EntityNative entityNative = StackValue.get(
                        EntityNative.class
                    );
                    this.entity.unwrap(entityNative);

                    final AnalysisUnitNative unitNative =
                        NI_LIB.gpr_node_unit(entityNative);
                    this.unit = AnalysisUnit.wrap(unitNative);
                } else {
                    this.unit = JNI_LIB.gpr_node_unit(this.entity);
                }

            }
            return this.unit;
        }

        /**
         * Get the descritpion of a field from its name.
         *
         * @param name The langkit field name to get the description for.
         * @return The Java description of the langkit field.
         */
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(String name) {
            return GprNode.fieldDescriptions.get(name);
        }

        /**
         * Get the children count of the node.
         *
         * @return The children count.
         */
        public int getChildrenCount() {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                return NI_LIB.gpr_node_children_count(thisNative);
            } else {
                return JNI_LIB.gpr_node_children_count(this.entity);
            }

        }

        /**
         * Get the child at the given position.
         *
         * @param n The index of the child to get.
         * @return The child at the given index.
         */
        public GprNode getChild(
            final int n
        ) {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final EntityNative resNative = StackValue.get(
                    EntityNative.class
                );
                NI_LIB.gpr_node_child(
                    thisNative,
                    n,
                    resNative
                );

                return fromEntity(Entity.wrap(resNative));
            } else {
                return fromEntity(JNI_LIB.gpr_node_child(
                    this.entity,
                    n
                ));
            }

        }

        /**
         * Get the text of the node.
         *
         * @return The text of the node.
         */
        public String getText() {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final TextNative resNative = StackValue.get(TextNative.class);
                Text.NONE.unwrap(resNative);
                NI_LIB.gpr_node_text(
                    thisNative,
                    resNative
                );

                try(final Text resText = Text.wrap(resNative)) {
                    return resText.getContent();
                }
            } else {
                try(
                    final Text resText = JNI_LIB.gpr_node_text(
                        this.entity
                    );
                ) {
                    return resText.getContent();
                }
            }

        }

        /**
         * Get the image of the node.
         *
         * @return The node's image.
         */
        public String getImage() {
            if(this.image == null) {

                if(ImageInfo.inImageCode()) {
                    final EntityNative thisNative = StackValue.get(
                        EntityNative.class
                    );
                    this.entity.unwrap(thisNative);

                    final TextNative resNative = StackValue.get(
                        TextNative.class
                    );
                    Text.NONE.unwrap(resNative);
                    NI_LIB.gpr_node_image(thisNative, resNative);

                    try(final Text resText = Text.wrap(resNative)) {
                        this.image = resText.getContent();
                    }
                } else {
                    try(
                        final Text resText = JNI_LIB.gpr_node_image(
                            this.entity
                        )
                    ) {
                        this.image = resText.getContent();
                    }
                }

            }

            return this.image;
        }

        /**
         * Get the source location range of the node.
         *
         * @return The source location range of the node.
         */
        public SourceLocationRange getSourceLocationRange() {

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final SourceLocationRangeNative resNative = StackValue.get(
                    SourceLocationRangeNative.class
                );
                NI_LIB.gpr_node_sloc_range(
                    thisNative,
                    resNative
                );

                return SourceLocationRange.wrap(resNative);
            } else {
                return JNI_LIB.gpr_node_sloc_range(this.entity);
            }

        }

        // ----- Dumping methods -----

        /**
         * Return the AST in a string.
         *
         * @return The string containing the representation of the AST
         * from the node.
         */
        @CompilerDirectives.TruffleBoundary
        public String dumpAST() {
            final StringBuilder builder = new StringBuilder();
            this.dumpAST(builder);
            return builder.toString();
        }

        /**
         * Dump the AST in the given string builder.
         *
         * @param builder The builder to dump the AST in.
         */
        @CompilerDirectives.TruffleBoundary
        public void dumpAST(
            final StringBuilder builder
        ) {
            this.dumpAST(builder, "");
        }

        /**
         * Dump a node field in the given string builder.
         *
         * @param builder The string builder to put the file in.
         * @param indent The current indentation string.
         * @param name The name of the field.
         * @param value The value of the field.
         */
        protected static void dumpField(
            final StringBuilder builder,
            final String indent,
            final String name,
            final GprNode value
        ) {
            builder.append(indent)
                .append(name)
                .append(":\n");
            value.dumpAST(builder, indent + "  ");
        }

        /**
         * Dump the AST in the given string builder with the indent level.
         *
         * @param builder The builder to dump the AST in.
         * @param indent The starting indent level.
         */
        @CompilerDirectives.TruffleBoundary
        protected void dumpAST(
            final StringBuilder builder,
            String indent
        ) {
            // Prepare the working variables
            String image = this.getImage();
            image = image.substring(1, image.length());
            final int childrenCount = this.getChildrenCount();

            // Print the node
            builder.append(indent)
                .append(image);
            if(this.isTokenNode()) {
                builder.append(": ")
                    .append(this.getText());
            }
            builder.append('\n');

            // Print the field of the node
            indent = indent + "|";
            if(this.isListType()) {
                for(int i = 0 ; i < childrenCount ; i++) {
                    final GprNode child = this.getChild(i);
                    dumpField(builder, indent, "item_" + i, child);
                }
            } else {
                for(int i = 0 ; i < childrenCount ; i++) {
                    final GprNode child = this.getChild(i);
                    final String name = this.getFieldNames()[i];
                    dumpField(builder, indent, name, child);
                }
            }
        }

        // ----- Visitor methods -----

        /**
         * Accept the given visitor.
         *
         * @param visitor The visitor to accept.
         * @return The result of the visit.
         */
        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        /**
         * Accept the given visitor.
         *
         * @param visitor The visitor to accept.
         * @param param The parameter of the visit.
         * @return The result of the visit.
         */
        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * Return the syntactic parent for this node. Return null for the root
         * node.
         */
        public GprNode parent(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_gpr_node_parent(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GprNode res = GprNode.NONE;
                if(propException == null) {
                    res = GprNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_gpr_node_parent(
                    this.entity
                );

                // Wrap and return the result
                return GprNode.fromEntity(res);
            }

        }

        
    

        /**
         * Return an array that contains the lexical parents, this node
         * included iff ``with_self`` is True. Nearer parents are first in the
         * list.
         */
        public GprNodeArray parents(
            final boolean withSelf
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments
                byte withSelfNative = (withSelf ? (byte) 1 : (byte) 0);

                // Create the result native
                final Pointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.gpr_gpr_node_parents(
                    thisNative,
                    withSelfNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GprNodeArray res = GprNodeArray.NONE;
                if(propException == null) {
                    res = GprNodeArray.wrap(resNative);

                    GprNodeArray.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final GprNodeArray res = JNI_LIB.gpr_gpr_node_parents(
                    withSelf,
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return an array that contains the direct lexical children.
         *
         * .. warning:: This constructs a whole array every-time you call it,
         *    and as such is less efficient than calling the ``Child`` built-
         *    in.
         */
        public GprNodeArray children(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final Pointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.gpr_gpr_node_children(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GprNodeArray res = GprNodeArray.NONE;
                if(propException == null) {
                    res = GprNodeArray.wrap(resNative);

                    GprNodeArray.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final GprNodeArray res = JNI_LIB.gpr_gpr_node_children(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the first token used to parse this node.
         */
        public Token tokenStart(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                // Get the node unit
                final AnalysisUnit currentUnit = this.getUnit();


                // Unwrap the arguments

                // Create the result native
                final TokenNative resNative =
                    StackValue.get(TokenNative.class);

                // Call the native function
                NI_LIB.gpr_gpr_node_token_start(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Token res = Token.NONE(currentUnit);
                if(propException == null) {
                    res = Token.wrap(resNative, currentUnit);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Token res = JNI_LIB.gpr_gpr_node_token_start(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the last token used to parse this node.
         */
        public Token tokenEnd(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                // Get the node unit
                final AnalysisUnit currentUnit = this.getUnit();


                // Unwrap the arguments

                // Create the result native
                final TokenNative resNative =
                    StackValue.get(TokenNative.class);

                // Call the native function
                NI_LIB.gpr_gpr_node_token_end(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Token res = Token.NONE(currentUnit);
                if(propException == null) {
                    res = Token.wrap(resNative, currentUnit);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Token res = JNI_LIB.gpr_gpr_node_token_end(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the 0-based index for Node in its parent's children.
         */
        public int childIndex(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CIntPointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.gpr_gpr_node_child_index(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                int res = 0;
                if(propException == null) {
                    res = resNative.read();

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final int res = JNI_LIB.gpr_gpr_node_child_index(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return the node's previous sibling, or null if there is no such
         * sibling.
         */
        public GprNode previousSibling(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_gpr_node_previous_sibling(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GprNode res = GprNode.NONE;
                if(propException == null) {
                    res = GprNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_gpr_node_previous_sibling(
                    this.entity
                );

                // Wrap and return the result
                return GprNode.fromEntity(res);
            }

        }

        
    

        /**
         * Return the node's next sibling, or null if there is no such sibling.
         */
        public GprNode nextSibling(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_gpr_node_next_sibling(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GprNode res = GprNode.NONE;
                if(propException == null) {
                    res = GprNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_gpr_node_next_sibling(
                    this.entity
                );

                // Wrap and return the result
                return GprNode.fromEntity(res);
            }

        }

        
    


        
    

        /**
         * Return whether the node is a ghost.
         *
         * Unlike regular nodes, ghost nodes cover no token in the input
         * source: they are logically located instead between two tokens. Both
         * the ``token_start`` and the ``token_end`` of all ghost nodes is the
         * token right after this logical position.
         */
        public boolean isGhost(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.gpr_gpr_node_is_ghost(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = (resNative.read() != 0);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.gpr_gpr_node_is_ghost(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }

        
    

        /**
         * Return a string containing the filename + the sloc in GNU conformant
         * format. Useful to create diagnostics from a node.
         */
        public String fullSlocImage(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final Pointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.gpr_gpr_node_full_sloc_image(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                String res = StringWrapper.NONE;
                if(propException == null) {
                    res = StringWrapper.wrap(resNative);

                    StringWrapper.release(
                        resNative
                    );
                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final String res = JNI_LIB.gpr_gpr_node_full_sloc_image(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }


        // ----- Override methods -----

        @Override
        public String toString() {
            return this.getImage();
        }

        @Override
        public boolean equals(Object o) {
            if(this == o) return true;
            if(!(o instanceof GprNode)) return false;
            final GprNode other = (GprNode) o;

            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                final EntityNative otherNative = StackValue.get(
                    EntityNative.class
                );
                other.entity.unwrap(otherNative);

                return NI_LIB.gpr_node_is_equivalent(
                    thisNative,
                    otherNative
                ) != 0;
            } else {
                return JNI_LIB.gpr_node_is_equivalent(
                    this.entity, other.entity
                ) != 0;
            }
        }

        @Override
        public int hashCode() {
            if(ImageInfo.inImageCode()) {
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);

                return NI_LIB.gpr_node_hash(thisNative);
            } else {
                return JNI_LIB.gpr_node_hash(this.entity);
            }

        }

        // ----- Inner classes -----

        /**
         * This class represents the none node without any concrete type.
         */
        private static final class NoneNode extends GprNode {
            NoneNode() {super(Entity.NONE);}
        }

    }

    // ===== Generated AST node wrapping classes =====

    
    

    /**

     */
    public static abstract
    class AllQualifier extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AllQualifier";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = AllQualifier.class.getMethod(
                        "pAsBool",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_as_bool",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AllQualifier NONE =
            new AllQualifierNone();

        // ----- Constructors -----

        protected AllQualifier(
            final Entity entity
        ) {
            super(entity);
        }

        public static AllQualifier fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AllQualifier.NONE :

                (AllQualifier) GprNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AllQualifier.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AllQualifier.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AllQualifier.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return AllQualifier.fieldDescriptions.getOrDefault(name, null);
        }


        // ----- Field accessors -----

        
    

        /**
         * Return whether this is an instance of AllQualifierPresent
         */
        public boolean pAsBool(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.gpr_all_qualifier_p_as_bool(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = (resNative.read() != 0);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.gpr_all_qualifier_p_as_bool(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * AllQualifier.
         */
        private static final class AllQualifierNone extends AllQualifier {
            AllQualifierNone() {super(Entity.NONE);}
        }

    }


    
    

    /**

     */
    public static 
    class AllQualifierAbsent extends AllQualifier {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AllQualifierAbsent";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            AllQualifier.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AllQualifierAbsent NONE =
            new AllQualifierAbsent(
                Entity.NONE
            );

        // ----- Constructors -----

        protected AllQualifierAbsent(
            final Entity entity
        ) {
            super(entity);
        }

        public static AllQualifierAbsent fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AllQualifierAbsent.NONE :

                new AllQualifierAbsent(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AllQualifierAbsent.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AllQualifierAbsent.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AllQualifierAbsent.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return AllQualifierAbsent.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class AllQualifierPresent extends AllQualifier {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AllQualifierPresent";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            AllQualifier.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AllQualifierPresent NONE =
            new AllQualifierPresent(
                Entity.NONE
            );

        // ----- Constructors -----

        protected AllQualifierPresent(
            final Entity entity
        ) {
            super(entity);
        }

        public static AllQualifierPresent fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AllQualifierPresent.NONE :

                new AllQualifierPresent(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AllQualifierPresent.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AllQualifierPresent.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AllQualifierPresent.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return AllQualifierPresent.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class AttributeDecl extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AttributeDecl";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fAttrName","fAttrIndex","fExpr"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = AttributeDecl.class.getMethod(
                        "fAttrName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_attr_name",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AttributeDecl.class.getMethod(
                        "fAttrIndex",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_attr_index",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AttributeDecl.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_expr",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AttributeDecl NONE =
            new AttributeDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected AttributeDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static AttributeDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AttributeDecl.NONE :

                new AttributeDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AttributeDecl.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AttributeDecl.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AttributeDecl.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return AttributeDecl.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public Identifier fAttrName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_attribute_decl_f_attr_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Identifier res = Identifier.NONE;
                if(propException == null) {
                    res = Identifier.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_attribute_decl_f_attr_name(
                    this.entity
                );

                // Wrap and return the result
                return Identifier.fromEntity(res);
            }

        }

        
    

        /**
         * This field can contain one of the following nodes:
         * ``OthersDesignator``, ``StringLiteralAt``
         *
         * This field may be null even when there are no parsing errors.
         */
        public GprNode fAttrIndex(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_attribute_decl_f_attr_index(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GprNode res = GprNode.NONE;
                if(propException == null) {
                    res = GprNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_attribute_decl_f_attr_index(
                    this.entity
                );

                // Wrap and return the result
                return GprNode.fromEntity(res);
            }

        }

        
    

        /**
         * This field contains a list that itself contains one of the following
         * nodes: ``BuiltinFunctionCall``, ``StringLiteralAt``, ``Terms``,
         * ``VariableReference``
         *
         * When there are no parsing errors, this field is never null.
         */
        public TermList fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_attribute_decl_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TermList res = TermList.NONE;
                if(propException == null) {
                    res = TermList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_attribute_decl_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return TermList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class AttributeReference extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "AttributeReference";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fAttributeName","fAttributeIndex"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = AttributeReference.class.getMethod(
                        "fAttributeName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_attribute_name",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = AttributeReference.class.getMethod(
                        "fAttributeIndex",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_attribute_index",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final AttributeReference NONE =
            new AttributeReference(
                Entity.NONE
            );

        // ----- Constructors -----

        protected AttributeReference(
            final Entity entity
        ) {
            super(entity);
        }

        public static AttributeReference fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                AttributeReference.NONE :

                new AttributeReference(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return AttributeReference.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return AttributeReference.fieldNames;
        }

        @Override
        public boolean isListType() {
            return AttributeReference.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return AttributeReference.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public Identifier fAttributeName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_attribute_reference_f_attribute_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Identifier res = Identifier.NONE;
                if(propException == null) {
                    res = Identifier.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_attribute_reference_f_attribute_name(
                    this.entity
                );

                // Wrap and return the result
                return Identifier.fromEntity(res);
            }

        }

        
    

        /**
         * This field can contain one of the following nodes:
         * ``OthersDesignator``, ``StringLiteral``
         *
         * This field may be null even when there are no parsing errors.
         */
        public GprNode fAttributeIndex(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_attribute_reference_f_attribute_index(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GprNode res = GprNode.NONE;
                if(propException == null) {
                    res = GprNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_attribute_reference_f_attribute_index(
                    this.entity
                );

                // Wrap and return the result
                return GprNode.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static abstract
    class BaseList extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "BaseList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BaseList NONE =
            new BaseListNone();

        // ----- Constructors -----

        protected BaseList(
            final Entity entity
        ) {
            super(entity);
        }

        public static BaseList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BaseList.NONE :

                (BaseList) GprNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return BaseList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return BaseList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return BaseList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return BaseList.fieldDescriptions.getOrDefault(name, null);
        }


        // ----- Field accessors -----


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * BaseList.
         */
        private static final class BaseListNone extends BaseList {
            BaseListNone() {super(Entity.NONE);}
        }

    }


    
    

    /**
     * List of CaseItem.
     */
    public static 
    class CaseItemList extends BaseList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "CaseItemList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            BaseList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final CaseItemList NONE =
            new CaseItemList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected CaseItemList(
            final Entity entity
        ) {
            super(entity);
        }

        public static CaseItemList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                CaseItemList.NONE :

                new CaseItemList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return CaseItemList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return CaseItemList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return CaseItemList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return CaseItemList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * List of GprNode.
     *
     * This list node can contain one of the following nodes:
     * ``AttributeDecl``, ``BuiltinFunctionCall``, ``CaseConstruction``,
     * ``EmptyDecl``, ``OthersDesignator``, ``PackageDecl``,
     * ``StringLiteralAt``, ``StringLiteral``, ``Terms``, ``TypedStringDecl``,
     * ``VariableDecl``, ``VariableReference``
     */
    public static 
    class GprNodeList extends BaseList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "GprNodeList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            BaseList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final GprNodeList NONE =
            new GprNodeList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected GprNodeList(
            final Entity entity
        ) {
            super(entity);
        }

        public static GprNodeList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                GprNodeList.NONE :

                new GprNodeList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return GprNodeList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return GprNodeList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return GprNodeList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return GprNodeList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * This list node can contain one of the following nodes:
     * ``OthersDesignator``, ``StringLiteral``
     */
    public static 
    class Choices extends GprNodeList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "Choices";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNodeList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final Choices NONE =
            new Choices(
                Entity.NONE
            );

        // ----- Constructors -----

        protected Choices(
            final Entity entity
        ) {
            super(entity);
        }

        public static Choices fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                Choices.NONE :

                new Choices(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return Choices.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return Choices.fieldNames;
        }

        @Override
        public boolean isListType() {
            return Choices.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return Choices.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * This list node can contain one of the following nodes:
     * ``BuiltinFunctionCall``, ``StringLiteralAt``, ``Terms``,
     * ``VariableReference``
     */
    public static 
    class TermList extends GprNodeList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "TermList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNodeList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final TermList NONE =
            new TermList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected TermList(
            final Entity entity
        ) {
            super(entity);
        }

        public static TermList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                TermList.NONE :

                new TermList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return TermList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return TermList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return TermList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return TermList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * List of Identifier.
     */
    public static 
    class IdentifierList extends BaseList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "IdentifierList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            BaseList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final IdentifierList NONE =
            new IdentifierList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected IdentifierList(
            final Entity entity
        ) {
            super(entity);
        }

        public static IdentifierList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                IdentifierList.NONE :

                new IdentifierList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return IdentifierList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return IdentifierList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return IdentifierList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return IdentifierList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * List of StringLiteral.
     */
    public static 
    class StringLiteralList extends BaseList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "StringLiteralList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            BaseList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final StringLiteralList NONE =
            new StringLiteralList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected StringLiteralList(
            final Entity entity
        ) {
            super(entity);
        }

        public static StringLiteralList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                StringLiteralList.NONE :

                new StringLiteralList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return StringLiteralList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return StringLiteralList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return StringLiteralList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return StringLiteralList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * List of TermList.
     */
    public static 
    class TermListList extends BaseList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "TermListList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            BaseList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final TermListList NONE =
            new TermListList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected TermListList(
            final Entity entity
        ) {
            super(entity);
        }

        public static TermListList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                TermListList.NONE :

                new TermListList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return TermListList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return TermListList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return TermListList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return TermListList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**
     * List of WithDecl.
     */
    public static 
    class WithDeclList extends BaseList {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "WithDeclList";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            true;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            BaseList.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final WithDeclList NONE =
            new WithDeclList(
                Entity.NONE
            );

        // ----- Constructors -----

        protected WithDeclList(
            final Entity entity
        ) {
            super(entity);
        }

        public static WithDeclList fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                WithDeclList.NONE :

                new WithDeclList(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return WithDeclList.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return WithDeclList.fieldNames;
        }

        @Override
        public boolean isListType() {
            return WithDeclList.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return WithDeclList.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class BuiltinFunctionCall extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "BuiltinFunctionCall";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fFunctionName","fParameters"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = BuiltinFunctionCall.class.getMethod(
                        "fFunctionName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_function_name",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = BuiltinFunctionCall.class.getMethod(
                        "fParameters",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_parameters",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final BuiltinFunctionCall NONE =
            new BuiltinFunctionCall(
                Entity.NONE
            );

        // ----- Constructors -----

        protected BuiltinFunctionCall(
            final Entity entity
        ) {
            super(entity);
        }

        public static BuiltinFunctionCall fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                BuiltinFunctionCall.NONE :

                new BuiltinFunctionCall(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return BuiltinFunctionCall.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return BuiltinFunctionCall.fieldNames;
        }

        @Override
        public boolean isListType() {
            return BuiltinFunctionCall.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return BuiltinFunctionCall.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public Identifier fFunctionName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_builtin_function_call_f_function_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Identifier res = Identifier.NONE;
                if(propException == null) {
                    res = Identifier.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_builtin_function_call_f_function_name(
                    this.entity
                );

                // Wrap and return the result
                return Identifier.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public Terms fParameters(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_builtin_function_call_f_parameters(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Terms res = Terms.NONE;
                if(propException == null) {
                    res = Terms.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_builtin_function_call_f_parameters(
                    this.entity
                );

                // Wrap and return the result
                return Terms.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class CaseConstruction extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "CaseConstruction";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fVarRef","fItems"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = CaseConstruction.class.getMethod(
                        "fVarRef",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_var_ref",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = CaseConstruction.class.getMethod(
                        "fItems",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_items",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final CaseConstruction NONE =
            new CaseConstruction(
                Entity.NONE
            );

        // ----- Constructors -----

        protected CaseConstruction(
            final Entity entity
        ) {
            super(entity);
        }

        public static CaseConstruction fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                CaseConstruction.NONE :

                new CaseConstruction(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return CaseConstruction.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return CaseConstruction.fieldNames;
        }

        @Override
        public boolean isListType() {
            return CaseConstruction.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return CaseConstruction.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public VariableReference fVarRef(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_case_construction_f_var_ref(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                VariableReference res = VariableReference.NONE;
                if(propException == null) {
                    res = VariableReference.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_case_construction_f_var_ref(
                    this.entity
                );

                // Wrap and return the result
                return VariableReference.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public CaseItemList fItems(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_case_construction_f_items(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                CaseItemList res = CaseItemList.NONE;
                if(propException == null) {
                    res = CaseItemList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_case_construction_f_items(
                    this.entity
                );

                // Wrap and return the result
                return CaseItemList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class CaseItem extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "CaseItem";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fChoice","fDecls"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = CaseItem.class.getMethod(
                        "fChoice",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_choice",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = CaseItem.class.getMethod(
                        "fDecls",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_decls",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final CaseItem NONE =
            new CaseItem(
                Entity.NONE
            );

        // ----- Constructors -----

        protected CaseItem(
            final Entity entity
        ) {
            super(entity);
        }

        public static CaseItem fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                CaseItem.NONE :

                new CaseItem(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return CaseItem.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return CaseItem.fieldNames;
        }

        @Override
        public boolean isListType() {
            return CaseItem.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return CaseItem.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field contains a list that itself contains one of the following
         * nodes: ``OthersDesignator``, ``StringLiteral``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Choices fChoice(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_case_item_f_choice(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Choices res = Choices.NONE;
                if(propException == null) {
                    res = Choices.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_case_item_f_choice(
                    this.entity
                );

                // Wrap and return the result
                return Choices.fromEntity(res);
            }

        }

        
    

        /**
         * This field contains a list that itself contains one of the following
         * nodes: ``AttributeDecl``, ``CaseConstruction``, ``EmptyDecl``,
         * ``VariableDecl``
         *
         * When there are no parsing errors, this field is never null.
         */
        public GprNodeList fDecls(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_case_item_f_decls(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GprNodeList res = GprNodeList.NONE;
                if(propException == null) {
                    res = GprNodeList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_case_item_f_decls(
                    this.entity
                );

                // Wrap and return the result
                return GprNodeList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class CompilationUnit extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "CompilationUnit";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fProject"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = CompilationUnit.class.getMethod(
                        "fProject",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_project",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final CompilationUnit NONE =
            new CompilationUnit(
                Entity.NONE
            );

        // ----- Constructors -----

        protected CompilationUnit(
            final Entity entity
        ) {
            super(entity);
        }

        public static CompilationUnit fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                CompilationUnit.NONE :

                new CompilationUnit(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return CompilationUnit.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return CompilationUnit.fieldNames;
        }

        @Override
        public boolean isListType() {
            return CompilationUnit.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return CompilationUnit.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public Project fProject(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_compilation_unit_f_project(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Project res = Project.NONE;
                if(propException == null) {
                    res = Project.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_compilation_unit_f_project(
                    this.entity
                );

                // Wrap and return the result
                return Project.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class EmptyDecl extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "EmptyDecl";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final EmptyDecl NONE =
            new EmptyDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected EmptyDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static EmptyDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                EmptyDecl.NONE :

                new EmptyDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return EmptyDecl.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return EmptyDecl.fieldNames;
        }

        @Override
        public boolean isListType() {
            return EmptyDecl.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return EmptyDecl.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static abstract
    class Expr extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "Expr";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final Expr NONE =
            new ExprNone();

        // ----- Constructors -----

        protected Expr(
            final Entity entity
        ) {
            super(entity);
        }

        public static Expr fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                Expr.NONE :

                (Expr) GprNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return Expr.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return Expr.fieldNames;
        }

        @Override
        public boolean isListType() {
            return Expr.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return Expr.fieldDescriptions.getOrDefault(name, null);
        }


        // ----- Field accessors -----


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * Expr.
         */
        private static final class ExprNone extends Expr {
            ExprNone() {super(Entity.NONE);}
        }

    }


    
    

    /**

     */
    public static 
    class Prefix extends Expr {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "Prefix";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fPrefix","fSuffix"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            Expr.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = Prefix.class.getMethod(
                        "fPrefix",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_prefix",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Prefix.class.getMethod(
                        "fSuffix",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_suffix",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final Prefix NONE =
            new Prefix(
                Entity.NONE
            );

        // ----- Constructors -----

        protected Prefix(
            final Entity entity
        ) {
            super(entity);
        }

        public static Prefix fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                Prefix.NONE :

                new Prefix(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return Prefix.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return Prefix.fieldNames;
        }

        @Override
        public boolean isListType() {
            return Prefix.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return Prefix.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field can contain one of the following nodes: ``Identifier``,
         * ``Prefix``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fPrefix(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_prefix_f_prefix(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_prefix_f_prefix(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public Identifier fSuffix(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_prefix_f_suffix(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Identifier res = Identifier.NONE;
                if(propException == null) {
                    res = Identifier.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_prefix_f_suffix(
                    this.entity
                );

                // Wrap and return the result
                return Identifier.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static abstract
    class SingleTokNode extends Expr {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "SingleTokNode";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            Expr.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final SingleTokNode NONE =
            new SingleTokNodeNone();

        // ----- Constructors -----

        protected SingleTokNode(
            final Entity entity
        ) {
            super(entity);
        }

        public static SingleTokNode fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                SingleTokNode.NONE :

                (SingleTokNode) GprNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return SingleTokNode.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return SingleTokNode.fieldNames;
        }

        @Override
        public boolean isListType() {
            return SingleTokNode.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return SingleTokNode.fieldDescriptions.getOrDefault(name, null);
        }


        // ----- Field accessors -----


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * SingleTokNode.
         */
        private static final class SingleTokNodeNone extends SingleTokNode {
            SingleTokNodeNone() {super(Entity.NONE);}
        }

    }


    
    

    /**

     */
    public static 
    class Identifier extends SingleTokNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "Identifier";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            SingleTokNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final Identifier NONE =
            new Identifier(
                Entity.NONE
            );

        // ----- Constructors -----

        protected Identifier(
            final Entity entity
        ) {
            super(entity);
        }

        public static Identifier fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                Identifier.NONE :

                new Identifier(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return Identifier.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return Identifier.fieldNames;
        }

        @Override
        public boolean isListType() {
            return Identifier.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return Identifier.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class NumLiteral extends SingleTokNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "NumLiteral";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            SingleTokNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final NumLiteral NONE =
            new NumLiteral(
                Entity.NONE
            );

        // ----- Constructors -----

        protected NumLiteral(
            final Entity entity
        ) {
            super(entity);
        }

        public static NumLiteral fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                NumLiteral.NONE :

                new NumLiteral(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return NumLiteral.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return NumLiteral.fieldNames;
        }

        @Override
        public boolean isListType() {
            return NumLiteral.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return NumLiteral.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class StringLiteral extends SingleTokNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "StringLiteral";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            SingleTokNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final StringLiteral NONE =
            new StringLiteral(
                Entity.NONE
            );

        // ----- Constructors -----

        protected StringLiteral(
            final Entity entity
        ) {
            super(entity);
        }

        public static StringLiteral fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                StringLiteral.NONE :

                new StringLiteral(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return StringLiteral.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return StringLiteral.fieldNames;
        }

        @Override
        public boolean isListType() {
            return StringLiteral.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return StringLiteral.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static abstract
    class LimitedNode extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "LimitedNode";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = LimitedNode.class.getMethod(
                        "pAsBool",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "p_as_bool",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LimitedNode NONE =
            new LimitedNodeNone();

        // ----- Constructors -----

        protected LimitedNode(
            final Entity entity
        ) {
            super(entity);
        }

        public static LimitedNode fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LimitedNode.NONE :

                (LimitedNode) GprNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return LimitedNode.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return LimitedNode.fieldNames;
        }

        @Override
        public boolean isListType() {
            return LimitedNode.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return LimitedNode.fieldDescriptions.getOrDefault(name, null);
        }


        // ----- Field accessors -----

        
    

        /**
         * Return whether this is an instance of LimitedPresent
         */
        public boolean pAsBool(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final CCharPointer resNative =
                    StackValue.get(SizeOf.get(VoidPointer.class));

                // Call the native function
                NI_LIB.gpr_limited_node_p_as_bool(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                boolean res = false;
                if(propException == null) {
                    res = (resNative.read() != 0);

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final boolean res = JNI_LIB.gpr_limited_node_p_as_bool(
                    this.entity
                );

                // Wrap and return the result
                return res;
            }

        }


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * LimitedNode.
         */
        private static final class LimitedNodeNone extends LimitedNode {
            LimitedNodeNone() {super(Entity.NONE);}
        }

    }


    
    

    /**

     */
    public static 
    class LimitedAbsent extends LimitedNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "LimitedAbsent";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            LimitedNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LimitedAbsent NONE =
            new LimitedAbsent(
                Entity.NONE
            );

        // ----- Constructors -----

        protected LimitedAbsent(
            final Entity entity
        ) {
            super(entity);
        }

        public static LimitedAbsent fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LimitedAbsent.NONE :

                new LimitedAbsent(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return LimitedAbsent.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return LimitedAbsent.fieldNames;
        }

        @Override
        public boolean isListType() {
            return LimitedAbsent.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return LimitedAbsent.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class LimitedPresent extends LimitedNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "LimitedPresent";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            LimitedNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final LimitedPresent NONE =
            new LimitedPresent(
                Entity.NONE
            );

        // ----- Constructors -----

        protected LimitedPresent(
            final Entity entity
        ) {
            super(entity);
        }

        public static LimitedPresent fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                LimitedPresent.NONE :

                new LimitedPresent(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return LimitedPresent.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return LimitedPresent.fieldNames;
        }

        @Override
        public boolean isListType() {
            return LimitedPresent.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return LimitedPresent.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class OthersDesignator extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "OthersDesignator";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final OthersDesignator NONE =
            new OthersDesignator(
                Entity.NONE
            );

        // ----- Constructors -----

        protected OthersDesignator(
            final Entity entity
        ) {
            super(entity);
        }

        public static OthersDesignator fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                OthersDesignator.NONE :

                new OthersDesignator(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return OthersDesignator.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return OthersDesignator.fieldNames;
        }

        @Override
        public boolean isListType() {
            return OthersDesignator.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return OthersDesignator.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class PackageDecl extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "PackageDecl";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fPkgName","fPkgSpec"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = PackageDecl.class.getMethod(
                        "fPkgName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_pkg_name",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = PackageDecl.class.getMethod(
                        "fPkgSpec",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_pkg_spec",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final PackageDecl NONE =
            new PackageDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected PackageDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static PackageDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                PackageDecl.NONE :

                new PackageDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return PackageDecl.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return PackageDecl.fieldNames;
        }

        @Override
        public boolean isListType() {
            return PackageDecl.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return PackageDecl.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public Identifier fPkgName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_package_decl_f_pkg_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Identifier res = Identifier.NONE;
                if(propException == null) {
                    res = Identifier.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_package_decl_f_pkg_name(
                    this.entity
                );

                // Wrap and return the result
                return Identifier.fromEntity(res);
            }

        }

        
    

        /**
         * This field can contain one of the following nodes:
         * ``PackageRenaming``, ``PackageSpec``
         *
         * When there are no parsing errors, this field is never null.
         */
        public GprNode fPkgSpec(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_package_decl_f_pkg_spec(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GprNode res = GprNode.NONE;
                if(propException == null) {
                    res = GprNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_package_decl_f_pkg_spec(
                    this.entity
                );

                // Wrap and return the result
                return GprNode.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class PackageExtension extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "PackageExtension";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fExtendedName"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = PackageExtension.class.getMethod(
                        "fExtendedName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_extended_name",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final PackageExtension NONE =
            new PackageExtension(
                Entity.NONE
            );

        // ----- Constructors -----

        protected PackageExtension(
            final Entity entity
        ) {
            super(entity);
        }

        public static PackageExtension fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                PackageExtension.NONE :

                new PackageExtension(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return PackageExtension.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return PackageExtension.fieldNames;
        }

        @Override
        public boolean isListType() {
            return PackageExtension.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return PackageExtension.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public IdentifierList fExtendedName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_package_extension_f_extended_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                IdentifierList res = IdentifierList.NONE;
                if(propException == null) {
                    res = IdentifierList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_package_extension_f_extended_name(
                    this.entity
                );

                // Wrap and return the result
                return IdentifierList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class PackageRenaming extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "PackageRenaming";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fRenamedName"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = PackageRenaming.class.getMethod(
                        "fRenamedName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_renamed_name",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final PackageRenaming NONE =
            new PackageRenaming(
                Entity.NONE
            );

        // ----- Constructors -----

        protected PackageRenaming(
            final Entity entity
        ) {
            super(entity);
        }

        public static PackageRenaming fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                PackageRenaming.NONE :

                new PackageRenaming(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return PackageRenaming.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return PackageRenaming.fieldNames;
        }

        @Override
        public boolean isListType() {
            return PackageRenaming.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return PackageRenaming.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public IdentifierList fRenamedName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_package_renaming_f_renamed_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                IdentifierList res = IdentifierList.NONE;
                if(propException == null) {
                    res = IdentifierList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_package_renaming_f_renamed_name(
                    this.entity
                );

                // Wrap and return the result
                return IdentifierList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class PackageSpec extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "PackageSpec";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fExtension","fDecls","fEndName"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = PackageSpec.class.getMethod(
                        "fExtension",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_extension",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = PackageSpec.class.getMethod(
                        "fDecls",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_decls",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = PackageSpec.class.getMethod(
                        "fEndName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_end_name",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final PackageSpec NONE =
            new PackageSpec(
                Entity.NONE
            );

        // ----- Constructors -----

        protected PackageSpec(
            final Entity entity
        ) {
            super(entity);
        }

        public static PackageSpec fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                PackageSpec.NONE :

                new PackageSpec(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return PackageSpec.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return PackageSpec.fieldNames;
        }

        @Override
        public boolean isListType() {
            return PackageSpec.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return PackageSpec.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field may be null even when there are no parsing errors.
         */
        public PackageExtension fExtension(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_package_spec_f_extension(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                PackageExtension res = PackageExtension.NONE;
                if(propException == null) {
                    res = PackageExtension.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_package_spec_f_extension(
                    this.entity
                );

                // Wrap and return the result
                return PackageExtension.fromEntity(res);
            }

        }

        
    

        /**
         * This field contains a list that itself contains one of the following
         * nodes: ``AttributeDecl``, ``CaseConstruction``, ``EmptyDecl``,
         * ``VariableDecl``
         *
         * When there are no parsing errors, this field is never null.
         */
        public GprNodeList fDecls(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_package_spec_f_decls(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GprNodeList res = GprNodeList.NONE;
                if(propException == null) {
                    res = GprNodeList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_package_spec_f_decls(
                    this.entity
                );

                // Wrap and return the result
                return GprNodeList.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public Identifier fEndName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_package_spec_f_end_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Identifier res = Identifier.NONE;
                if(propException == null) {
                    res = Identifier.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_package_spec_f_end_name(
                    this.entity
                );

                // Wrap and return the result
                return Identifier.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class Project extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "Project";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fContextClauses","fProjectDecl"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = Project.class.getMethod(
                        "fContextClauses",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_context_clauses",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = Project.class.getMethod(
                        "fProjectDecl",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_project_decl",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final Project NONE =
            new Project(
                Entity.NONE
            );

        // ----- Constructors -----

        protected Project(
            final Entity entity
        ) {
            super(entity);
        }

        public static Project fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                Project.NONE :

                new Project(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return Project.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return Project.fieldNames;
        }

        @Override
        public boolean isListType() {
            return Project.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return Project.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public WithDeclList fContextClauses(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_project_f_context_clauses(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                WithDeclList res = WithDeclList.NONE;
                if(propException == null) {
                    res = WithDeclList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_project_f_context_clauses(
                    this.entity
                );

                // Wrap and return the result
                return WithDeclList.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public ProjectDeclaration fProjectDecl(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_project_f_project_decl(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                ProjectDeclaration res = ProjectDeclaration.NONE;
                if(propException == null) {
                    res = ProjectDeclaration.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_project_f_project_decl(
                    this.entity
                );

                // Wrap and return the result
                return ProjectDeclaration.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class ProjectDeclaration extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "ProjectDeclaration";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fQualifier","fProjectName","fExtension","fDecls","fEndName"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = ProjectDeclaration.class.getMethod(
                        "fQualifier",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_qualifier",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = ProjectDeclaration.class.getMethod(
                        "fProjectName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_project_name",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = ProjectDeclaration.class.getMethod(
                        "fExtension",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_extension",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = ProjectDeclaration.class.getMethod(
                        "fDecls",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_decls",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = ProjectDeclaration.class.getMethod(
                        "fEndName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_end_name",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ProjectDeclaration NONE =
            new ProjectDeclaration(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ProjectDeclaration(
            final Entity entity
        ) {
            super(entity);
        }

        public static ProjectDeclaration fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ProjectDeclaration.NONE :

                new ProjectDeclaration(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return ProjectDeclaration.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return ProjectDeclaration.fieldNames;
        }

        @Override
        public boolean isListType() {
            return ProjectDeclaration.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return ProjectDeclaration.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * This field may be null even when there are no parsing errors.
         */
        public ProjectQualifier fQualifier(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_project_declaration_f_qualifier(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                ProjectQualifier res = ProjectQualifier.NONE;
                if(propException == null) {
                    res = ProjectQualifier.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_project_declaration_f_qualifier(
                    this.entity
                );

                // Wrap and return the result
                return ProjectQualifier.fromEntity(res);
            }

        }

        
    

        /**
         * This field can contain one of the following nodes: ``Identifier``,
         * ``Prefix``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fProjectName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_project_declaration_f_project_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_project_declaration_f_project_name(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }

        
    

        /**
         * This field may be null even when there are no parsing errors.
         */
        public ProjectExtension fExtension(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_project_declaration_f_extension(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                ProjectExtension res = ProjectExtension.NONE;
                if(propException == null) {
                    res = ProjectExtension.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_project_declaration_f_extension(
                    this.entity
                );

                // Wrap and return the result
                return ProjectExtension.fromEntity(res);
            }

        }

        
    

        /**
         * This field contains a list that itself contains one of the following
         * nodes: ``AttributeDecl``, ``CaseConstruction``, ``EmptyDecl``,
         * ``PackageDecl``, ``TypedStringDecl``, ``VariableDecl``
         *
         * When there are no parsing errors, this field is never null.
         */
        public GprNodeList fDecls(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_project_declaration_f_decls(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                GprNodeList res = GprNodeList.NONE;
                if(propException == null) {
                    res = GprNodeList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_project_declaration_f_decls(
                    this.entity
                );

                // Wrap and return the result
                return GprNodeList.fromEntity(res);
            }

        }

        
    

        /**
         * This field can contain one of the following nodes: ``Identifier``,
         * ``Prefix``
         *
         * When there are no parsing errors, this field is never null.
         */
        public Expr fEndName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_project_declaration_f_end_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Expr res = Expr.NONE;
                if(propException == null) {
                    res = Expr.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_project_declaration_f_end_name(
                    this.entity
                );

                // Wrap and return the result
                return Expr.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class ProjectExtension extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "ProjectExtension";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fIsAll","fPathName"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = ProjectExtension.class.getMethod(
                        "fIsAll",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_is_all",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = ProjectExtension.class.getMethod(
                        "fPathName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_path_name",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ProjectExtension NONE =
            new ProjectExtension(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ProjectExtension(
            final Entity entity
        ) {
            super(entity);
        }

        public static ProjectExtension fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ProjectExtension.NONE :

                new ProjectExtension(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return ProjectExtension.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return ProjectExtension.fieldNames;
        }

        @Override
        public boolean isListType() {
            return ProjectExtension.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return ProjectExtension.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public AllQualifier fIsAll(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_project_extension_f_is_all(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AllQualifier res = AllQualifier.NONE;
                if(propException == null) {
                    res = AllQualifier.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_project_extension_f_is_all(
                    this.entity
                );

                // Wrap and return the result
                return AllQualifier.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public StringLiteral fPathName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_project_extension_f_path_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                StringLiteral res = StringLiteral.NONE;
                if(propException == null) {
                    res = StringLiteral.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_project_extension_f_path_name(
                    this.entity
                );

                // Wrap and return the result
                return StringLiteral.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static abstract
    class ProjectQualifier extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "ProjectQualifier";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ProjectQualifier NONE =
            new ProjectQualifierNone();

        // ----- Constructors -----

        protected ProjectQualifier(
            final Entity entity
        ) {
            super(entity);
        }

        public static ProjectQualifier fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ProjectQualifier.NONE :

                (ProjectQualifier) GprNode.dispatchNodeCreation(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return ProjectQualifier.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return ProjectQualifier.fieldNames;
        }

        @Override
        public boolean isListType() {
            return ProjectQualifier.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return ProjectQualifier.fieldDescriptions.getOrDefault(name, null);
        }


        // ----- Field accessors -----


        // ----- Inner classes -----

        /**
         * This class represents the none value of the abstract node
         * ProjectQualifier.
         */
        private static final class ProjectQualifierNone extends ProjectQualifier {
            ProjectQualifierNone() {super(Entity.NONE);}
        }

    }


    
    

    /**

     */
    public static 
    class ProjectQualifierAbstract extends ProjectQualifier {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "ProjectQualifierAbstract";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            ProjectQualifier.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ProjectQualifierAbstract NONE =
            new ProjectQualifierAbstract(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ProjectQualifierAbstract(
            final Entity entity
        ) {
            super(entity);
        }

        public static ProjectQualifierAbstract fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ProjectQualifierAbstract.NONE :

                new ProjectQualifierAbstract(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return ProjectQualifierAbstract.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return ProjectQualifierAbstract.fieldNames;
        }

        @Override
        public boolean isListType() {
            return ProjectQualifierAbstract.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return ProjectQualifierAbstract.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class ProjectQualifierAggregate extends ProjectQualifier {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "ProjectQualifierAggregate";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            ProjectQualifier.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ProjectQualifierAggregate NONE =
            new ProjectQualifierAggregate(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ProjectQualifierAggregate(
            final Entity entity
        ) {
            super(entity);
        }

        public static ProjectQualifierAggregate fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ProjectQualifierAggregate.NONE :

                new ProjectQualifierAggregate(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return ProjectQualifierAggregate.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return ProjectQualifierAggregate.fieldNames;
        }

        @Override
        public boolean isListType() {
            return ProjectQualifierAggregate.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return ProjectQualifierAggregate.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class ProjectQualifierAggregateLibrary extends ProjectQualifier {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "ProjectQualifierAggregateLibrary";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            ProjectQualifier.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ProjectQualifierAggregateLibrary NONE =
            new ProjectQualifierAggregateLibrary(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ProjectQualifierAggregateLibrary(
            final Entity entity
        ) {
            super(entity);
        }

        public static ProjectQualifierAggregateLibrary fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ProjectQualifierAggregateLibrary.NONE :

                new ProjectQualifierAggregateLibrary(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return ProjectQualifierAggregateLibrary.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return ProjectQualifierAggregateLibrary.fieldNames;
        }

        @Override
        public boolean isListType() {
            return ProjectQualifierAggregateLibrary.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return ProjectQualifierAggregateLibrary.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class ProjectQualifierConfiguration extends ProjectQualifier {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "ProjectQualifierConfiguration";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            ProjectQualifier.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ProjectQualifierConfiguration NONE =
            new ProjectQualifierConfiguration(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ProjectQualifierConfiguration(
            final Entity entity
        ) {
            super(entity);
        }

        public static ProjectQualifierConfiguration fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ProjectQualifierConfiguration.NONE :

                new ProjectQualifierConfiguration(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return ProjectQualifierConfiguration.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return ProjectQualifierConfiguration.fieldNames;
        }

        @Override
        public boolean isListType() {
            return ProjectQualifierConfiguration.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return ProjectQualifierConfiguration.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class ProjectQualifierLibrary extends ProjectQualifier {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "ProjectQualifierLibrary";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            ProjectQualifier.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ProjectQualifierLibrary NONE =
            new ProjectQualifierLibrary(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ProjectQualifierLibrary(
            final Entity entity
        ) {
            super(entity);
        }

        public static ProjectQualifierLibrary fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ProjectQualifierLibrary.NONE :

                new ProjectQualifierLibrary(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return ProjectQualifierLibrary.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return ProjectQualifierLibrary.fieldNames;
        }

        @Override
        public boolean isListType() {
            return ProjectQualifierLibrary.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return ProjectQualifierLibrary.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class ProjectQualifierStandard extends ProjectQualifier {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "ProjectQualifierStandard";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            ProjectQualifier.fieldDescriptions
        );

        // Initialisation of the method map
        static {
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final ProjectQualifierStandard NONE =
            new ProjectQualifierStandard(
                Entity.NONE
            );

        // ----- Constructors -----

        protected ProjectQualifierStandard(
            final Entity entity
        ) {
            super(entity);
        }

        public static ProjectQualifierStandard fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                ProjectQualifierStandard.NONE :

                new ProjectQualifierStandard(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return ProjectQualifierStandard.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return ProjectQualifierStandard.fieldNames;
        }

        @Override
        public boolean isListType() {
            return ProjectQualifierStandard.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return ProjectQualifierStandard.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class StringLiteralAt extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "StringLiteralAt";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fStrLit","fAtLit"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = StringLiteralAt.class.getMethod(
                        "fStrLit",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_str_lit",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = StringLiteralAt.class.getMethod(
                        "fAtLit",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_at_lit",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final StringLiteralAt NONE =
            new StringLiteralAt(
                Entity.NONE
            );

        // ----- Constructors -----

        protected StringLiteralAt(
            final Entity entity
        ) {
            super(entity);
        }

        public static StringLiteralAt fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                StringLiteralAt.NONE :

                new StringLiteralAt(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return StringLiteralAt.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return StringLiteralAt.fieldNames;
        }

        @Override
        public boolean isListType() {
            return StringLiteralAt.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return StringLiteralAt.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public StringLiteral fStrLit(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_string_literal_at_f_str_lit(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                StringLiteral res = StringLiteral.NONE;
                if(propException == null) {
                    res = StringLiteral.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_string_literal_at_f_str_lit(
                    this.entity
                );

                // Wrap and return the result
                return StringLiteral.fromEntity(res);
            }

        }

        
    

        /**
         * This field may be null even when there are no parsing errors.
         */
        public NumLiteral fAtLit(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_string_literal_at_f_at_lit(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                NumLiteral res = NumLiteral.NONE;
                if(propException == null) {
                    res = NumLiteral.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_string_literal_at_f_at_lit(
                    this.entity
                );

                // Wrap and return the result
                return NumLiteral.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class Terms extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "Terms";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fTerms"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = Terms.class.getMethod(
                        "fTerms",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_terms",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final Terms NONE =
            new Terms(
                Entity.NONE
            );

        // ----- Constructors -----

        protected Terms(
            final Entity entity
        ) {
            super(entity);
        }

        public static Terms fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                Terms.NONE :

                new Terms(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return Terms.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return Terms.fieldNames;
        }

        @Override
        public boolean isListType() {
            return Terms.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return Terms.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public TermListList fTerms(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_terms_f_terms(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TermListList res = TermListList.NONE;
                if(propException == null) {
                    res = TermListList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_terms_f_terms(
                    this.entity
                );

                // Wrap and return the result
                return TermListList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class TypeReference extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "TypeReference";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fVarTypeName"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = TypeReference.class.getMethod(
                        "fVarTypeName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_var_type_name",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final TypeReference NONE =
            new TypeReference(
                Entity.NONE
            );

        // ----- Constructors -----

        protected TypeReference(
            final Entity entity
        ) {
            super(entity);
        }

        public static TypeReference fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                TypeReference.NONE :

                new TypeReference(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return TypeReference.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return TypeReference.fieldNames;
        }

        @Override
        public boolean isListType() {
            return TypeReference.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return TypeReference.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public IdentifierList fVarTypeName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_type_reference_f_var_type_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                IdentifierList res = IdentifierList.NONE;
                if(propException == null) {
                    res = IdentifierList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_type_reference_f_var_type_name(
                    this.entity
                );

                // Wrap and return the result
                return IdentifierList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class TypedStringDecl extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "TypedStringDecl";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fTypeId","fStringLiterals"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = TypedStringDecl.class.getMethod(
                        "fTypeId",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_type_id",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = TypedStringDecl.class.getMethod(
                        "fStringLiterals",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_string_literals",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final TypedStringDecl NONE =
            new TypedStringDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected TypedStringDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static TypedStringDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                TypedStringDecl.NONE :

                new TypedStringDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return TypedStringDecl.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return TypedStringDecl.fieldNames;
        }

        @Override
        public boolean isListType() {
            return TypedStringDecl.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return TypedStringDecl.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public Identifier fTypeId(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_typed_string_decl_f_type_id(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Identifier res = Identifier.NONE;
                if(propException == null) {
                    res = Identifier.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_typed_string_decl_f_type_id(
                    this.entity
                );

                // Wrap and return the result
                return Identifier.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public StringLiteralList fStringLiterals(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_typed_string_decl_f_string_literals(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                StringLiteralList res = StringLiteralList.NONE;
                if(propException == null) {
                    res = StringLiteralList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_typed_string_decl_f_string_literals(
                    this.entity
                );

                // Wrap and return the result
                return StringLiteralList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class VariableDecl extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "VariableDecl";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fVarName","fVarType","fExpr"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = VariableDecl.class.getMethod(
                        "fVarName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_var_name",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = VariableDecl.class.getMethod(
                        "fVarType",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_var_type",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = VariableDecl.class.getMethod(
                        "fExpr",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_expr",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final VariableDecl NONE =
            new VariableDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected VariableDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static VariableDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                VariableDecl.NONE :

                new VariableDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return VariableDecl.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return VariableDecl.fieldNames;
        }

        @Override
        public boolean isListType() {
            return VariableDecl.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return VariableDecl.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public Identifier fVarName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_variable_decl_f_var_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                Identifier res = Identifier.NONE;
                if(propException == null) {
                    res = Identifier.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_variable_decl_f_var_name(
                    this.entity
                );

                // Wrap and return the result
                return Identifier.fromEntity(res);
            }

        }

        
    

        /**
         * This field may be null even when there are no parsing errors.
         */
        public TypeReference fVarType(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_variable_decl_f_var_type(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TypeReference res = TypeReference.NONE;
                if(propException == null) {
                    res = TypeReference.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_variable_decl_f_var_type(
                    this.entity
                );

                // Wrap and return the result
                return TypeReference.fromEntity(res);
            }

        }

        
    

        /**
         * This field contains a list that itself contains one of the following
         * nodes: ``BuiltinFunctionCall``, ``StringLiteralAt``, ``Terms``,
         * ``VariableReference``
         *
         * When there are no parsing errors, this field is never null.
         */
        public TermList fExpr(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_variable_decl_f_expr(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                TermList res = TermList.NONE;
                if(propException == null) {
                    res = TermList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_variable_decl_f_expr(
                    this.entity
                );

                // Wrap and return the result
                return TermList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class VariableReference extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "VariableReference";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fVariableName","fAttributeRef"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = VariableReference.class.getMethod(
                        "fVariableName",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_variable_name",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = VariableReference.class.getMethod(
                        "fAttributeRef",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_attribute_ref",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final VariableReference NONE =
            new VariableReference(
                Entity.NONE
            );

        // ----- Constructors -----

        protected VariableReference(
            final Entity entity
        ) {
            super(entity);
        }

        public static VariableReference fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                VariableReference.NONE :

                new VariableReference(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return VariableReference.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return VariableReference.fieldNames;
        }

        @Override
        public boolean isListType() {
            return VariableReference.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return VariableReference.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public IdentifierList fVariableName(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_variable_reference_f_variable_name(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                IdentifierList res = IdentifierList.NONE;
                if(propException == null) {
                    res = IdentifierList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_variable_reference_f_variable_name(
                    this.entity
                );

                // Wrap and return the result
                return IdentifierList.fromEntity(res);
            }

        }

        
    

        /**
         * This field may be null even when there are no parsing errors.
         */
        public AttributeReference fAttributeRef(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_variable_reference_f_attribute_ref(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                AttributeReference res = AttributeReference.NONE;
                if(propException == null) {
                    res = AttributeReference.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_variable_reference_f_attribute_ref(
                    this.entity
                );

                // Wrap and return the result
                return AttributeReference.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }


    
    

    /**

     */
    public static 
    class WithDecl extends GprNode {

        // ----- Static -----

        
    

        /** The name of the node kind */
        public static final String kindName = "WithDecl";

        /** The names of the fields associated to the node */
        public static final String[] fieldNames = {
            "fIsLimited","fPathNames"
        };

        /** If the node is a list node */
        public static final boolean isListType =
            false;

        /** The map containing the node's fields description. */
        public static final Map<String, GprParserField>
        fieldDescriptions = new HashMap<>(
            GprNode.fieldDescriptions
        );

        // Initialisation of the method map
        static {
            try {
                
                {
                    // Get the Java method of the field
                    Method method = WithDecl.class.getMethod(
                        "fIsLimited",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_is_limited",
                        new GprParserField(method, parameters)
                    );
                }
                
                {
                    // Get the Java method of the field
                    Method method = WithDecl.class.getMethod(
                        "fPathNames",
                        new Class[]{}
                    );

                    // Create the parameter list
                    List<Param> parameters = new ArrayList<>();

                    // Add the method and the parameters in maps
                    fieldDescriptions.put(
                        "f_path_names",
                        new GprParserField(method, parameters)
                    );
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }


        // ----- Attributes -----

        /** Singleton that represents the none node. */
        public static final WithDecl NONE =
            new WithDecl(
                Entity.NONE
            );

        // ----- Constructors -----

        protected WithDecl(
            final Entity entity
        ) {
            super(entity);
        }

        public static WithDecl fromEntity(
            final Entity entity
        ) {
            return entity.node.isNull() ?
                WithDecl.NONE :

                new WithDecl(entity);
        }

        // ----- Instance methods -----

        @Override
        public String getKindName() {
            return WithDecl.kindName;
        }

        @Override
        public String[] getFieldNames() {
            return WithDecl.fieldNames;
        }

        @Override
        public boolean isListType() {
            return WithDecl.isListType;
        }

        @Override
        @CompilerDirectives.TruffleBoundary
        public GprParserField getFieldDescription(
            final String name
        ) {
            return WithDecl.fieldDescriptions.getOrDefault(name, null);
        }

        // ----- Visitor methods -----

        public <T> T accept(BasicVisitor<T> visitor) {
            return visitor.visit(this);
        }

        public <T, P> T accept(ParamVisitor<T, P> visitor, P param) {
            return visitor.visit(this, param);
        }

        // ----- Field accessors -----

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public LimitedNode fIsLimited(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_with_decl_f_is_limited(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                LimitedNode res = LimitedNode.NONE;
                if(propException == null) {
                    res = LimitedNode.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_with_decl_f_is_limited(
                    this.entity
                );

                // Wrap and return the result
                return LimitedNode.fromEntity(res);
            }

        }

        
    

        /**
         * When there are no parsing errors, this field is never null.
         */
        public StringLiteralList fPathNames(
            
        ) {

            // Verify that arguments are not null

            if(ImageInfo.inImageCode()) {
                // Unwrap the current node
                final EntityNative thisNative = StackValue.get(
                    EntityNative.class
                );
                this.entity.unwrap(thisNative);



                // Unwrap the arguments

                // Create the result native
                final EntityNative resNative =
                    StackValue.get(EntityNative.class);

                // Call the native function
                NI_LIB.gpr_with_decl_f_path_names(
                    thisNative,
                    resNative
                );

                // Get the potential exception of the native property call
                final LangkitException propException = getLastException();

                // Wrap the result if the property returned normally
                StringLiteralList res = StringLiteralList.NONE;
                if(propException == null) {
                    res = StringLiteralList.fromEntity(Entity.wrap(resNative));

                }

                // Release the unwrapped parameters


                // If the property exception is not null throw it
                if(propException != null) {
                    throw propException;
                }

                // Return the result
                return res;
            } else {
                // Call the native function
                final Entity res = JNI_LIB.gpr_with_decl_f_path_names(
                    this.entity
                );

                // Wrap and return the result
                return StringLiteralList.fromEntity(res);
            }

        }


        // ----- Inner classes -----


    }



}
