/* Table of opcodes for the sparc.
   Copyright (C) 1989 Free Software Foundation, Inc.

This file is part of GAS, the GNU Assembler, and GDB, the GNU disassembler.

GAS/GDB is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GAS/GDB is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GAS or GDB; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#if !defined(__STDC__) && !defined(const)
#define const
#endif

/*
 * Structure of an opcode table entry.
 */
struct sparc_opcode
{
    const char *name;
    unsigned long int match;	/* Bits that must be set.  */
    unsigned long int lose;	/* Bits that must not be set.  */
    const char *args;
};

/*
   All sparc opcodes are 32 bits, except for the `set' instruction (really
   a macro), which is 64 bits.  It is handled as a special case.

   The match component is a mask saying which bits must match a
   particular opcode in order for an instruction to be an instance
   of that opcode.

   The args component is a string containing one character
   for each operand of the instruction.

Kinds of operands:
   #    Number used by optimizer.  It is ignored.
   1    rs1 register.
   2    rs2 register.
   d    rd register.
   e    frs1 floating point register.
   f    frs2 floating point register.
   g    frsd floating point register.
   b    crs1 coprocessor register
   c    crs2 coprocessor register
   D    crsd coprocessor register
   h    22 high bits.
   i    13 bit Immediate.
   l    22 bit PC relative immediate.
   L    30 bit PC relative immediate.
   a    Annul.  The annul bit is set.
   A    Alternate address space.  Stored as 8 bits.
   C    Coprocessor state register.
   F    floating point state register.
   p    Processor state register.
   q    Floating point queue.
   r    Single register that is both rs1 and rsd.
   Q    Coprocessor queue.
   S    Special case.
   t    Trap base register.
   w    Window invalid mask register.
   y    Y register.

*/

/* The order of the opcodes in this table is significant:

   * The assembler requires that all instances of the same mnemonic must be
   consecutive.  If they aren't, the assembler will bomb at runtime.

   * The disassembler should not care about the order of the opcodes.  */

static struct sparc_opcode sparc_opcodes[] =
{

{ "ldd",        0xc1980000, 0x0060001f, "[1],D" }, /* ldd [rs1+%g0],d */
{ "ldd",        0xc1982000, 0x00601fff, "[1],D" }, /* ldd [rs1+0],d */
{ "ldd",        0xc1982000, 0x00600000, "[1+i],D" },
{ "ldd",        0xc1982000, 0x00600000, "[i+1],D" },
{ "ldd",        0xc1980000, 0x00600000, "[1+2],D" },
{ "ldd",        0xc1180000, 0x00e0001f, "[1],g" }, /* ldd [rs1+%g0],d */
{ "ldd",        0xc1182000, 0x00e01fff, "[1],g" }, /* ldd [rs1+0],d */
{ "ldd",        0xc1182000, 0x00e00000, "[1+i],g" },
{ "ldd",        0xc1182000, 0x00e00000, "[i+1],g" },
{ "ldd",        0xc1180000, 0x00e00000, "[1+2],g" },
{ "ldd",	0xc0180000, 0x01e0001f, "[1],d" }, /* ldd [rs1+%g0],d */
{ "ldd",        0xc0182000, 0x01e01fff, "[1],d" }, /* ldd [rs1+0],d */
{ "ldd",	0xc0182000, 0x01e00000, "[1+i],d" },
{ "ldd",	0xc0182000, 0x01e00000, "[i+1],d" },
{ "ldd",	0xc0180000, 0x01e00000, "[1+2],d" },
{ "ld",         0xc1880000, 0x0070001f, "[1],C" }, /* ld [rs1+%g0],d */
{ "ld",         0xc1882000, 0x00701fff, "[1],C" }, /* ld [rs1+0],d */
{ "ld",         0xc1882000, 0x00700000, "[1+i],C" },
{ "ld",         0xc1882000, 0x00700000, "[i+1],C" },
{ "ld",         0xc1880000, 0x00700000, "[1+2],C" },
{ "ld",         0xc1800000, 0x0078001f, "[1],D" }, /* ld [rs1+%g0],d */
{ "ld",         0xc1802000, 0x00781fff, "[1],D" }, /* ld [rs1+0],d */
{ "ld",         0xc1802000, 0x00780000, "[1+i],D" },
{ "ld",         0xc1802000, 0x00780000, "[i+1],D" },
{ "ld",         0xc1800000, 0x00780000, "[1+2],D" },
{ "ld",         0xc1080000, 0x00f0001f, "[1],F" }, /* ld [rs1+%g0],d */
{ "ld",         0xc1082000, 0x00f01fff, "[1],F" }, /* ld [rs1+0],d */
{ "ld",         0xc1082000, 0x00f00000, "[1+i],F" },
{ "ld",         0xc1082000, 0x00f00000, "[i+1],F" },
{ "ld",         0xc1080000, 0x00f00000, "[1+2],F" },
{ "ld",         0xc1000000, 0x00f8001f, "[1],g" }, /* ld [rs1+%g0],d */
{ "ld",         0xc1002000, 0x00f81fff, "[1],g" }, /* ld [rs1+0],d */
{ "ld",         0xc1002000, 0x00f80000, "[1+i],g" },
{ "ld",         0xc1002000, 0x00f80000, "[i+1],g" },
{ "ld",         0xc1000000, 0x00f80000, "[1+2],g" },
{ "ld",	        0xc0000000, 0x01f8001f, "[1],d" }, /* ld [rs1+%g0],d */
{ "ld",		0xc0002000, 0x01f81fff, "[1],d" }, /* ld [rs1+0],d */
{ "ld",		0xc0002000, 0x01f80000, "[1+i],d" },
{ "ld",		0xc0002000, 0x01f80000, "[i+1],d" },
{ "ld",		0xc0000000, 0x01f80000, "[1+2],d" },
{ "ldstuba",    0xc0d80000, 0x0100001f, "[1]A,d" }, /* ldstuba [rs1+%g0],d */
{ "ldstuba",	0xc0d82000, 0x01001fff, "[1]A,d" }, /* ldstuba [rs1+0],d */
{ "ldstuba",	0xc0d80000, 0x01000000, "[1+2]A,d" },
{ "ldsha",      0xc0d00000, 0x0128001f, "[1]A,d" }, /* ldsha [rs1+%g0],d */
{ "ldsha",	0xc0d02000, 0x01281fff, "[1]A,d" }, /* ldsha [rs1+0],d */
{ "ldsha",	0xc0d00000, 0x01280000, "[1+2]A,d" },
{ "ldsba",      0xc0c80000, 0x0130001f, "[1]A,d" }, /* ldsba [rs1+%g0],d */
{ "ldsba",	0xc0c82000, 0x01301fff, "[1]A,d" }, /* ldsba [rs1+0],d */
{ "ldsba",	0xc0c80000, 0x01300000, "[1+2]A,d" },
{ "ldda",       0xc0980000, 0x0160001f, "[1]A,d" }, /* ldda [rs1+%g0],d */
{ "ldda",	0xc0982000, 0x01601fff, "[1]A,d" }, /* ldda [rs1+0],d */
{ "ldda",	0xc0980000, 0x01600000, "[1+2]A,d" },
{ "lduha",      0xc0900000, 0x0168001f, "[1]A,d" }, /* lduha [rs1+%g0],d */
{ "lduha",      0xc0902000, 0x01681fff, "[1]A,d" }, /* lduha [rs1+0],d */
{ "lduha",	0xc0900000, 0x01680000, "[1+2]A,d" },
{ "ldstub",     0xc0680000, 0x0190001f, "[1],d" }, /* ldstub [rs1+%g0],d */
{ "ldstub",	0xc0682000, 0x01901fff, "[1],d" }, /* ldstub [rs1+0],d */
{ "ldstub",	0xc0682000, 0x01900000, "[1+i],d" },
{ "ldstub",	0xc0682000, 0x01900000, "[i+1],d" },
{ "ldstub",	0xc0680000, 0x01900000, "[1+2],d" },
{ "lda",        0xc0800000, 0x0178001f, "[1]A,d" }, /* lda [rs1+%g0],d */
{ "lda",	0xc0802000, 0x01781fff, "[1]A,d" }, /* lda [rs1+0],d */
{ "lda",	0xc0800000, 0x01780000, "[1+2]A,d" },
{ "ldsh",       0xc0500000, 0x0000000d, "[1],d" }, /* ldsh [rs1+%g0],d */
{ "ldsh",       0xc0502000, 0x01a81fff, "[1],d" }, /* ldsh [rs1+0],d */
{ "ldsh",	0xc0502000, 0x01a80000, "[1+i],d" },
{ "ldsh",	0xc0502000, 0x01a80000, "[i+1],d" },
{ "ldsh",	0xc0500000, 0x01a80000, "[1+2],d" },
{ "ldsb",       0xc0480000, 0x01b0001f, "[1],d" }, /* ldsb [rs1+%g0],d */
{ "ldsb",	0xc0482000, 0x01b01fff, "[1],d" }, /* ldsb [rs1+0],d */
{ "ldsb",	0xc0482000, 0x01b00000, "[1+i],d" },
{ "ldsb",	0xc0482000, 0x01b00000, "[i+1],d" },
{ "ldsb",	0xc0480000, 0x01b00000, "[1+2],d" },
{ "ldub",       0xc0080000, 0x01f0001f, "[1],d" }, /* ldub [rs1+%g0],d */
{ "ldub",       0xc0082000, 0x01f01fff, "[1],d" }, /* ldub [rs1+0],d */
{ "ldub",	0xc0082000, 0x01f00000, "[1+i],d" },
{ "ldub",	0xc0082000, 0x01f00000, "[i+1],d" },
{ "ldub",	0xc0080000, 0x01f00000, "[1+2],d" },
{ "lduba",      0xc0880000, 0x0170001f, "[1]A,d" }, /* lduba [rs1+%g0],d */
{ "lduba",      0xc0882000, 0x01701fff, "[1]A,d" }, /* lduba [rs1+0],d */
{ "lduba",	0xc0880000, 0x01700000, "[1+2]A,d" },
{ "lduh",	0xc0102000, 0x01e80000, "[1+i],d" },
{ "lduh",	0xc0102000, 0x01e80000, "[i+1],d" },
{ "lduh",	0xc0100000, 0x01e8001f, "[1],d" }, /* lduh [rs1+%g0],d */
{ "lduh",	0xc0102000, 0x01e81fff, "[1],d" }, /* lduh [rs1+0],d */
{ "lduh",	0xc0100000, 0x01e80000, "[1+2],d" },

{ "st",	        0xc0200000, 0x01d8001f, "d,[1]" }, /* st d,[rs1+%g0] */
{ "st",	        0xc0202000, 0x01d81fff, "d,[1]" }, /* st d,[rs1+0] */
{ "st",		0xc0202000, 0x01d80000, "d,[1+i]" },
{ "st",		0xc0202000, 0x01d80000, "d,[i+1]" },
{ "st",		0xc0200000, 0x01d80000, "d,[1+2]" },
{ "st",		0xc1200000, 0x00d8001f, "g,[1]" }, /* st d[rs1+%g0] */
{ "st",		0xc1202000, 0x00d81fff, "g,[1]" }, /* st d,[rs1+0] */
{ "st",		0xc1202000, 0x00d80000, "g,[1+i]" },
{ "st",		0xc1202000, 0x00d80000, "g,[i+1]" },
{ "st",		0xc1200000, 0x00d80000, "g,[1+2]" },
{ "st",		0xc1100000, 0x00c0001f, "F,[1]" }, /* st d,[rs1+%g0] */
{ "st",	        0xc1102000, 0x00c01fff, "F,[1]" }, /* st d,[rs1+0] */
{ "st",		0xc1102000, 0x00c00000, "F,[1+i]" },
{ "st",		0xc1102000, 0x00c00000, "F,[i+1]" },
{ "st",		0xc1100000, 0x00c00000, "F,[1+2]" },
{ "st",		0xc1a00000, 0x0058001f, "D,[1]" }, /* st d,[rs1+%g0] */
{ "st",		0xc1a02000, 0x00581fff, "D,[1]" }, /* st d,[rs1+0] */
{ "st",		0xc1a02000, 0x00580000, "D,[1+i]" },
{ "st",		0xc1a02000, 0x00580000, "D,[i+1]" },
{ "st",		0xc1a00000, 0x00580000, "D,[1+2]" },
{ "st",		0xc1a80000, 0x0050001f, "C,[1]" }, /* st d,[rs1+%g0] */
{ "st",		0xc1a82000, 0x00501fff, "C,[1]" }, /* st d,[rs1+0] */
{ "st",		0xc1a82000, 0x00500000, "C,[1+i]" },
{ "st",		0xc1a82000, 0x00500000, "C,[i+1]" },
{ "st",		0xc1a80000, 0x00500000, "C,[1+2]" },
{ "sta",        0xc0a00000, 0x0108001f, "d,[1]A" }, /* sta d,[rs1+%g0] */
{ "sta",	0xc0a02000, 0x01081fff, "d,[1]A" }, /* sta d,[rs1+0] */
{ "sta",	0xc0a00000, 0x01080000, "d,[1+2]A" },

{ "stb",        0xc0280000, 0x01d0001f, "d,[1]" }, /* stb d,[rs1+%g0] */
{ "stb",	0xc0282000, 0x01d01fff, "d,[1]" }, /* stb d,[rs1+0] */
{ "stb",	0xc0282000, 0x01d00000, "d,[1+i]" },
{ "stb",	0xc0282000, 0x01d00000, "d,[i+1]" },
{ "stb",	0xc0280000, 0x01d00000, "d,[1+2]" },
{ "stba",       0xc0a80000, 0x01000000, "d,[1+2]A" },
{ "stba",	0xc0a80000, 0x0100001f, "d,[1]A" }, /* stba d,[rs1+%g0] */
{ "stba",	0xc0a82000, 0x01001fff, "d,[1]A" }, /* stba d,[rs1+0] */

{ "std",        0xc0380000, 0x01c0001f, "d,[1]" }, /* std d,[rs1+%g0] */
{ "std",	0xc0382000, 0x01c01fff, "d,[1]" }, /* std d,[rs1+0] */
{ "std",	0xc0382000, 0x01c00000, "d,[1+i]" },
{ "std",	0xc0382000, 0x01c00000, "d,[i+1]" },
{ "std",	0xc0380000, 0x01c00000, "d,[1+2]" },
{ "std",	0xc1380000, 0x00c0001f, "g,[1]" }, /* std d,[rs1+%g0] */
{ "std",        0xc1382000, 0x00c01fff, "g,[1]" }, /* std d,[rs1+0] */
{ "std",	0xc1382000, 0x00c00000, "g,[1+i]" },
{ "std",	0xc1382000, 0x00c00000, "g,[i+1]" },
{ "std",	0xc1380000, 0x00c00000, "g,[1+2]" },
{ "std",        0xc1300000, 0x00c8001f, "q,[1]" }, /* std d,[rs1+%g0] */
{ "std",	0xc1302000, 0x00c81fff, "q,[1]" }, /* std d,[rs1+0] */
{ "std",	0xc1302000, 0x00c80000, "q,[1+i]" },
{ "std",	0xc1302000, 0x00c80000, "q,[i+1]" },
{ "std",	0xc1300000, 0x00c80000, "q,[1+2]" },
{ "std",	0xc1b80000, 0x0040001f, "D,[1]" }, /* std d,[rs1+%g0] */
{ "std",	0xc1b82000, 0x00401fff, "D,[1]" }, /* std d,[rs1+0] */
{ "std",	0xc1b82000, 0x00400000, "D,[1+i]" },
{ "std",	0xc1b82000, 0x00400000, "D,[i+1]" },
{ "std",	0xc1b80000, 0x00400000, "D,[1+2]" },
{ "std",	0xc1b00000, 0x0048001f, "Q,[1]" }, /* std d,[rs1+%g0] */
{ "std",	0xc1b02000, 0x00481fff, "Q,[1]" }, /* std d,[rs1+0] */
{ "std",	0xc1b02000, 0x00480000, "Q,[1+i]" },
{ "std",	0xc1b02000, 0x00480000, "Q,[i+1]" },
{ "std",	0xc1b00000, 0x00480000, "Q,[1+2]" },
{ "stda",       0xc0b80000, 0x01400000, "d,[1+2]A" },
{ "stda",	0xc0b80000, 0x0140001f, "d,[1]A" }, /* stda d,[rs1+%g0] */
{ "stda",	0xc0b82000, 0x01401fff, "d,[1]A" }, /* stda d,[rs1+0] */

{ "sth",        0xc0300000, 0x01c8001f, "d,[1]" }, /* sth d,[rs1+%g0] */
{ "sth",	0xc0302000, 0x01c81fff, "d,[1]" }, /* sth d,[rs1+0] */
{ "sth",	0xc0300000, 0x01c80000, "d,[1+2]" },
{ "sth",	0xc0302000, 0x01c80000, "d,[1+i]" },
{ "sth",	0xc0302000, 0x01c80000, "d,[i+1]" },
{ "stha",       0xc0b00000, 0x0148001f, "d,[1]A" }, /* stha d,[rs1+%g0] */
{ "stha",       0xc0b02000, 0x01481fff, "d,[1]A" }, /* stha d,[rs1+0] */
{ "stha",	0xc0b00000, 0x01480000, "d,[1+2]A" },

{ "swap",       0xc0780000, 0x0180001f, "[1],d" }, /* swap [rs1+%g0],d */
{ "swap",       0xc0782000, 0x01801fff, "[1],d" }, /* swap [rs1+0],d */
{ "swap",       0xc0782000, 0x01800000, "[1+i],d" },
{ "swap",       0xc0782000, 0x01800000, "[i+1],d" },
{ "swap",       0xc0780000, 0x01800000, "[1+2],d" },
{ "swapa",      0xc0f82000, 0x01000000, "[1+2]A,d" },
{ "swapa",      0xc0f80000, 0x0100001f, "[1]A,d" }, /* swapa [rs1+%g0],d */
{ "swapa",      0xc0f82000, 0x01001fff, "[1]A,d" }, /* swapa [rs1+0],d */

{ "restore",    0x81e80000, 0x7e17e01f, "" }, /* restore %g0,%g0,%g0 */
{ "restore",    0x81e82000, 0x7e14dfff, "" }, /* restore %g0,0,%g0 */
{ "restore",	0x81e82000, 0x00000000, "1,i,d" },
{ "restore",	0x81e80000, 0x00000000, "1,2,d" },
{ "rett",       0x81c82000, 0x40300000, "1,i,d" },
{ "rett",	0x81c80000, 0x40300000, "1,2,d" },
{ "save",       0x81e02000, 0x40180000, "1,i,d" },
{ "save",	0x81e00000, 0x40180000, "1,2,d" },

{ "ret",	0x81c7e008, 0x00001ff7, "" }, /* jmpl %i7+8,%g0 */
{ "retl",       0x81c3e008, 0x00001ff7, "" }, /* jmpl %o7+8,%g0 */

{ "jmpl",       0x81c00000, 0x4038001f, "1,d" }, /* jmpl rs1+%g0,d */
{ "jmpl",	0x81c02000, 0x4037c000, "i,d" }, /* jmpl %g0+i,d */
{ "jmpl",	0x81c02000, 0x40380000, "1+i,d" },
{ "jmpl",	0x81c02000, 0x40380000, "i+1,d" },
{ "jmpl",	0x81c00000, 0x40380000, "1+2,d" },
{ "wr",         0x81982000, 0x40600000, "1,i,t" },
{ "wr",         0x81980000, 0x40600000, "1,2,t" },
{ "wr",         0x81902000, 0x40680000, "1,i,w" },
{ "wr",         0x81900000, 0x40680000, "1,2,w" },
{ "wr",         0x81882000, 0x40700000, "1,i,p" },
{ "wr",         0x81880000, 0x40700000, "1,2,p" },
{ "wr",         0x81802000, 0x40780000, "1,i,y" },
{ "wr",         0x81800000, 0x40780000, "1,2,y" },

{ "rd", 	0x81580000, 0x40a00000, "t,d" },
{ "rd", 	0x81500000, 0x40a80000, "w,d" },
{ "rd", 	0x81480000, 0x40b00000, "p,d" },
{ "rd",         0x81400000, 0x40b80000, "y,d" },

{ "sra",	0x81382000, 0x00000000, "1,i,d" },
{ "sra",	0x81380000, 0x00000000, "1,2,d" },
{ "srl",        0x81302000, 0x40c80000, "1,i,d" },
{ "srl",	0x81300000, 0x40c80000, "1,2,d" },
{ "sll",        0x81282000, 0x40d00000, "1,i,d" },
{ "sll",	0x81280000, 0x40d00000, "1,2,d" },

{ "mulscc",     0x81202000, 0x40d80000, "1,i,d" },
{ "mulscc",	0x81200000, 0x40d80000, "1,2,d" },

{ "clr",        0x80100000, 0x4e87e01f, "d" }, /* or %g0,%g0,d */
{ "clr",        0x80102000, 0x41efdfff, "d" }, /* or %g0,0,d   */

{ "orncc",      0x80b02000, 0x04048000, "1,i,d" },
{ "orncc",      0x80b02000, 0x04048000, "i,1,d" },
{ "orncc",	0x80b00000, 0x04048000, "1,2,d" },

{ "tst",        0x80900000, 0x7f6fe000, "2" }, /* orcc %g0, rs2, %g0 */
{ "tst",        0x80900000, 0x7f68201f, "1" }, /* orcc rs1, %g0, %g0 */
{ "tst",        0x80902000, 0x7f681fff, "1" }, /* orcc rs1, 0, %g0 */
  
{ "orcc",       0x80902000, 0x41680000, "1,i,d" },
{ "orcc",	0x80902000, 0x41680000, "i,1,d" },
{ "orcc",	0x80900000, 0x41680000, "1,2,d" },
{ "orn",        0x80302000, 0x41c80000, "1,i,d" },
{ "orn",	0x80302000, 0x41c80000, "i,1,d" },
{ "orn",	0x80300000, 0x41c80000, "1,2,d" },

{ "mov",        0x81800000, 0x4078001f, "1,y" }, /* wr rs1,%g0,%y */
{ "mov",        0x81802000, 0x40781fff, "1,y" }, /* wr rs1,0,%y */
{ "mov",        0x80102000, 0x41efc000, "i,d" }, /* or %g0,i,d   */
{ "mov",        0x80100000, 0x41efe000, "2,d" }, /* or %g0,rs2,d */

{ "or",	        0x80102000, 0x40800000, "1,i,d" },
{ "or",		0x80102000, 0x40800000, "i,1,d" },
{ "or",		0x80100000, 0x40800000, "1,2,d" },

{ "andncc",     0x80a82000, 0x41500000, "1,i,d" },
{ "andncc",	0x80a82000, 0x41500000, "i,1,d" },
{ "andncc",	0x80a80000, 0x41500000, "1,2,d" },
{ "andn",       0x80282000, 0x41d00000, "1,i,d" },
{ "andn",	0x80282000, 0x41d00000, "i,1,d" },
{ "andn",	0x80280000, 0x41d00000, "1,2,d" },

{ "cmp",        0x80a02000, 0x7d580000, "1,i" },     /* subcc rs1,i,%g0 */
{ "cmp",	0x80a00000, 0x7d580000, "1,2" },     /* subcc rs1,rs2,%g0 */

{ "subcc",      0x80a02000, 0x41580000, "1,i,d" },
{ "subcc",	0x80a00000, 0x41580000, "1,2,d" },
{ "sub",	0x80202000, 0x41d80000, "1,i,d" },
{ "sub",	0x80200000, 0x41d80000, "1,2,d" },
{ "subx",	0x80602000, 0x41980000, "1,i,d" },
{ "subx",	0x80600000, 0x41980000, "1,2,d" },
{ "subxcc",     0x80e02000, 0x41180000, "1,i,d" },
{ "subxcc",	0x80e00000, 0x41180000, "1,2,d" },

{ "andcc",      0x80882000, 0x41700000, "1,i,d" },
{ "andcc",	0x80882000, 0x41700000, "i,1,d" },
{ "andcc",	0x80880000, 0x41700000, "1,2,d" },
{ "and",        0x80082000, 0x41f00000, "1,i,d" },
{ "and",	0x80082000, 0x41f00000, "i,1,d" },
{ "and",	0x80080000, 0x41f00000, "1,2,d" },

{ "inc",	0x80002001, 0x00001ffe, "r" },       /* add rs1,1,rsd */

{ "addxcc",     0x80c02000, 0x41380000, "1,i,d" },
{ "addxcc",     0x80c02000, 0x41380000, "i,1,d" },
{ "addxcc",     0x80c00000, 0x41380000, "1,2,d" },
{ "addcc",      0x80802000, 0x41780000, "1,i,d" },
{ "addcc",      0x80802000, 0x41780000, "i,1,d" },
{ "addcc",      0x80800000, 0x41780000, "1,2,d" },
{ "addx",       0x80402000, 0x41b80000, "1,i,d" },
{ "addx",	0x80402000, 0x41b80000, "i,1,d" },
{ "addx",	0x80400000, 0x41b80000, "1,2,d" },
{ "add",        0x80002000, 0x41f80000, "1,i,d" },
{ "add",	0x80002000, 0x41f80000, "i,1,d" },
{ "add",	0x80000000, 0x41f80000, "1,2,d" },

{ "call",       0x9fc00000, 0x4038001f, "1" }, /* jmpl rs1+%g0, %o7 */
{ "call",	0x9fc00000, 0x4038001f, "1,#" },
{ "call",	0x40000000, 0x80000000, "L" },
{ "call",	0x40000000, 0x80000000, "L,#" },

{ "bvc",        0x3e800000, 0xc1400000, ",al" },
{ "bvc",	0x1e800000, 0xc1400000, "l" },
{ "bvs",	0x2e800000, 0xc1400000, ",al" },
{ "bvs",	0x0e800000, 0xc1400000, "l" },
{ "bpos",	0x3c800000, 0xc1400000, ",al" },
{ "bpos",	0x1c800000, 0xc1400000, "l" },
{ "bneg",	0x2c800000, 0xc1400000, ",al" },
{ "bneg",	0x0c800000, 0xc1400000, "l" },
{ "bcc",	0x3a800000, 0xc1400000, ",al" },
{ "bcc",	0x1a800000, 0xc1400000, "l" },
{ "bcs",	0x2a800000, 0xc1400000, ",al" },
{ "bcs",	0x0a800000, 0xc1400000, "l" },
{ "blu",	0x2a800000, 0xc1400000, ",al" },
{ "blu",	0x0a800000, 0xc1400000, "l" }, /* same as bcs */
{ "bgeu",	0x3a800000, 0xc1400000, ",al" },
{ "bgeu",	0x1a800000, 0xc1400000, "l" }, /* same as bcc */
{ "bgu",	0x38800000, 0xc1400000, ",al" },
{ "bgu",	0x18800000, 0xc1400000, "l" },
{ "bleu",	0x28800000, 0xc1400000, ",al" },
{ "bleu",	0x08800000, 0xc1400000, "l" },
{ "bge",	0x36800000, 0xc1400000, ",al" },
{ "bge",	0x16800000, 0xc1400000, "l" },
{ "bl",		0x26800000, 0xc1400000, ",al" },
{ "bl",		0x06800000, 0xc1400000, "l" },
{ "bg",		0x34800000, 0xc1400000, ",al" },
{ "bg",		0x14800000, 0xc1400000, "l" },
{ "ble",	0x24800000, 0xc1400000, ",al" },
{ "ble",	0x04800000, 0xc1400000, "l" },
{ "be",		0x22800000, 0xc1400000, ",al" },
{ "be",		0x02800000, 0xc1400000, "l" },
{ "bne",	0x32800000, 0xc1400000, ",al" },
{ "bne",	0x12800000, 0xc1400000, "l" },
{ "b",		0x30800000, 0xc1400000, ",al" },
{ "b",		0x10800000, 0xc1400000, "l" },
{ "ba",		0x30800000, 0xc1400000, ",al" },
{ "ba",		0x10800000, 0xc1400000, "l" },
{ "bn", 	0x20800000, 0xc1400000, ",al" },
{ "bn",		0x00800000, 0xc1400000, "l" },

{ "jmp",        0x81c00000, 0x7e38201f, "1" }, /* jmpl rs1+%g0,%g0 */
{ "jmp",        0x81c02000, 0x7e3fc000, "i" }, /* jmpl %g0+i,%g0 */

{ "nop",	0x01000000, 0xfe3fffff, "", }, /* sethi 0, %g0 */

{ "set",        0x01000000, 0xc0c00000, "Sh,d" },

{ "sethi",      0x01000000, 0xc0c00000, "h,d" },

{ "taddcctv",   0x81102000, 0x40e00000, "1,i,d" },
{ "taddcctv",   0x81100000, 0x40e00000, "1,2,d" },
{ "taddcc",     0x81002000, 0x40f80000, "1,i,d" },
{ "taddcc",     0x81000000, 0x40f80000, "1,2,d" },

{ "tvc",	0x9fd02000, 0x402fc000, "i" }, /* tvc %g0+i */
{ "tvc",        0x9fd02000, 0x40280000, "1+i" },
{ "tvc",	0x9fd00000, 0x40280000, "1+2" },
{ "tvc",	0x9fd00000, 0x4028001f, "1" }, /* tvc rs1+%g0 */
{ "tpos",	0x9dd02000, 0x402fc000, "i" }, /* tpos %g0+i */
{ "tpos",	0x9dd02000, 0x40280000, "1+i" },
{ "tpos",	0x9dd00000, 0x40280000, "1+2" },
{ "tpos",	0x9dd00000, 0x4028001f, "1" }, /* tpos rs1+%g0 */
{ "tcc",        0x9bd02000, 0x402fc000, "i" }, /* tcc %g0+i */
{ "tcc",	0x9bd02000, 0x40280000, "1+i" },
{ "tcc",	0x9bd00000, 0x40280000, "1+2" },
{ "tcc",        0x9bd00000, 0x4028001f, "1" }, /* tcc rs1+%g0 */
{ "tgu",	0x99d02000, 0x402fc000, "i" }, /* tgu %g0+i */
{ "tgu",	0x99d02000, 0x40280000, "1+i" },
{ "tgu",	0x99d00000, 0x40280000, "1+2" },
{ "tgu",	0x99d00000, 0x4028001f, "1" }, /* tgu rs1+%g0 */
{ "tge",	0x97d02000, 0x402fc000, "i" }, /* tge %g0+i */
{ "tge",	0x97d02000, 0x40280000, "1+i" },
{ "tge",	0x97d00000, 0x40280000, "1+2" },
{ "tge",        0x97d00000, 0x4028001f, "1" }, /* tge rs1+%g0 */
{ "tg",		0x95d02000, 0x402fc000, "i" }, /* tg %g0+i */
{ "tg",		0x95d02000, 0x40280000, "1+i" },
{ "tg",		0x95d00000, 0x40280000, "1+2" },
{ "tg",	        0x95d00000, 0x4028001f, "1" }, /* tg rs1+%g0 */
{ "tne",        0x93d02000, 0x402fc000, "i" }, /* tne %g0+i */
{ "tne",	0x93d02000, 0x40280000, "1+i" },
{ "tne",	0x93d00000, 0x40280000, "1+2" },
{ "tne",        0x93d00000, 0x4028001f, "1" }, /* tne rs1+%g0 */
{ "tleu",       0x8bd02000, 0x502fc000, "i" }, /* tleu %g0+i */
{ "tleu",	0x8bd02000, 0x50280000, "1+i" },
{ "tleu",	0x8bd00000, 0x50280000, "1+2" },
{ "tleu",       0x8bd00000, 0x5028001f, "1" }, /* tleu rs1+%g0 */
{ "ta",		0x91d02000, 0x402d0000, "1+i" },
{ "ta",		0x91d00000, 0x40280000, "1+2" },
{ "ta",		0x91d00000, 0x4028001f, "1" }, /* ta rs1+%g0 */
{ "ta",	        0x91d02000, 0x402fc000, "i" }, /* ta %g0+i */
{ "tvs",	0x8fd02000, 0x502fc000, "i" }, /* tvs %g0+i */
{ "tvs",	0x8fd02000, 0x50280000, "1+i" },
{ "tvs",	0x8fd00000, 0x50280000, "1+2" },
{ "tvs",	0x8fd00000, 0x5028001f, "1" }, /* tvs rs1+%g0 */
{ "tneg",	0x8dd02000, 0x502fc000, "i" }, /* tneg %g0+i */
{ "tneg",	0x8dd02000, 0x50280000, "1+i" },
{ "tneg",	0x8dd00000, 0x50280000, "1+2" },
{ "tneg",       0x8dd00000, 0x5028001f, "1" }, /* tneg rs1+%g0 */
{ "tcs",        0x8bd02000, 0x502fc000, "i" }, /* tcs %g0+i */
{ "tcs",	0x8bd02000, 0x50280000, "1+i" },
{ "tcs",	0x8bd00000, 0x50280000, "1+2" },
{ "tcs",        0x8bd00000, 0x5028001f, "1" }, /* tcs rs1+%g0 */
{ "tl",		0x87d02000, 0x502fc000, "i" }, /* tl %g0+i */
{ "tl",		0x87d02000, 0x50280000, "1+i" },
{ "tl",		0x87d00000, 0x50280000, "1+2" },
{ "tl",	        0x87d00000, 0x5028001f, "1" }, /* tl rs1+%g0 */
{ "tle",	0x85d02000, 0x502fc000, "i" }, /* tle %g0+i */
{ "tle",	0x85d02000, 0x50280000, "1+i" },
{ "tle",	0x85d00000, 0x50280000, "1+2" },
{ "tle",        0x85d00000, 0x5028001f, "1" }, /* tle rs1+%g0 */
{ "te",	        0x83d02000, 0x502fc000, "i" }, /* te %g0+i */
{ "te",		0x83d02000, 0x50280000, "1+i" },
{ "te",		0x83d00000, 0x50280000, "1+2" },
{ "te",		0x83d00000, 0x5028001f, "1" }, /* te rs1+%g0 */
{ "tn",		0x81d02000, 0x502fc000, "i" }, /* tn %g0+i */
{ "tn",	        0x81d02000, 0x50280000, "1+i" },
{ "tn",		0x81d00000, 0x50280000, "1+2" },
{ "tn",	        0x81d00000, 0x5028001f, "1" }, /* tn rs1+%g0 */

{ "tsubcc",     0x81080000, 0x40f00000, "1,2,d" },
{ "tsubcc",     0x81082000, 0x40f00000, "1,i,d" },
{ "tsubcctv",   0x80580000, 0x40a00000, "1,2,d" },
{ "tsubcctv",   0x80582000, 0x40a00000, "1,i,d" },

{ "unimp",      0x00000000, 0x00000000, "l" },

{ "iflush",     0x81d80000, 0x40200000, "1+2" },
{ "iflush",     0x81d82000, 0x40200000, "1+i" },

{ "xnorcc",     0x80b80000, 0x41400000, "1,2,d" },
{ "xnorcc",	0x80b82000, 0x41400000, "1,i,d" },
{ "xnorcc",	0x80b82000, 0x41400000, "i,1,d" },
{ "xorcc",      0x80980000, 0x41600000, "1,2,d" },
{ "xorcc",	0x80982000, 0x41600000, "1,i,d" },
{ "xorcc",	0x80982000, 0x41600000, "i,1,d" },
{ "xnor",       0x80380000, 0x41c00000, "1,2,d" },
{ "xnor",	0x80382000, 0x41c00000, "1,i,d" },
{ "xnor",	0x80382000, 0x41c00000, "i,1,d" },
{ "xor",        0x80180000, 0x41e00000, "1,2,d" },
{ "xor",	0x80182000, 0x41e00000, "1,i,d" },
{ "xor",	0x80182000, 0x41e00000, "i,1,d" },

{ "fpop1",      0x81a00000, 0x40580000, "[1+2],d" },
{ "fpop2",      0x81a80000, 0x40500000, "[1+2],d" },

{ "fb",         0x31800000, 0xc0400000, ",al" },
{ "fb",         0x11800000, 0xc0400000, "l" },
{ "fba",        0x31800000, 0xc0400000, ",al" },
{ "fba",        0x11800000, 0xc0400000, "l" },
{ "fbn",        0x21800000, 0xc0400000, ",al" },
{ "fbn",        0x01800000, 0xc0400000, "l" },
{ "fbu",        0x2f800000, 0xc0400000, ",al" },
{ "fbu",        0x0f800000, 0xc0400000, "l" },
{ "fbg",        0x2d800000, 0xc0400000, ",al" },
{ "fbg",        0x0d800000, 0xc0400000, "l" },
{ "fbug",       0x2b800000, 0xc0400000, ",al" },
{ "fbug",       0x0b800000, 0xc0400000, "l" },
{ "fbl",        0x29800000, 0xc0400000, ",al" },
{ "fbl",        0x09800000, 0xc0400000, "l" },
{ "fbul",       0x27800000, 0xc0400000, ",al" },
{ "fbul",       0x07800000, 0xc0400000, "l" },
{ "fblg",       0x25800000, 0xc0400000, ",al" },
{ "fblg",       0x05800000, 0xc0400000, "l" },
{ "fbne",       0x23800000, 0xc0400000, ",al" },
{ "fbne",       0x03800000, 0xc0400000, "l" },
{ "fbe",        0x33800000, 0xc0400000, ",al" },
{ "fbe",        0x13800000, 0xc0400000, "l" },
{ "fbue",       0x35800000, 0xc0400000, ",al" },
{ "fbue",       0x15800000, 0xc0400000, "l" },
{ "fbge",       0x37800000, 0xc0400000, ",al" },
{ "fbge",       0x17800000, 0xc0400000, "l" },
{ "fbuge",      0x39800000, 0xc0400000, ",al" },
{ "fbuge",      0x19800000, 0xc0400000, "l" },
{ "fble",       0x3b800000, 0xc0400000, ",al" },
{ "fble",       0x1b800000, 0xc0400000, "l" },
{ "fbule",      0x3d800000, 0xc0400000, ",al" },
{ "fbule",      0x1d800000, 0xc0400000, "l" },
{ "fbo",        0x3f800000, 0xc0400000, ",al" },
{ "fbo",        0x1f800000, 0xc0400000, "l" },

{ "cba",        0x31c00000, 0xce000000, ",al" },
{ "cba",        0x11c00000, 0xce000000, "l" },
{ "cbn",        0x21c00000, 0xde000000, ",al" },
{ "cbn",        0x01c00000, 0xde000000, "l" },
{ "cb3",        0x2fc00000, 0xc0000000, ",al" },
{ "cb3",        0x0fc00000, 0xc0000000, "l" },
{ "cb2",        0x2dc00000, 0xc0000000, ",al" },
{ "cb2",        0x0dc00000, 0xc0000000, "l" },
{ "cb23",       0x2bc00000, 0xc0000000, ",al" },
{ "cb23",       0x0bc00000, 0xc0000000, "l" },
{ "cb1",        0x29c00000, 0xc0000000, ",al" },
{ "cb1",        0x09c00000, 0xc0000000, "l" },
{ "cb13",       0x27c00000, 0xc0000000, ",al" },
{ "cb13",       0x07c00000, 0xc0000000, "l" },
{ "cb12",       0x25c00000, 0xc0000000, ",al" },
{ "cb12",       0x05c00000, 0xc0000000, "l" },
{ "cb123",      0x23c00000, 0xc0000000, ",al" },
{ "cb123",      0x03c00000, 0xc0000000, "l" },
{ "cb0",        0x33c00000, 0xc0000000, ",al" },
{ "cb0",        0x13c00000, 0xc0000000, "l" },
{ "cb03",       0x35c00000, 0xc0000000, ",al" },
{ "cb03",       0x15c00000, 0xc0000000, "l" },
{ "cb02",       0x37c00000, 0xc0000000, ",al" },
{ "cb02",       0x17c00000, 0xc0000000, "l" },
{ "cb023",      0x39c00000, 0xc0000000, ",al" },
{ "cb023",      0x19c00000, 0xc0000000, "l" },
{ "cb013",      0x3dc00000, 0xc0000000, ",al" },
{ "cb013",      0x1dc00000, 0xc0000000, "l" },
{ "cb012",      0x3fc00000, 0xc0000000, ",al" },
{ "cb012",      0x1fc00000, 0xc0000000, "l" },

{ "fstoi",      0x81a01a20, 0x400025c0, "f,g" },
{ "fdtoi",      0x81a01a40, 0x400025a0, "f,g" },
{ "fxtoi",      0x81a01a60, 0x40002580, "f,g" },

{ "fitox",      0x81a01980, 0x40002660, "f,g" },
{ "fitod",      0x81a01900, 0x400026e0, "f,g" },
{ "fitos",      0x81a01880, 0x40002660, "f,g" },

{ "fstod",      0x81a01920, 0x400026c0, "f,g" },
{ "fstox",      0x81a019a0, 0x40002640, "f,g" },
{ "fdtos",      0x81a018c0, 0x40002720, "f,g" },
{ "fdtox",      0x81a019c0, 0x40002620, "f,g" },
{ "fxtos",      0x81a018e0, 0x40002700, "f,g" },
{ "fxtod",      0x81a01960, 0x40002680, "f,g" },

{ "fdivx",      0x81a009e0, 0x40083600, "e,f,g" },
{ "fdivd",      0x81a009c0, 0x40003620, "e,f,g" },
{ "fdivs",      0x81a009a0, 0x40003640, "e,f,g" },
{ "fmuls",      0x81a00920, 0x400036c0, "e,f,g" },
{ "fmuld",      0x81a00940, 0x400036a0, "e,f,g" },
{ "fmulx",      0x81a00960, 0x40003680, "e,f,g" },

{ "fsqrts",     0x81a00520, 0x40003ac0, "f,g" },
{ "fsqrtd",     0x81a00540, 0x40003aa8, "f,g" },
{ "fsqrtx",     0x81a00560, 0x40003a80, "f,g" },

{ "fabss",      0x81a00120, 0x40003ec0, "f,g" },
{ "fnegs",      0x81a000a0, 0x40003f40, "f,g" },
{ "fmovs",      0x81a00020, 0x40003fc0, "f,g" },

{ "fsubx",      0x81a008e0, 0x40003700, "e,f,g" },
{ "fsubd",      0x81a008c0, 0x40003720, "e,f,g" },
{ "fsubs",      0x81a008a0, 0x40003740, "e,f,g" },
{ "faddx",      0x81a00860, 0x40003780, "e,f,g" },
{ "faddd",      0x81a00840, 0x400037a0, "e,f,g" },
{ "fadds",      0x81a00820, 0x400037c0, "e,f,g" },

{ "fcmpex",     0x81a80ae0, 0x40003500, "e,f" },
{ "fcmped",     0x81a80ac0, 0x40003520, "e,f" },
{ "fcmpes",     0x81a80aa0, 0x40003540, "e,f" },
{ "fcmpx",      0x81a80a60, 0x40003580, "e,f" },
{ "fcmpd",      0x81a80a40, 0x400035a0, "e,f" },
{ "fcmps",      0x81a80a20, 0x400035c0, "e,f" },

{ "cpop1",      0x81b00000, 0x40480000, "[1+2],d" },
{ "cpop2",      0x81b80000, 0x40400000, "[1+2],d" },

};

#define NUMOPCODES ((sizeof sparc_opcodes)/(sizeof *sparc_opcodes))

