/* Core dump and executable file functions below target vector, for GDB.
   Copyright 1986, 87, 89, 91, 92, 93, 94, 95, 96, 97, 1998, 2000
   Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "defs.h"
#include "gdb_string.h"
#include <errno.h>
#include <signal.h>

#define _LARGEFILE64_SOURCE /* Required for using stat64/open64 */
#include <fcntl.h>

#include <sys/stat.h>
#include "elf-bfd.h"
#include "frame.h"		/* required by inferior.h */
#include "inferior.h"
#include "symtab.h"
#include "command.h"
#include "bfd.h"
#include "target.h"
#include "gdbcore.h"
#include "gdbthread.h"
#include "gdbcmd.h"

#include <sys/core.h>
#ifndef IS_CMA_PID
#define IS_CMA_PID(pid)		0
#endif

/* Current bfd of the core. */
bfd *current_core_bfd;

/* For debugging a core produced using dumpcore command */
int live_core = 0;

/* List of all available core_fns.  On gdb startup, each core file register
   reader calls add_core_fns() to register information on each core format it
   is prepared to read. */

static struct core_fns *core_file_fns = NULL;

/* The core_fns for a core file handler that is prepared to read the core
   file currently open on core_bfd. */

static struct core_fns *core_vec = NULL;

static void core_files_info (struct target_ops *);

#ifdef SOLIB_ADD
static int solib_add_stub (PTR);
#endif

static struct core_fns *sniff_core_bfd (bfd *);

static boolean gdb_check_format (bfd *);

static void core_open (char *, int);

void core_detach (char *, int);

static void core_close (int);

static void core_close_cleanup (void *ignore);

static void get_core_registers (int);

static void add_to_thread_list (bfd *, asection *, PTR);

static int ignore (CORE_ADDR, char *);

static char *core_file_to_sym_file (char *);

static int core_file_thread_alive (int tid);

static void init_core_ops (void);

void _initialize_corelow (void);

struct target_ops core_ops;

#ifdef HP_XMODE
/* 
 * External functions defined in top.c to check the executable format
 * of the program to exec and to launch a different gdb if necessary.
 */
extern int xmode_exec_format_is_different (char *executable);
extern void xmode_launch_other_gdb (char *new_file, enum xmode_cause reason);
#endif

#ifdef CORE_TRUNCATION_DETECTION
  #if defined(GDB_TARGET_IS_HPPA_20W) || defined (HP_IA64)
         void check_core_truncation (void);
  #else
         void check_core_truncation_pa32 (void);
  #endif
#endif  /* CORE_TRUNCATION_DETECTION */

/* Link a new core_fns into the global core_file_fns list.  Called on gdb
   startup by the _initialize routine in each core file register reader, to
   register information about each format the the reader is prepared to
   handle. */

void
add_core_fns (cf)
     struct core_fns *cf;
{
  cf->next = core_file_fns;
  core_file_fns = cf;
}

/* The default function that core file handlers can use to examine a
   core file BFD and decide whether or not to accept the job of
   reading the core file. */

int
default_core_sniffer (our_fns, abfd)
     struct core_fns *our_fns;
     bfd *abfd;
{
  int result;

  result = (bfd_get_flavour (abfd) == our_fns -> core_flavour);
  return (result);
}

/* Walk through the list of core functions to find a set that can
   handle the core file open on ABFD.  Default to the first one in the
   list of nothing matches.  Returns pointer to set that is
   selected. */

static struct core_fns *
sniff_core_bfd (abfd)
     bfd *abfd;
{
  struct core_fns *cf;
  struct core_fns *yummy = NULL;
  int matches = 0;;

  for (cf = core_file_fns; cf != NULL; cf = cf->next)
    {
      if (cf->core_sniffer (cf, abfd))
	{
	  yummy = cf;
	  matches++;
	}
    }
  if (matches > 1)
    {
      warning ("\"%s\": ambiguous core format, %d handlers match",
	       bfd_get_filename (abfd), matches);
    }
  else if (matches == 0)
    {
      warning ("\"%s\": no core file handler recognizes format, using default",
	       bfd_get_filename (abfd));
    }
  if (yummy == NULL)
    {
      yummy = core_file_fns;
    }
  return (yummy);
}

/* The default is to reject every core file format we see.  Either
   BFD has to recognize it, or we have to provide a function in the
   core file handler that recognizes it. */

int
default_check_format (abfd)
     bfd *abfd;
{
  return (0);
}

/* Attempt to recognize core file formats that BFD rejects. */

static boolean 
gdb_check_format (abfd)
     bfd *abfd;
{
  struct core_fns *cf;

  for (cf = core_file_fns; cf != NULL; cf = cf->next)
    {
      if (cf->check_format (abfd))
	{
	  return (true);
	}
    }
  return (false);
}

/* Discard all vestiges of any previous core file and mark data and stack
   spaces as empty.  */

/* ARGSUSED */
static void
core_close (quitting)
     int quitting;
{
  char *name;

  if (core_bfd)
    {
      inferior_pid = 0;		/* Avoid confusion from thread stuff */

      /* Clear out solib state while the bfd is still open. See
         comments in clear_solib in solib.c. */
#ifdef CLEAR_SOLIB
      CLEAR_SOLIB ();
#endif

      name = bfd_get_filename (core_bfd);
      if (!bfd_close (core_bfd))
	warning ("cannot close \"%s\": %s",
		 name, bfd_errmsg (bfd_get_error ()));
      free (name);
      core_bfd = NULL;
      if (core_ops.to_sections)
	{
	  free ((PTR) core_ops.to_sections);
	  core_ops.to_sections = NULL;
	  core_ops.to_sections_end = NULL;
	}
    }
  core_vec = NULL;
}

static void
core_close_cleanup (void *ignore)
{
  core_close (0/*ignored*/);
}

#ifdef SOLIB_ADD
/* Stub function for catch_errors around shared library hacking.  FROM_TTYP
   is really an int * which points to from_tty.  */

static int
solib_add_stub (from_ttyp)
     PTR from_ttyp;
{
  /* Fixed DTS CLLbs14321 - auto-solib-add doesn't work for core files.
     The second parameter to SOLIB_ADD should be zero to indicate that
     it is not from tty.

     The same thing is done in  wait_for_inferior when it adds shared
     libraries automatically.
  */
  SOLIB_ADD (NULL, 0, &current_target);
  re_enable_breakpoints_in_shlibs ();
  return 0;
}
#endif /* SOLIB_ADD */

/* Look for sections whose names start with `.reg/' so that we can extract the
   list of threads in a core file.  */

static void
add_to_thread_list (abfd, asect, reg_sect_arg)
     bfd *abfd;
     asection *asect;
     PTR reg_sect_arg;
{
  int thread_id;
  asection *reg_sect = (asection *) reg_sect_arg;

  if (strncmp (bfd_section_name (abfd, asect), ".reg/", 5) != 0)
    return;

  thread_id = atoi (bfd_section_name (abfd, asect) + 5);

  add_thread (thread_id);

/* Warning, Will Robinson, looking at BFD private data! */

  if (reg_sect != NULL
      && asect->filepos == reg_sect->filepos)	/* Did we find .reg? */
    inferior_pid = thread_id;	/* Yes, make it current */
}

#ifdef HP_MXN

/* bindu 101601: For MxN threads, gdb needs to get both the kernel thread list and the 
 * thread list maintained by the pthread library. libmxndbg will help us collate this
 * list. libmxndbg will ask us for the kernel thread list and gets the pthreads 
 * thread list directly by reading the data of libpthread. 
 */

static bfd *the_bfd;
static void
read_thread_list(bfd *abfd)
{
  the_bfd = abfd;
  update_thread_state(0);
}

/* bindu 101601: Callback provided for libmxndbg to get the kernel thread list. */

struct thr_list* 
read_kthread_list()
{
#if defined(GDB_TARGET_IS_HPPA_20W) || defined (HP_IA64)
  return bfd_kthread_list(the_bfd); 
#else
  return hpux_bfd_kthread_list(the_bfd); 
#endif
}

#endif
extern char* map_si_code_to_str (int signal, int si_code);

/* This routine opens and sets up the core file bfd.  */

static void
core_open (filename, from_tty)
     char *filename;
     int from_tty;
{
  const char *p;
  int siggy;
  int si_code;
  struct cleanup *old_chain;
  char *temp;
  bfd *temp_bfd;
  int ontop;
  int scratch_chan;

  target_preopen (from_tty);
  if (!filename)
    {
      error (core_bfd ?
	     "No core file specified.  (Use `detach' to stop debugging a core file.)"
	     : "No core file specified.");
    }

  filename = tilde_expand (filename);
  if (filename[0] != '/')
    {
      temp = concat (current_directory, "/", filename, NULL);
      free (filename);
      filename = temp;
    }

  old_chain = make_cleanup (free, filename);

  /* srikanth, open core files in large file mode for PA64. */
#ifndef SEEK_64_BITS
  scratch_chan = open (filename, write_files ? O_RDWR : O_RDONLY, 0);
#else 
  scratch_chan = open64 (filename, write_files ? O_RDWR : O_RDONLY, 0);
#endif
  if (scratch_chan < 0)
    perror_with_name (filename);

#ifdef HP_XMODE
  /* For cross mode support between 32-bit and 64-bit applications.
     If the core file format is different from what this
     gdb understands, launch the gdb that will understand
     this core file. */
      
  if (xmode_exec_format_is_different (filename))
    {
      if (exec_bfd == NULL)
        xmode_launch_other_gdb (filename, xmode_corefile);
      else
        {
	  printf ("Core file is in different format from executable\n");
	  error ("To unload the executable use the \"file\" command with no argument\n");
	}
    }
#endif

  temp_bfd = bfd_fdopenr (filename, gnutarget, scratch_chan);
  if (temp_bfd == NULL)
    perror_with_name (filename);

  if (!bfd_check_format (temp_bfd, bfd_core) &&
      !gdb_check_format (temp_bfd))
    {
      /* Do it after the err msg */
      /* FIXME: should be checking for errors from bfd_close (for one thing,
         on error it does not free all the storage associated with the
         bfd).  */
      make_cleanup_bfd_close (temp_bfd);
      error ("\"%s\" is not a core dump: %s",
	     filename, bfd_errmsg (bfd_get_error ()));
    }

  /* Looks semi-reasonable.  Toss the old core file and work on the new.  */

  discard_cleanups (old_chain);	/* Don't free filename any more */
  unpush_target (&core_ops);
  core_bfd = temp_bfd;

#ifdef BFD_IS_SWIZZLED
  is_swizzled = BFD_IS_SWIZZLED (core_bfd);
#endif

  old_chain = make_cleanup (core_close_cleanup, 0 /*ignore*/);

  /* Find a suitable core file handler to munch on core_bfd */
  core_vec = sniff_core_bfd (core_bfd);

#ifdef CORE_TRUNCATION_DETECTION
  #if defined(GDB_TARGET_IS_HPPA_20W) || defined (HP_IA64)
     check_core_truncation ();
  #else
    check_core_truncation_pa32 ();
  #endif
#endif  /* CORE_TRUNCATION_DETECTION */

  validate_files ();

  /* Find the data section */
  if (build_section_table (core_bfd, &core_ops.to_sections,
			   &core_ops.to_sections_end))
    error ("\"%s\": Can't find sections: %s",
	   bfd_get_filename (core_bfd), bfd_errmsg (bfd_get_error ()));

  set_gdbarch_from_file (core_bfd);

  ontop = !push_target (&core_ops);
  discard_cleanups (old_chain);

  p = bfd_core_file_failing_command (core_bfd);
  if (p)
    {
      printf_filtered ("Core was generated by `%s'.\n", p);
      if (strlen(p) == 14)	/* Workaround for JAGaf40286 */
        warning ("%s is 14 characters in length.  Due to a limitation \n\
	in the HP-UX kernel, core files contain only the first 14 characters \n\
	of an executable's name.  Check if %s is a truncated name. \n\
	If it is so, core-file, packcore and other commands dealing with \n\
	core files will exhibit incorrect behavior.  To avoid this, issue \n\
	exec-file and symbol-file commands with the full name of the executable \n\
	that produced the core; then issue the core-file, packcore or other \n\
	core file command of interest.\n", p, p);
     }

  siggy = bfd_core_file_failing_signal (core_bfd);
  if (siggy > 0)
    /* NOTE: target_signal_from_host() converts a target signal value
       into gdb's internal signal value.  Unfortunatly gdb's internal
       value is called ``target_signal'' and this function got the
       name ..._from_host(). */
    printf_filtered ("Program terminated with signal %d, %s.\n", siggy,
		     target_signal_to_string (target_signal_from_host (siggy)));

#ifdef GDB_TARGET_IS_HPUX
  si_code = bfd_core_file_failing_si_code (core_bfd);
  printf_filtered ("%s\n", map_si_code_to_str (siggy, si_code));
#endif

  /* Build up thread list from BFD sections. */

  init_thread_list ();
  bfd_map_over_sections (core_bfd, add_to_thread_list,
			 bfd_get_section_by_name (core_bfd, ".reg"));

  current_core_bfd = core_bfd;
  if (ontop)
    {
      /* Fetch all registers from core file.  */
      target_fetch_registers (-1);

      /* Add symbols and section mappings for any shared libraries.  */
#ifdef SOLIB_ADD
      catch_errors (solib_add_stub, &from_tty, (char *) 0,
		    RETURN_MASK_ALL);
#endif


#ifdef HP_MXN
#if defined(GDB_TARGET_IS_HPPA_20W) || defined(HP_IA64)
      /* If libpthread init is not done, do not go to libmxndbg to get the
	 the thread list. Consider this as non-mxn case. */
      if (is_mxn && !mxn_libpthread_init_done ())
	is_mxn = 0;
#endif

      if (is_mxn)
        {
          init_thread_list ();
          read_thread_list(core_bfd);
        }
#endif

      /* Now, set up the frame cache, and print the top of stack.  */
      flush_cached_frames ();
      select_frame (get_current_frame (), 0);
      print_stack_frame (selected_frame, selected_frame_level, 1);
    }
  else
    {
      warning (
		"you won't be able to access this core file until you terminate\n\
your %s; do ``info files''", target_longname);
    }
}

void
core_detach (args, from_tty)
     char *args;
     int from_tty;
{
  if (args)
    error ("Too many arguments");
  unpush_target (&core_ops);
  reinit_frame_cache ();
  if (from_tty)
    printf_filtered ("No core file now.\n");
}


/* Try to retrieve registers from a section in core_bfd, and supply
   them to core_vec->core_read_registers, as the register set numbered
   WHICH.

   If inferior_pid is zero, do the single-threaded thing: look for a
   section named NAME.  If inferior_pid is non-zero, do the
   multi-threaded thing: look for a section named "NAME/PID", where
   PID is the shortest ASCII decimal representation of inferior_pid.

   HUMAN_NAME is a human-readable name for the kind of registers the
   NAME section contains, for use in error messages.

   If REQUIRED is non-zero, print an error if the core file doesn't
   have a section by the appropriate name.  Otherwise, just do nothing.  */

/* Stacey 10/8/2001   
   The following two global variables are part of a work around for 
   the fact that GDB cannot handle 64 bit register values from PA2.0 
   machines - to allow GDB to read the true 64 bit values of registers
   from a core file if necessary.  They represent the following quantities:

   1.  Contents of save state structure from core file.  This data structure
       contains the 32 and 64 bit values of all registers from the moment
       the program died.  
   2.  Offset of start of 64 bit register values in save state structure.
   
   So core_save_state_contents + register64_offset is the address of 
   the beginning of 64 bit register values in the save state_structure.  
   For more info on the save state structure, see 
   
   /usr/include/machine/save_state.h 
   
   These variables should be removed and the code below should be adjusted
   in the event that GDB become able to handle 64 bit register values from 
   PA2.0 machines.  */

char *core_save_state_contents=NULL;
CORE_ADDR register64_offset=0x0;


#ifndef HP_IA64
static void
get_core_register_section (char *name,
			   int which,
			   char *human_name,
			   int required)
{
  char section_name[100];
  sec_ptr section;
  bfd_size_type size;
  char *contents;
#ifdef HP_MXN
/* bindu 101601: If no associated kernel thread, ask for registers from
   libmxndbg. Check infttrace.c for the use of TID_MARKER. */
  if (inferior_pid & TID_MARKER)
    {
      fetch_inferior_registers(-1);
      registers_fetched ();
      return;
    }
#endif


  if (inferior_pid && (IS_CMA_PID(inferior_pid)) == 0)
    sprintf (section_name, "%s/%d", name, inferior_pid);
  else
    strcpy (section_name, name);

  section = bfd_get_section_by_name (core_bfd, section_name);
  if (! section)
    {
      if (required)
	warning ("Couldn't find %s registers in core file.\n", human_name);
      return;
    }

  size = bfd_section_size (core_bfd, section);

  /* Stacey 10/8/2001
     Since 'contents' is now a global variable: core_save_state_contents
     the memory it points to should be allocated on the heap, 
     not on the stack because we don't wish to have it freed automatically 
     after this function returns.  

     contents = alloca (size); */

   core_save_state_contents = (char *) xmalloc (size);

  if (! bfd_get_section_contents (core_bfd, section, core_save_state_contents,
				  (file_ptr) 0, size))
    {
      warning ("Couldn't read %s registers from `%s' section in core file.\n",
	       human_name, name);
      return;
    }

  core_vec->core_read_registers (core_save_state_contents, size, which, 
				 ((CORE_ADDR)
				  bfd_section_vma (core_bfd, section)));
}
#endif	/* ! HP_IA64 */

/* ia64_get_core_registers() - cloned from get_core_registers on ia64 
   branch.

   Get the registers out of a core file.  This is the machine-
   independent part.  Fetch_core_registers is the machine-dependent
   part, typically implemented in the xm-file for each architecture.  */

/* We just get all the registers, so we don't use regno.  */

static void
ia64_get_core_registers()
{
  sec_ptr reg_sec;
  unsigned size;
  char *the_regs;
  char secname[30];
  enum bfd_flavour our_flavour = bfd_get_flavour (core_bfd);
  struct core_fns *cf = NULL;

  if (core_file_fns == NULL)
    {
      fprintf_filtered (gdb_stderr,
		     "Can't fetch registers from this type of core file\n");
      return;
    }

#ifdef HP_MXN
/* bindu 101601: If no associated kernel thread, ask for registers from
   libmxndbg. Check infttrace.c for the use of TID_MARKER. */
  if (inferior_pid & TID_MARKER)
    {
      fetch_inferior_registers(-1);
      registers_fetched ();
      return;
    }
#endif

  /* Thread support.  If inferior_pid is non-zero, then we have found a core
     file with threads (or multiple processes).  In that case, we need to
     use the appropriate register section, else we just use `.reg'. */

  /* XXX - same thing needs to be done for floating-point (.reg2) sections. */

  if (inferior_pid && (IS_CMA_PID(inferior_pid)) == 0)
    sprintf (secname, ".reg/%d", inferior_pid);
  else
    strcpy (secname, ".reg");

  reg_sec = bfd_get_section_by_name (core_bfd, secname);
  if (!reg_sec)
    goto cant;
  size = bfd_section_size (core_bfd, reg_sec);
  the_regs = (char *) alloca (size);
  /* Look for the core functions that match this flavor.  Default to the
     first one if nothing matches. */
  for (cf = core_file_fns; cf != NULL; cf = cf->next)
    {
      if (our_flavour == cf->core_flavour)
	{
	  break;
	}
    }
  if (cf == NULL)
    {
      cf = core_file_fns;
    }
  if (cf != NULL &&
      bfd_get_section_contents (core_bfd, reg_sec, the_regs, (file_ptr) 0, size) &&
      cf->core_read_registers != NULL)
    {
      (cf->core_read_registers (the_regs, size, 0,
			      (CORE_ADDR) bfd_section_vma (abfd, reg_sec)));
    }
  else
    {
    cant:
      fprintf_filtered (gdb_stderr,
			"Couldn't fetch registers from core file: %s\n",
			bfd_errmsg (bfd_get_error ()));
    }

  /* Now do it again for the float registers, if they exist.  */
  reg_sec = bfd_get_section_by_name (core_bfd, ".reg2");
  if (reg_sec)
    {
      size = bfd_section_size (core_bfd, reg_sec);
      the_regs = (char *) alloca (size);
      if (cf != NULL &&
	  bfd_get_section_contents (core_bfd, reg_sec, the_regs, (file_ptr) 0, size) &&
	  cf->core_read_registers != NULL)
	{
	  (cf->core_read_registers (the_regs, size, 2,
			       (unsigned) bfd_section_vma (abfd, reg_sec)));
	}
      else
	{
	  fprintf_filtered (gdb_stderr,
		       "Couldn't fetch register set 2 from core file: %s\n",
			    bfd_errmsg (bfd_get_error ()));
	}
    }
  registers_fetched ();
} /* end ia64_get_core_registers */


/* Get the registers out of a core file.  This is the machine-
   independent part.  Fetch_core_registers is the machine-dependent
   part, typically implemented in the xm-file for each architecture.  */

/* We just get all the registers, so we don't use regno.  */

/* ARGSUSED */

static void
get_core_registers (regno)
     int regno;
{
  int status;

  if (core_vec == NULL
      || core_vec->core_read_registers == NULL)
    {
      fprintf_filtered (gdb_stderr,
		     "Can't fetch registers from this type of core file\n");
      return;
    }

#ifdef HP_MXN
/* bindu 101601: If no associated kernel thread, ask for registers from 
   libmxndbg. Check infttrace.c for the use of TID_MARKER. */
  if (inferior_pid & TID_MARKER)
    {
      fetch_inferior_registers(-1);
      registers_fetched ();
      return;
    }
#endif

#ifndef HP_IA64
  get_core_register_section (".reg", 0, "general-purpose", 1);
  get_core_register_section (".reg2", 2, "floating-point", 0);
  get_core_register_section (".reg-xfp", 3, "extended floating-point", 0);
#else
  ia64_get_core_registers();
#endif

  registers_fetched ();
}

static char *
core_file_to_sym_file (core)
     char *core;
{
  CONST char *failing_command;
  char *p;
  char *temp;
  bfd *temp_bfd;
  int scratch_chan;

  if (!core)
    error ("No core file specified.");

  core = tilde_expand (core);
  make_cleanup (free, core);
  if (core[0] != '/')
    {
      temp = concat (current_directory, "/", core, NULL);
      core = temp;
    }

  scratch_chan = open (core, write_files ? O_RDWR : O_RDONLY, 0);
  if (scratch_chan < 0)
    perror_with_name (core);

#ifdef HP_XMODE
  /* For cross mode support between 32-bit and 64-bit applications.
     If the core file format is different from what this
     gdb understands, launch the gdb that will understand
     this core file. */
      
  if (xmode_exec_format_is_different (core))
    {
      if (exec_bfd == NULL)
        xmode_launch_other_gdb (core, xmode_corefile);
      else
        {
	  printf ("Core file is in different format from executable\n");
	  error ("To unload the executable use the \"file\" command with no argument\n");
	}
    }
#endif

  temp_bfd = bfd_fdopenr (core, gnutarget, scratch_chan);
  if (temp_bfd == NULL)
    perror_with_name (core);

  if (!bfd_check_format (temp_bfd, bfd_core))
    {
      /* Do it after the err msg */
      /* FIXME: should be checking for errors from bfd_close (for one thing,
         on error it does not free all the storage associated with the
         bfd).  */
      make_cleanup_bfd_close (temp_bfd);
      error ("\"%s\" is not a core dump: %s",
	     core, bfd_errmsg (bfd_get_error ()));
    }

  /* Find the data section */
  if (build_section_table (temp_bfd, &core_ops.to_sections,
			   &core_ops.to_sections_end))
    error ("\"%s\": Can't find sections: %s",
	   bfd_get_filename (temp_bfd), bfd_errmsg (bfd_get_error ()));

  failing_command = bfd_core_file_failing_command (temp_bfd);

  /* If we found a filename, remember that it is probably saved
     relative to the executable that created it.  If working directory
     isn't there now, we may not be able to find the executable.  Rather
     than trying to be sauve about finding it, just check if the file
     exists where we are now.  If not, then punt and tell our client
     we couldn't find the sym file.
   */
  p = (char *) failing_command;
  if ((p != NULL) && (access (p, F_OK) != 0))
    p = NULL;

  /* bfd__close would free up failing_command storage! */
  bfd_close (temp_bfd);

  return p;
}

static void
core_files_info (t)
     struct target_ops *t;
{
  print_section_info (t, core_bfd);
}

/* If mourn is being called in all the right places, this could be say
   `gdb internal error' (since generic_mourn calls breakpoint_init_inferior).  */

static int
ignore (addr, contents)
     CORE_ADDR addr;
     char *contents;
{
  return 0;
}


/* Okay, let's be honest: threads gleaned from a core file aren't
   exactly lively, are they?  On the other hand, if we don't claim
   that each & every one is alive, then we don't get any of them
   to appear in an "info thread" command, which is quite a useful
   behaviour.
 */
static int
core_file_thread_alive (tid)
     int tid;
{
  return 1;
}

/* Fill in core_ops with its defined operations and properties.  */

static void
init_core_ops ()
{
  core_ops.to_shortname = "core";
  core_ops.to_longname = "Local core dump file";
  core_ops.to_doc =
    "Use a core file as a target.  Specify the filename of the core file.";
  core_ops.to_open = core_open;
  core_ops.to_close = core_close;
  core_ops.to_attach = find_default_attach;
  core_ops.to_require_attach = find_default_require_attach;
  core_ops.to_detach = core_detach;
  core_ops.to_require_detach = find_default_require_detach;
  core_ops.to_fetch_registers = get_core_registers;
  core_ops.to_xfer_memory = xfer_memory;
  core_ops.to_files_info = core_files_info;
  core_ops.to_insert_breakpoint = ignore;
  core_ops.to_remove_breakpoint = ignore;
  core_ops.to_create_inferior = find_default_create_inferior;
  core_ops.to_clone_and_follow_inferior = find_default_clone_and_follow_inferior;
  core_ops.to_thread_alive = core_file_thread_alive;
  core_ops.to_core_file_to_sym_file = core_file_to_sym_file;
  core_ops.to_stratum = core_stratum;
  core_ops.to_has_memory = 1;
  core_ops.to_has_stack = 1;
  core_ops.to_has_registers = 1;
  core_ops.to_magic = OPS_MAGIC;
}

/* non-zero if we should not do the add_target call in
   _initialize_corelow; not initialized (i.e., bss) so that
   the target can initialize it (i.e., data) if appropriate.
   This needs to be set at compile time because we don't know
   for sure whether the target's initialize routine is called
   before us or after us. */
int coreops_suppress_target;

void
_initialize_corelow ()
{
  init_core_ops ();

  if (!coreops_suppress_target)
    add_target (&core_ops);

/* dumpcore command - JAGae75018*/
  add_show_from_set
    (add_set_cmd ("live-core", class_support, var_zinteger,
                  (char *) &live_core,
                  "Set to debug core generated by dumpcore command\n",
                  &setlist),
     &showlist);
}

int
is_live_core ()
{
  return live_core;
}

#ifdef CORE_TRUNCATION_DETECTION 
/* Function:	void check_core_truncation (void)
 * Description:	Checks if a core is truncated.  On IA and PA64, the kernel dumps each
 *		loaded memory segment (data, mmf, stack, etc.) into separate
 *		program segments in an ELF file.  This routine checks if any
 *		program segment's offset and size in the core file exceeds
 *		the size of the core file.  If so a warning is issued.
 * Input:	None.
 * Output:	None.
 * Globals:	core_bfd needs to be valid for this function.
 * Notes:	There is no need to detect a truncated ELF header or a truncated
 *		program header table because it is already detected for all 
 *		files.  Also, section truncation isn't detected here because 
 *		there is already functionality in gdb that will detect missing
 *		or incomplete sections.  
 *		JAGaf03652.	- Bharath, 6 Feb 2004.
 *              Fix for JAGaf09230 for wdb.pa
 */ 
#if defined(GDB_TARGET_IS_HPPA_20W) || defined(HP_IA64)

void 
check_core_truncation (void)
{
  int i;
  struct stat64 st_buf; /* Use stat64 as stat cannot open file with size > 2GB */
  Elf_Internal_Phdr *phdr;

  /* Read in ELF program headers if they haven't been read in already. */

  elf_text_start_vma (core_bfd);

  /* Find out the size of the core file. */
  if (stat64 (core_bfd->filename, &st_buf) == -1)
  {
    warning ("Unable to proceed with determining if the core file is\n\
truncated because: stat() on %s failed - %s.\n\
If you have problems with debugging, it could be due to a truncated core",
	     core_bfd->filename, strerror (errno));
    return;
  }

  /* If any program header's location and size go beyond the core file size
   * issue a warning about the core being truncated.
   */
  phdr = elf_tdata (core_bfd)->phdr;
  for (i = 0; i < elf_elfheader (core_bfd)->e_phnum; i++, phdr++)
  {
    if ((phdr->p_offset + phdr->p_filesz) > st_buf.st_size)
    {
      warning ("%s is truncated.\n\
      Proceeding anyway, but you may run into problems.  To avoid problems\n\
      ensure that core file is complete.  Common reasons for truncation\n\
      include system limit on core file size and lack of disk space.", 
      core_bfd->filename);
      break;
    }
  }
}

#else

/* Function:	void check_core_truncation_pa32 (void)
 * Description:	Checks if a core is truncated.  On PA32, the 
 * 		total size of each header and its corresponding 
 * 		section is calculated and compared with the size 
 * 		of the corefile. If it exceeds, then a warning is 
 *		issued.
 * Input:	None.
 * Output:	None.
 * Globals:	core_bfd needs to be valid for this function.

                This is a fix for JAGaf09230
 */

   void
   check_core_truncation_pa32 (void)
   {
      int val;
      struct stat64 st_buf; /* Use stat64 as stat cannot open file with size > 2GB */
      struct corehead chdr;
      long core_size = 0;
   
      /* Find out the size of the core file. */
      if (stat64 (core_bfd->filename, &st_buf) == -1)
      {
        error ("Unable to stat() %s.  Reason: %s.\n", 
               core_bfd->filename, strerror (errno));
        return;
      }
      if (bfd_seek (core_bfd, 0, SEEK_SET) != 0)
        error ("Error in bfd_seek:  \n", strerror (errno));
      while (1)
       {
         val = bfd_read ((void *) &chdr, 1, sizeof (struct corehead), core_bfd);
         if (val <= 0)
            break;
         core_size += sizeof (struct corehead) + chdr.len;
         bfd_seek (core_bfd, chdr.len, SEEK_CUR);
       }
         if (core_size > st_buf.st_size)
              {
                 warning ("%s is truncated.\n\nProceeding anyway, but you may run into problems. \nTo avoid problems ensure that core file is not truncated. \nCommon reasons for truncation include system limit on core file size\nand lack of disk space.\n", core_bfd->filename);
              }
       }
     #endif
#endif  /* CORE_TRUNCATION_DETECTION */
