# Copyright (C) 1999, 2000 Milan Zamazal
#
# COPYRIGHT NOTICE
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
# for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; see the file COPYING.  If not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.


"""Interface to GNATS databases.

This module is the only interface for accessing GNATS databases.  Access is
provided through the class 'Gnats'.

GNATS databases are accessed only by GNATS commands (both public and internal).
"""

__author__  = "Milan Zamazal <pdm@freesoft.cz>"
__version__ = "$Id: gnats.py,v 1.20 2000/07/24 19:08:44 pdm Exp $"
__copyright__ = "GNU General Public License, version 2"


import os
import popen2
import rfc822
import string
import StringIO
import time
import whrandom

from gnats2w import config, mail, problem
from gnats2w.i18n import L


# Classes


class Gnats:
    """Gnats database accessor.

    It provides the following groups of methods:

     - Database information: 'directory', 'site', 'version', 'categories'
     - People: 'responsible', 'responsible_full', 'id2email', 'administrator'
     - Problem reports: 'problem', 'update_problem', 'check_problem', 'submit'
     - Searching: 'query'
     - Dumping PR in GNATS format: 'dump'
     - Auxiliary: 'date', 'update_trail'
    """

    # Constants


    _GTAGS = ('Number',
              'Category',
              'Synopsis',
              'Confidential',
              'Severity',
              'Priority',
              'Responsible',
              'State',
              'Class',
              'Submitter-Id',
              'Arrival-Date',
              'Closed-Date',
              'Last-Modified',
              'Originator',
              'Release',
              'Organization',
              'Environment',
              'Description',
              'How-To-Repeat',
              'Fix',
              'Release-Note',
              'Audit-Trail',
              'Unformatted'
              )

    _GTAGS_MULTILINE = ('Organization', 'Environment', 'Description',
                        'How-To-Repeat', 'Fix', 'Audit-Trail', 'Unformatted')


    # Constructor
    
    
    def __init__ (self, directory=None, site=None):
        """Arguments:

         directory -- GNATS database.  If it is an empty string, use the
           default directory. If it is 'None', use 'config.GNATS_DIR'.

         site -- Site to submit bugs to.  If it is an empty string, use the
           default directory. If it is 'None', use 'config.SITE'.
        """
        if directory == None:
            self._directory = config.GNATS_DIR
        else:
            self._directory = directory
        if site == None:
            self._site = config.SITE
        else:
            self._site = site


    # Private methods
    

    def _protect (self, str):
        """Protect string 'str' to be passed as an argument to a shell command.

        This replaces any apostrophes and NULL characters by dots and removes
        trailing backslashes.
        """
        while str and str[-1] == '\\':
            str = str[:-1]
        for dangerous in "'", "\0":
            str = string.replace (str, dangerous, ".")
        return str


    def _extract_first (self, list, sort=0):
        """Return list of the first elements of 'list' sublists.

        If 'list' is 'None', return 'None'.
        Iff sort is true, sort the resulting list.
        """
        if list == None:
            return None
        items = map (lambda i: i[0], list)
        if sort:
            items.sort ()
        return items


    def _directory_command (self, command):
        """Add appropriate '--directory' to 'command' if needed.

        Return the resulting string.
        """
        if self._directory:
            command = command + (" --directory='%s' " % \
                                 self._protect (self._directory))
        return command
        
        
    def _get_problem (self, id):
        """Get raw text of the problem # 'id'.

        Return string.
        If the problem cannot be found or accessed, return 'None'.
        """
        command = self._directory_command ('query-pr')
        try:
            f = os.popen (command + " --full '%s'" % self._protect (str (id)))
            text = f.read ()
            if len (text) < 40:         # "query-pr: no PRs matched"
                return None
            return text
        except:
            return None


    def _parse_problem (self, text):
        """Parse problem 'text'.

        Return dictionary of items and their values suitable for creation of a
        'Problem' instance.
        """
        # Parse mail headers
        s = StringIO.StringIO (text)
        m = rfc822.Message (s)
        name = email = None
        for h in 'Sender', 'From', 'Reply-To':
            n, e = m.getaddr (h)
            if n:
                name = n
            if e:
                email = e
        dictionary = {'originator': name, 'email': email}
        # Parse body
        m.rewindbody ()
        lines = m.fp.readlines ()
        m.fp.close ()
        for l in range (len (lines)):
            line = lines[l]
            if line[0] == '>':
                tag, value = string.split (line[1:], ':', 1)
                tag = string.lower (tag)
                value = value[:-1]
                dictionary[tag] = value
            else:
                dictionary[tag] = dictionary[tag] + line
        # Return result
        return dictionary


    def _lock (self, id='', repeat=5, timeout=(2,5)):
        """Lock problem 'id'.

        Arguments:
        
         id -- The number of the problem, it can be either string or integer.
           If id is not specified or is an empty string, lock the whole
           database.
         repeat -- Number of retries on an error.
         timeout -- Time idle between retries in seconds.  It can be either
           non-negative integer or a pair of non-negative integers.  If the
           latter, generate random timeouts in the specified interval.

        Return an empty string, on repetitive error return string describing
        the problem.
        """
        # Compose locking command
        command = os.path.join (config.GNATS_PRG_DIR, 'pr-edit') + ' '
        command = self._directory_command (command)
        if id:
            command = command + "--lock=www '%s'" % self._protect (str (id))
        else:
            command = command + '--lockdb'
        # Lock
        while 1:
            p = popen2.Popen3 (command, 1)
            p.tochild.close ()
            p.fromchild.close ()
            desc = p.childerr.read ()
            p.childerr.close ()
            result = p.wait ()
            if result:
                if repeat > 0:
                    repeat = repeat - 1
                    if type (timeout) == type (0):
                        timeout_now = timeout
                    else:
                        timeout_now = whrandom.randint (timeout[0], timeout[1])
                    time.sleep (timeout_now)
                else:
                    return desc
            else:
                return ''


    def _unlock (self, id=''):
        """Unlock problem 'id'.

        'id' is the number of the problem, it can be either string or integer.
        If id is not specified or is an empty string, unlock the whole
        database.

        Return an empty string, on error return string describing the problem.
        """
        # Compose unlocking command
        command = os.path.join (config.GNATS_PRG_DIR, 'pr-edit') + ' '
        command = self._directory_command (command)
        if id:
            command = command + "--unlock '%s'" % self._protect (str (id))
        else:
            command = command + '--unlockdb'
        # Unlock
        p = popen2.Popen3 (command, 1)
        p.tochild.close ()
        p.fromchild.close ()
        desc = p.childerr.read ()
        p.childerr.close ()
        result = p.wait ()
        if result:
            return desc
        return ''


    # Public methods
    

    def date (self, time_tuple):
        """Return date in the GNATS form as a string.

        'time_tuple' is the Python standard time representation as a 9-tuple.
        """
        return time.strftime ('%a %b %d %H:%M:%S %Z %Y', time_tuple)


    def directory (self):
        """Return GNATS database directory of the instance.
        """
        return self._directory


    def site (self):
        """Return GNATS site of the instance.
        """
        return self._site


    def version (self):
        """Return GNATS version as a string, including name.
        """
        try:
            f = os.popen ('query-pr --version')
            version = f.read ()
            f.close ()
            version = string.split (version)
            version = version[1]
            return 'GNATS ' + version
        except:
            return 'GNATS ???'


    def categories (self):
        """Return a sorted list of all categories of the instance.

        If it cannot be retrieved, return 'None'.
        """
        command = 'send-pr -L'
        if self._site:
            command = "%s '%s'" % (command, self._protect (self._site))
        try:
            f = os.popen (command)
            f.readline ()               # ignore "Known categories:"
            raw = f.read ()
            result = string.split (raw)
            result.sort ()
            return result
        except:
            return None


    def categories_full (self):
        """Return full category listing for the instance directory.

        It is a list of 5-tuples
        (CATEGORY, CONFIDENTIAL, DESCRIPTION, RESPONSIBLE, NOTIFY), where the
        items mean:

          CATEGORY -- name of the category as a string
          CONFIDENTIAL -- whether the category is confidential (true/false)
          DESCRIPTION -- description of the category
          RESPONSIBLE -- ID of the responsible person
          NOTIFY -- list of email addresses to send notifications to

        Category is considered confidential, iff the first character of its
        description is '*'.

        Order of categories returned is the same as in the 'categories' file.

        If the list cannot be retrieved, return 'None'.
        """
        command = self._directory_command ('query-pr --list-categories')
        result = []
        try:
            f = os.popen (command)
            while 1:
                line = f.readline ()
                if not line:
                    break
                line = string.rstrip (line)
                line = string.split (line, ':')
                if line[1] and line[1][0] == '*':
                    confidential = 1
                    line[1] = line[1][1:]
                else:
                    confidential = 0
                if line[3]:
                    notify = string.split (line[3], ',')
                else:
                    notify = []
                result.append ((line[0], confidential, line[1], line[2],
                                notify))
        except:
            return None
        return result


    def classes (self):
        """Return list of all classes for the GNATS db.

        If it cannot be retrieved, return 'None'.
        """
        return self._extract_first (self.classes_full ())


    def classes_full (self):
        """Return a list of all information about responsible persons.

        The list contains lists of the form [CLASS, TYPE, DESCRIPTION].
        CLASS is the class name.
        TYPE is type of the class, by default 'class'.
        DESCRIPTION is description of the class.

        If the list cannot be retrieved, return 'None'.
        """
        command = self._directory_command ('query-pr --list-classes')
        try:
            items = []
            f = os.popen (command)
            while 1:
                line = f.readline ()
                if not line:
                    break
                if line[0] == '#':
                    continue
                line = string.rstrip (line)
                line = string.split (line, ':')
                if len (line) >= 3:
                    if not line[1]:
                        line[1] = 'class'
                    items.append (line)
            f.close ()
            return items
        except:
            return None        

            
    def responsible (self):
        """Return a sorted list of all responsible persons for the GNATS db.

        If it cannot be retrieved, return 'None'.
        """
        return self._extract_first (self.responsible_full (), 1)
    

    def responsible_full (self):
        """Return a list of all information about responsible persons.

        The list contains lists of the form [RESPONSIBLE, FULL_NAME, EMAIL].
        RESPONSIBLE is the responsible tag.
        FULL_NAME is the full name of the person.
        EMAIL is his e-mail (as stated in the 'responsible' file).

        If the list cannot be retrieved, return 'None'.
        """
        command = self._directory_command ('query-pr --list-responsible')
        try:
            items = []
            f = os.popen (command)
            while 1:
                line = f.readline ()
                if not line:
                    break
                if line[0] == '#':
                    continue
                line = string.rstrip (line)
                line = string.split (line, ':')
                if len (line) >= 3:
                    if not line[2]:
                        line[2] = line[0]
                    line[2] = mail.full_address (line[2])
                    items.append (line)
            f.close ()
            return items
        except:
            return None


    def states (self):
        """Return list of all states of GNATS db.
        """
        return self._extract_first (self.states_full ())


    def states_full (self):
        """Return a list of all information about responsible persons.

        The list contains lists of the form [STATE, TYPE, DESCRIPTION].
        STATE is the name of the state.
        TYPE is type of the state, by default `open'.
        DESCRIPTION is description of the state.

        If the list cannot be retrieved, return 'None'.
        """
        command = self._directory_command ('query-pr --list-states')
        try:
            items = []
            f = os.popen (command)
            while 1:
                line = f.readline ()
                if not line:
                    break
                if line[0] == '#':
                    continue
                line = string.rstrip (line)
                line = string.split (line, ':')
                if len (line) >= 3:
                    if not line[1]:
                        line[1] = 'open'
                    items.append (line)
            f.close ()
            return items
        except:
            return None

    
    def submitters (self):
        """Return sorted list of submitters.
        """
        return self._extract_first (self.submitters_full (), 1)
    

    def submitters_full (self):
        """Return a list of all information about submitters.

        Each list element is a list of the form
        [ID, DESCRIPTION, TYPE, RTIME, CONTACT, NOTIFY], where:

         - ID is submitter id
         - DESCRIPTION is its textual description (e.g. name)
         - TYPE is a type of the submitter in any form
         - RTIME is a response time for serious bugs in hours; if it is none,
           it is set to -1
         - CONTACT is id of the contact person
         - NOTIFY is a list of e-mail addresses to notify

        If the information about submitters cannot be retrieved, return 'None'.
        """
        command = self._directory_command ('query-pr --list-submitters')
        try:
            items = []
            f = os.popen (command)
            while 1:
                line = f.readline ()
                if not line:
                    break
                if line[0] == '#':
                    continue
                line = string.split (line, ':')
                if len (line) == 0:
                    continue
                if line[3]:
                    line[3] = string.atoi (line[3])
                else:
                    line[3] = -1
                line[5] = string.split (line[5], ',')
                items.append (line)    
            f.close ()
            return items
        except:
            return None


    def id2email (self, id):
        """Return a pair (NAME, EMAIL) for id.

        Search through responsible, if nothing is found, try submitters.
        
        If name or e-mail cannot be found, return an empty string instead of
        it.
        """
        items = self.responsible_full ()
        if items:
            for i in items:
                if i[0] == id:
                    name = string.strip (i[1])
                    email = string.strip (i[2])
                    return name, mail.full_address (email)
        items = self.submitters_full ()
        if items:
            for i in items:
                if i[0] == id:
                    name = string.strip (i[1])
                    email = string.strip (i[4])
                    return name, mail.full_address (email)
        return '', ''


    def administrator (self):
        """Return a pair (NAME, EMAIL) for GNATS administrator.

        If it cannot be found, return an empty string.
        """
        return self.id2email (mail.user (config.ADMIN_MAIL))

            
    def problem (self, id):
        """Retrieve problem # 'id' from the database.

        'id' can be a string or an integer.

        Return appropriate 'Problem' instance.
        If the problem is not found or is not accessible, return 'None'.
        """
        text = self._get_problem (id)
        if not text:
            return None
        dictionary = self._parse_problem (text)
        return problem.Problem (self, dictionary, id)


    def update_problem (self, problem):
        """Update 'problem' in database.

        'problem' is a 'Problem' instance.

        Return an empty string on success, error message otherwise.
        """
        id = problem.id ()
        command = os.path.join (config.GNATS_PRG_DIR, 'pr-edit')
        command = self._directory_command (command)
        command = "%s '%s'" % (command, self._protect (id))
        # Lock problem
        result = self._lock (id)
        if result:
            return desc
        # Update problem
        try:
            p = popen2.Popen3 (command, 1)
            p.tochild.write (self.dump (problem))
            p.tochild.close ()
            p.fromchild.close ()
            desc = p.childerr.read ()
            p.childerr.close ()
            result = p.wait ()
        finally:
            # Unlock problem
            self._unlock (id)
        if result:
            return desc
        return ''


    def check_problem (self, problem):
        """Check problem for correctness.

        If the problem is correct, return an empty string, otherwise return
        description of errors.
        """
        id = problem.id ()
        if id:
            result = self._lock (id)
            if result:
                return result
        try:
            command = os.path.join (config.GNATS_PRG_DIR, 'pr-edit') + ' -c'
            inp, outp = popen2.popen2 (self._directory_command (command))
            outp.write (self.dump (problem))
            outp.close ()
            result = inp.read ()
            inp.close ()
        finally:
            if id:
                self._unlock (id)
        return result


    def submit (self, problem):
        """Queue 'Problem' instance 'problem'.

        Return whether it succeeded.
        """
        command = os.path.join (config.GNATS_PRG_DIR, 'queue-pr')
        command = self._directory_command (command)
        command = command + ' -q'
        f = os.popen (command, 'w')
        f.write (self.dump (problem))
        result = f.close ()
        return not result


    def query (self, params):
        """Query database according to 'params'.

        'params' is a list of pairs (TAG, VALUE), where TAG is the queried tag
        and VALUE is its requested value.

        Tags can be from the following set: 'category', 'release',
        'responsible', 'confidential', 'class', 'severity', 'priority',
        'state', 'notclosed', 'originator', 'synopsis', 'text', 'multitext',
        'adatef', 'adatet', 'mdatef', 'mdatet', 'cdatef', 'cdatet'.

        Return a pair (QUERY, RESULT).
        QUERY is a text query string used.
        RESULT is a list of 'query-pr' output lines.
        On any error, RESULT is a string describing the error.
        """
        # Convert params to an (option, value) dictionary
        skip_closed = 0
        params_dict = {}
        for t, v in params:
            if not v:
                continue
            if t in ('category', 'release', 'responsible', 'confidential',
                     'class', 'severity', 'priority', 'state', 'originator',
                     'synopsis', 'text', 'multitext'):
                if params_dict.has_key (t):
                    params_dict[t] = params_dict[t] + '|' + v
                else:
                    params_dict[t] = v
            elif t == 'notclosed':
                skip_closed = 1
            elif t == 'adatef':
                params_dict['arrived-after'] = v
            elif t == 'adatet':
                params_dict['arrived-before'] = v
            elif t == 'mdatef':
                params_dict['modified-after'] = v
            elif t == 'mdatet':
                params_dict['modified-before'] = v
            elif t == 'cdatef':
                params_dict['closed-after'] = v
            elif t == 'cdatet':
                params_dict['closed-before'] = v
        # Build query
        if skip_closed:
            query = '--skip-closed '
        else:
            query = ''
        for k, v in params_dict.items ():
            query = query + "--%s='%s' " % (k, self._protect (v))
        if query == '':
            query = "--text='.'"
        # Perform query
        command = self._directory_command ('query-pr --summary ')
        try:
            f, dummy, err = popen2.popen3 (command + query)
            dummy.close ()
            result = f.readlines ()
            f.close ()
            errors = err.read ()
            err.close ()
        except Exception, instance:
            return query, str (instance)
        # Return result
        if errors and not result:
            return query, L(errors)
        output = []
        for r in result:
            output.append (string.split (r, None, 7))
        return query, output


    def dump (self, problem):
        """Dump 'Problem' instance 'problem' for GNATS input.

        Return dumped text.
        """
        items = problem.items ()
        # Mail headers
        if items.has_key ('originator'):
            from_ = '"%s" <%s>' % (items['originator'], items['email'])
        else:
            from_ = '%s' % items['email']
        to = 'GNATS WWW Submitting Form <%s>' % config.ADMIN_MAIL
        result = '''From: %s
To: %s
Subject: %s
Date: %s
Mime-Version: 1.0
Content-Type: text/plain; charset=%s

''' % (mail.mimify_header_value (from_),
       to,
       mail.mimify_header_value (items['synopsis']),
       mail.date (),
       L.charset ())
        # GNATS fields
        for f in self._GTAGS:
            result = result + '>%s:' % f
            try:
                value = items[string.lower (f)]
            except KeyError:
                value = ''
            if f in self._GTAGS_MULTILINE:
                linesplit = string.split (value, '\n')
                for i in range (len (linesplit)):
                    if linesplit[i] and linesplit[i][0] == '>':
                        linesplit[i] = ' ' + linesplit[i]
                value = string.join (linesplit, '\n')
                result = result + '\n%s' % value
                if not value or value[-1] != '\n':
                    result = result + '\n'
            else:
                result = result + '%s\n' % value
        # Return
        return result


    def update_trail (self, items, changes, changer,
                      reason_responsible, reason_state):
        """Return new audit trail as a string.

        Arguments:

          items -- problem dictionary of original values
          changes -- problem dictionary of udpated values
          changer -- identification of the person, who performed the changes
          reason_responsible -- reason for changing responsible
          reason_state -- reason for changing state
        """
        update = ''
        for k, v in changes.items ():
            kcap = string.split (k, '-')
            kcap = map (lambda s: string.upper (s[0]) + s[1:], kcap)
            kcap = string.join (kcap, '-')
            if k == 'responsible':
                update = update + \
                         'Responsible-Changed-<From>-<To>: %s>-<%s\n' % \
                         (items['responsible'], changes['responsible'])
                update = update + 'Responsible-Changed-Why:\n%s\n' % \
                         reason_responsible
            elif k == 'state':
                update = update + \
                         'State-Changed-<From>-<To>: %s>-<%s\n' % \
                         (items['state'], changes['state'])
                update = update + 'State-Changed-Why:\n%s\n' % reason_state
            update = update + '%s-Changed-By: %s\n' % (kcap, changer)
            update = update + '%s-Changed-When: %s\n' % \
                     (kcap, self.date (time.localtime (time.time ())))
        try:
            old = changes['audit-trail']
        except KeyError:
            old = ''
        if not old:
            try:
                old = items['audit-trail']
            except KeyError:
                pass
        if update:
            return update + '\n' + old
        else:
            return old
