# Copyright (C) 1999 Milan Zamazal
#
# COPYRIGHT NOTICE
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
# for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; see the file COPYING.  If not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.


"""Internationalization with respect to the language used.

It works as follows:

- Language operations are performed through instances of the class 'Language'.
  See its documentation for available methods.

- To allow setting a particular language once and then sharing it in all
  modules, there is provided the class 'Interface' and its instance 'L'.  To
  use it, just do 'from i18n import L' and apply 'L.set_language (lang_code)'
  once.  Then access 'Language' methods through its 'L' instance.

The module constant 'LANG_CODING' is a dictionary with language codes as a keys
and character set codings as values.  The item with an empty string as a key
defines the default value.
"""

__author__  = "Milan Zamazal <pdm@zamazal.org>"
__version__ = "$Id: i18n.py,v 1.7 2001/04/15 07:26:22 pdm Exp $"
__copyright__ = "GNU General Public License, version 2"


import os

from gnats2w import config


LANG_CODING = {'': 'US-ASCII',
               'en': 'US-ASCII',
               'cs': 'ISO-8859-2'
               }


class Language:
    """Message translator and other language dependent things.

    To use this class, create its instance for a particular language.  The
    class looks for translation dictionary with the name equal to the language
    code in the 'config.DICTIONARY_DIR' directory.  If it is found and
    successfully loaded, message translation is possible.

    The dictionary in the mentioned language file must be named 'dictionary'.
    It contains English messages as keys and translated message as values.
    Translation is performed by calling instance of the 'Language' class with
    the message as the only argument.  If the message is not found as a key in
    the dictionary, it is return without change, otherwise the corresponding
    dictionary value is return.

    See gettext manual for some advises about using string translation.

    The method 'language' returns language of the instance.

    The method 'charset' returns character set used for the language.
    """
    
    def __init__ (self, language):
        """Read dictionary for 'language'.

        'language' is an ISO language code, given simply as a string.

        On any error make an empty dictionary.
        """
        self._set_language (language)
        

    def _set_language (self, language):
        """Set new language of the instance.

        'language' can be either string (language code) or list of strings
        (language codes).  It sets the "best" language from available choices.
        """
        if not language or language == ['']:
            language = [config.LANGUAGE]
        if type (language) == type (''):
            language = [language]
        weight = -1
        self._language = ''
        self._dictionary = {}
        for l in language:
            if LANG_CODING.has_key (l):
                w = 1
            else:
                w = 0
            dictfile = os.path.join (config.DICTIONARY_DIR, l)
            try:
                if not l:
                    raise Exception ()
                globals = {}
                execfile (dictfile, globals)
                my_dictionary = globals['dictionary']
            except:
                my_dictionary = {}
            dictfile = os.path.join (config.DICTIONARY_DIR_LOCAL, l)
            try:
                globals = {}
                execfile (dictfile, globals)
                my_dictionary.update (globals['dictionary'])
            except:
                pass
            if l == 'en':
                w = 2
            elif my_dictionary.keys ():
                w = w + 1
            if w > weight:
                self._language = l
                self._dictionary = my_dictionary
                weight = w


    def language (self):
        """Return language of the instance.
        """
        return self._language
    

    def charset (self):
        """Return charset for the language.
        """
        try:
            return LANG_CODING[self._language]
        except KeyError:
            return LANG_CODING['']

        
    def __call__ (self, message):
        """Return message in language of the instance.

        If no item for 'message' and the language is present, return 'message'.
        """
        if self._dictionary.has_key (message):
            return self._dictionary[message]
        else:
            return message


class Interface (Language):
    """Language access for all modules.

    This is like 'Language', but provides public 'set_language' method.
    """

    def __init__ (self):
        """Initialize instance with default language.
        """
        Language.__init__ (self, config.LANGUAGE)


    def set_language (self, language):
        """Set 'language' as the current language.

        'language' can be either string (language code) or list of strings
        (language codes).  It sets the "best" language from available choices.
        """
        self._set_language (language)


L = Interface ()
