/*	Copyright (C) 1995 Free Software Foundation, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, the Free Software Foundation gives permission
 * for additional uses of the text contained in its release of GUILE.
 *
 * The exception is that, if you link the GUILE library with other files
 * to produce an executable, this does not by itself cause the
 * resulting executable to be covered by the GNU General Public License.
 * Your use of that executable is in no way restricted on account of
 * linking the GUILE library code into it.
 *
 * This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU General Public License.
 *
 * This exception applies only to the code released by the
 * Free Software Foundation under the name GUILE.  If you copy
 * code from other Free Software Foundation releases into a copy of
 * GUILE, as the General Public License permits, the exception does
 * not apply to the code that you add in this way.  To avoid misleading
 * anyone as to the status of such modified files, you must delete
 * this exception notice from them.
 *
 * If you write modifications of your own for GUILE, it is your choice
 * whether to permit this exception to apply to your modifications.
 * If you do not wish that, delete this exception notice.  
 */




#include <stdio.h>
#include <sys/param.h>
#include "gscm.h"
#include "_scm.h"



void scm_init_guile ();
static char version_string[] = "GNU Guile, version iii";


/* {Object Id's}
 *
 * An id is a name for an object.  By this interface, ids are explicitly
 * allocated and freed.  Any object can have any number of ids.
 * while an id is allocated, it protects the object to which it belongs.
 */

static SCM * indirects = 0;
static int free_indirect;
static SCM n_indirects;  /* Used as a C integer type...not as an SCM object */

#ifdef __STDC__
long
gscm_mk_objid (SCM obj)
#else
long
gscm_mk_objid (obj)
     SCM obj;
#endif
{
  if (indirects == 0)
    {
      DEFER_INTS;
      indirects = scm_mkarray (256, 0);
      if (!indirects)
	{
	  ALLOW_INTS;
	  return -1;
	}
      n_indirects = 256;
      {
	int x;
	for (x = 0; x < 256; ++x)
	  indirects[x] = MAKINUM (x + 1);
	free_indirect = 0;
	n_indirects = 256;
      }
      ALLOW_INTS;
    }

  if (free_indirect == n_indirects)
    {
      /* This sucks: */
      if ((2 * n_indirects) > MOST_POSITIVE_FIXNUM)
	return -1;
      {
	SCM * new_indirects;
	DEFER_INTS;
	new_indirects = scm_mkarray (2 * n_indirects, 0);
	if (!new_indirects)
	  return -1;
	scm_free_array (indirects);
	indirects = new_indirects;
	{
	  int x;
	  x = n_indirects;
	  n_indirects *= 2;
	  while (x < n_indirects)
	    indirects[x] = MAKINUM (x + 1);


	  bcopy (indirects, new_indirects, 2 * n_indirects);
	}
	ALLOW_INTS;
      }
    }
  {
    int id;
    id = free_indirect;
    free_indirect = INUM (indirects[id]);
    indirects[id] = obj;
    return id;
  }
}

#ifdef __STDC__
SCM
gscm_id2obj (long n)
#else
SCM
gscm_id2obj (n)
     long n;
#endif
{
  return indirects[n];
}

#ifdef __STDC__
void
gscm_free_id (long n)
#else
void
gscm_free_id (n)
     long n;
#endif
{
  indirects[n] = MAKINUM (free_indirect);
  free_indirect = n;
}

#ifdef __STDC__
void
gscm_id_reassign (long n, SCM obj)
#else
void
gscm_id_reassign (n, obj)
     long n;
     SCM obj;
#endif
{
  indirects[n] = obj;
}

PROC (s_sys_id, "%id", 1, 0, 0, gscm_sys_id);
#ifdef __STDC__
SCM
gscm_sys_id(SCM n)
#else
SCM
gscm_sys_id (n)
     SCM n;
#endif
{
  int cn;
  ASSERT (INUMP (n), n, ARG1, s_sys_id);
  cn = INUM (n);
  ASSERT (!((cn >= n_indirects) || (cn < 0)), n, OUTOFRANGE, s_sys_id);
  return indirects [n];
}



extern int scm_verbose;
int gscm_default_verbosity = 2;


PROC (s_sys_default_verbosity, "%default-verbosity", 0, 0, 0, gscm_sys_default_verbosity);
#ifdef __STDC__
SCM 
gscm_sys_default_verbosity (void)
#else
SCM 
gscm_dflt_verbosity ()
#endif
{
  return MAKINUM (gscm_default_verbosity);
}


#ifdef __STDC__
void
gscm_verbosity (int n)
#else
void
gscm_verbosity (n)
     int n;
#endif
{

  gscm_default_verbosity = n;
}

#ifdef __STDC__
void
gscm_with_verbosity (int n, void (*fn)P((void *)), void * data)
#else
void
gscm_with_verbosity (n, fn, data)
     int n;
     void (*fn)P((void *));
     void * data;
#endif
{
  int oldv;
  oldv = scm_verbose;
  scm_verbose = n;
  fn (data);
  scm_verbose = oldv;
}


/* {Initialization}
 */


/* Normally the default heap size is used (indicated by
 * passing 0 to scm_init_scm).  But applications can override 
 * this if they need to.
 */

static char init_file_name[MAXPATHLEN];
static int init_file_processed = 0;


static int init_heap_size = 0;  /* in units of 1024 bytes. */
#ifdef __STDC__
void
gscm_set_init_heap_size (int x)
#else
void
gscm_set_init_heap_size (x)
     int x;
#endif
{
  init_heap_size = x;
}

#ifdef __STDC__
int
gscm_init_heap_size (void)
#else
int
gscm_init_heap_size ()
#endif
{
  return init_heap_size;
}
extern SCM *scm_loc_tick_signal;

char *getenv ();
char * gscm_last_attempted_init_file = "<none>";

#ifdef __STDC__
GSCM_status
gscm_init_from_fn (char *initfile, int argc, char **argv, void (*init_fn) ())
#else
GSCM_status
gscm_init_from_fn (initfile, argc, argv, init_fn)
     char *initfile;
     int argc;
     char **argv;
     void (*init_fn) ();
#endif
{
  /* Init all the built-in parts of SCM. */
/*  scm_init_scm (scm_verbose, init_heap_size); */

  /* Save the argument list to be the return value of (program-arguments).
   */
  progargs = scm_makfromstrs (argc, argv);

  scm_exitval = MAKINUM (EXIT_SUCCESS);
  scm_errjmp_bad = 0;
  errno = 0;
  scm_alrm_deferred = 0;
  scm_sig_deferred = 0;
  scm_ints_disabled = 1;

#if 0
/* !!! */
#ifdef SIGALRM
  scm_make_subr (s_alarm, tc7_subr_1, "alarm");
#ifndef AMIGA
  scm_make_subr ("pause", tc7_subr_0, "pause");
#endif
#endif

#ifndef AMIGA
  scm_make_subr ("sleep", tc7_subr_1, "sleep");
#endif

  scm_make_subr ("raise", tc7_subr_1, "raise");
  
#ifdef TICKS
  scm_loc_tick_signal = &CDR (scm_sysintern ("ticks-interrupt", SCM_UNDEFINED));
  scm_make_subr ("ticks", tc7_subr_1o, "ticks");
#endif
#endif
  scm_init_variable();
  scm_init_gsubr();
  scm_init_kw();
  init_fn ();	/* call initialization of extensions files */
#ifdef DLD
  init_dynl ();
#else
#ifdef SUN_DL
  init_dynl ();
#endif
#endif

  if (initfile == NULL)
    {
      initfile = getenv ("GUILE_INIT_PATH");
      if (initfile == NULL)
	initfile = IMPLINIT;
    }

  if (initfile == NULL)
    {
      init_file_processed = 1;
      return GSCM_OK;
    }
  else
    {
      int verb;
      GSCM_status status;
      SCM answer;

      gscm_last_attempted_init_file = initfile;
      verb = scm_verbose;
      scm_verbose = -1;
      init_file_processed = 0;
      strncpy (init_file_name, initfile, MAXPATHLEN);
      status = gscm_seval_file (&answer, -1, initfile);
      if ((status == GSCM_OK) && (answer == BOOL_F))
	status = GSCM_ERROR_OPENING_INIT_FILE;
      scm_verbose = verb;
      return status;
    }
}

#ifdef __STDC__
void
gscm_take_stdin (void)
#else
void
gscm_take_stdin ()
#endif
{

  if (isatty(fileno(stdin))) setbuf(stdin, 0); /* turn off stdin buffering */
  scm_take_stdin = 1;
}

#ifdef __STDC__
void
gscm_verbose (int n)
#else
void
gscm_verbose (n)
     int n;
#endif
{
  scm_verbose = n;
}




/* {Managing Top Levels}
 */

struct seval_str_frame
{
  GSCM_status status;
  SCM * answer;
  GSCM_top_level top;
  char * str;
};

#ifdef __STDC__
static void
_seval_str_fn (void * vframe)
#else
static void
_seval_str_fn (vframe)
     void * vframe;
#endif
{
  struct seval_str_frame * frame;
  frame = (struct seval_str_frame *)vframe;
  frame->status = gscm_seval_str (frame->answer, frame->top, frame->str);
}


#ifdef __STDC__
GSCM_status
gscm_create_top_level (GSCM_top_level * answer)
#else
GSCM_status
gscm_create_top_level (answer)
     GSCM_top_level * answer;
#endif
{
  SCM it;
  GSCM_status stat;
  struct seval_str_frame frame;

  frame.str = "(gscm-create-top-level)";
  frame.top = -1;
  frame.answer = &it;
  gscm_with_verbosity (-1, _seval_str_fn, &frame);
  stat = frame.status;
  if (stat == GSCM_OK)
    *answer = (GSCM_top_level)gscm_mk_objid (it);
  return stat;
}

#ifdef __STDC__
GSCM_status
gscm_destroy_top_level (GSCM_top_level it)
#else
GSCM_status
gscm_destroy_top_level (it)
     GSCM_top_level it;
#endif
{
  char buf[300];
  GSCM_status stat;
  struct seval_str_frame frame;

  sprintf (buf, "(gscm-destroy-top-level (\%\%gscm-indirect %d))", it);
  frame.str = buf;
  frame.top = -1;
  frame.answer = 0;
  gscm_with_verbosity (-1, _seval_str_fn, &frame);
  stat = frame.status;
  return stat;
}


/* {Top Level Evaluation}
 * 
 * Top level evaluation has to establish a dynamic root context,
 * enable Scheme signal handlers, and catch global escapes (errors, quits,
 * aborts, restarts, and execs) from the interpreter.
 */

extern unsigned int scm_tick_count;
extern unsigned int scm_ticken;


/* {Printing Objects to Strings} 
 */

#ifdef __STDC__
static GSCM_status
gscm_portprint_obj (SCM port, SCM obj)
#else
static GSCM_status
gscm_portprint_obj (port, obj)
     SCM port;
     SCM obj;
#endif
{
  scm_iprin1 (obj, port, 1);
  return GSCM_OK;
}

#ifdef __STDC__
static GSCM_status
gscm_strprint_obj (SCM * answer, SCM obj)
#else
static GSCM_status
gscm_strprint_obj (answer, obj)
     SCM * answer;
     SCM obj;
#endif
{
  SCM str;
  SCM port;
  GSCM_status stat;
  str = scm_makstr (64, 0);
  port = scm_mkstrport (MAKINUM (0), str, OPN | WRTNG, "gscm_strprint_obj");
  stat = gscm_portprint_obj (port, obj);
  if (stat == GSCM_OK)
    *answer = str;
  else
    *answer = BOOL_F;
  return stat;
}

#ifdef __STDC__
static GSCM_status
gscm_cstr (char ** answer, SCM obj)
#else
static GSCM_status
gscm_cstr (answer, obj)
     char ** answer;
     SCM obj;
#endif
{
  SCM sstr;
  GSCM_status stat;

  *answer = (char *)malloc (LENGTH (sstr));
  stat = GSCM_OK;
  if (!*answer)
    stat = GSCM_OUT_OF_MEM;
  else
    bcopy (CHARS (sstr), *answer, LENGTH (sstr));
  return stat;
}
     

/* {Invoking The Interpreter}
 */

#ifdef _UNICOS
typedef int setjmp_type;
#else
typedef long setjmp_type;
#endif

extern SCM *scm_loc_loadpath;
extern long scm_linum;

#ifdef __STDC__
static GSCM_status
_eval_port (SCM * answer, GSCM_top_level toplvl, SCM port, int printp)
#else
static GSCM_status
_eval_port (answer, toplvl, port, printp)
     SCM * answer;
     GSCM_top_level toplvl;
     SCM port;
     int printp;
#endif
{
  SCM saved_inp;
  GSCM_status status;
  setjmp_type i;
  static int deja_vu = 0;
  SCM ignored;

  if (deja_vu)
    return GSCM_ILLEGALLY_REENTERED;

  ++deja_vu;
  /* Take over signal handlers for all the interesting signals.
   */
  scm_init_signals ();


  /* Default return values:
   */
  if (!answer)
    answer = &ignored;
  status = GSCM_OK;
  *answer = BOOL_F;

  /* Perform evalutation under a new dynamic root.
   *
   */
  BASE (rootcont) = (STACKITEM *) & i;
  saved_inp = cur_inp;
  i = setjmp (JMPBUF (rootcont));
  cur_inp = saved_inp;
 drloop:
  switch ((int) i)
    {
    default:
      {
	char *name;
	name = scm_errmsgs[i - WNA].s_response;
	if (name)
	  {
	    SCM proc;
	    proc = CDR (scm_intern (name, (sizet) strlen (name)));
	    if (NIMP (proc))
	      scm_apply (proc, EOL, EOL);
	  }
	if ((i = scm_errmsgs[i - WNA].parent_err))
	  goto drloop;
	def_err_response ();
	goto leave;
      }

    case 0:
      scm_exitval = MAKINUM (EXIT_SUCCESS);
      scm_errjmp_bad = 0;
      errno = 0;
      scm_alrm_deferred = 0;
      scm_sig_deferred = 0;
      scm_ints_disabled = 0;

    case -2:
      scm_alrm_deferred = 0;
      scm_sig_deferred = 0;
      scm_errjmp_bad = 0;
      scm_ints_disabled = 0;
      /* need to close loading files here. */
      cur_inp = port;
      *scm_loc_loadpath = BOOL_F;

      {
	SCM top_env;
	top_env = (toplvl == -1
		   ? EOL
		   : gscm_id2obj (toplvl));
	*answer = scm_repl (nullstr, top_env);
      }
      cur_inp = saved_inp;
      if (printp)
	status = gscm_strprint_obj (answer, *answer);
      goto return_fixing_signals;

    case -1:
      status = GSCM_QUIT;
      goto leave;

    case -3:
      status = GSCM_RESTART;
      goto leave;
    }
 leave:
  scm_alrm_deferred = 0;
  scm_sig_deferred = 0;

 return_fixing_signals:
  scm_errjmp_bad = 1;
  scm_ints_disabled = 1;
  scm_restore_signals ();
#ifdef TICKS
  scm_ticken = 0;
#endif
  --deja_vu;
  return status;
}

#ifdef __STDC__
static GSCM_status
seval_str (SCM *answer, GSCM_top_level toplvl, char * str)
#else
static GSCM_status
seval_str (answer, toplvl, str)
     SCM *answer;
     GSCM_top_level toplvl;
     char * str;
#endif
{
  SCM scheme_str;
  SCM port;
  SCM oloadpath;
  long olninum;
  GSCM_status status;

  oloadpath = *scm_loc_loadpath;
  olninum = scm_linum;
  scheme_str = scm_makfromstr (str, strlen (str), 0);
  *scm_loc_loadpath = makfrom0str ("(no input file)");
  scm_linum = 1;
  port = scm_mkstrport (MAKINUM (0), scheme_str, OPN | RDNG, "gscm_seval_str");
  status = _eval_port (answer, toplvl, port, 0);
  scm_linum = olninum;
  *scm_loc_loadpath = oloadpath;
  return status;
}


extern STACKITEM * scm_stack_base;

#ifdef __STDC__
GSCM_status
gscm_seval_str (SCM *answer, GSCM_top_level toplvl, char * str)
#else
GSCM_status
gscm_seval_str (answer, toplvl, str)
     SCM *answer;
     GSCM_top_level toplvl;
     char * str;
#endif
{
  STACKITEM i;
  GSCM_status status;
  scm_stack_base = &i;
  status = seval_str (answer, toplvl, str);
  scm_stack_base = 0;
  return status;
}

#ifdef __STDC__
void
format_load_command (char * buf, char *file_name)
#else
void
format_load_command (buf, file_name)
     char * buf;
     char *file_name;
#endif
{
  char quoted_name[MAXPATHLEN + 1];
  int source;
  int dest;

  for (source = dest = 0; file_name[source]; ++source)
    {
      if (file_name[source] == '"')
	quoted_name[dest++] = '\\';
      quoted_name[dest++] = file_name[source];
    }
  quoted_name[dest] = 0;
  sprintf (buf, "(try-load \"%s\")", quoted_name);
}

#ifdef __STDC__
GSCM_status
gscm_seval_file (SCM *answer, GSCM_top_level toplvl, char * file_name)
#else
GSCM_status
gscm_seval_file (answer, toplvl, file_name)
     SCM *answer;
     GSCM_top_level toplvl;
     char * file_name;
#endif
{
  char command[MAXPATHLEN * 3];
  format_load_command (command, file_name);
  return gscm_seval_str (answer, toplvl, command);
}


#ifdef __STDC__
static GSCM_status
eval_str (char ** answer, GSCM_top_level toplvl, char * str)
#else
static GSCM_status
eval_str (answer, toplvl, str)
     char ** answer;
     GSCM_top_level toplvl;
     char * str;
#endif
{
  SCM sanswer;
  SCM scheme_str;
  SCM port;
  GSCM_status status;
  SCM oloadpath;
  long olninum;

  oloadpath = *scm_loc_loadpath;
  olninum = scm_linum;
  scheme_str = scm_makfromstr (str, strlen (str), 0);
  *scm_loc_loadpath = makfrom0str ("(no input file)");
  scm_linum = 1;
  port = scm_mkstrport (MAKINUM(0), scheme_str, OPN | RDNG, "gscm_eval_str");
  status = _eval_port (&sanswer, toplvl, port, 1);
  if (answer)
    {
      if (status == GSCM_OK)
	status = gscm_cstr (answer, sanswer);
      else
	*answer = 0;
    }
  scm_linum = olninum;
  *scm_loc_loadpath = oloadpath;
  return status;
}


#ifdef __STDC__
GSCM_status
gscm_eval_str (char ** answer, GSCM_top_level toplvl, char * str)
#else
GSCM_status
gscm_eval_str (answer, toplvl, str)
     char ** answer;
     GSCM_top_level toplvl;
     char * str;
#endif
{
  STACKITEM i;
  GSCM_status status;
  scm_stack_base = &i;
  status = eval_str (answer, toplvl, str);
  scm_stack_base = 0;
  return status;
}


#ifdef __STDC__
GSCM_status
gscm_eval_file (char ** answer, GSCM_top_level toplvl, char * file_name)
#else
GSCM_status
gscm_eval_file (answer, toplvl, file_name)
     char ** answer;
     GSCM_top_level toplvl;
     char * file_name;
#endif
{
  char command[MAXPATHLEN * 3];
  format_load_command (command, file_name);
  return gscm_eval_str (answer, toplvl, command);
}




/* {Error Messages}
 */


#ifdef __GNUC__
# define AT(X)  [X] =
#else
# define AT(X)
#endif 

static char * gscm_error_msgs[] =
{
  AT(GSCM_OK) "No error.",
  AT(GSCM_QUIT) "QUIT executed.",
  AT(GSCM_RESTART) "RESTART executed.",
  AT(GSCM_ILLEGALLY_REENTERED) "Gscm function was illegally reentered.",
  AT(GSCM_OUT_OF_MEM) "Out of memory.",
  AT(GSCM_ERROR_OPENING_FILE) "Error opening file.",
  AT(GSCM_ERROR_OPENING_INIT_FILE) "Error opening init file."
};

#ifdef __STDC__
char *
gscm_error_msg (int n)
#else
char *
gscm_error_msg (n)
     int n;
#endif
{
  if ((n < 0) || (n > (sizeof (gscm_error_msgs) / sizeof (char *))))
    return "Unrecognized error.";
  else
    return gscm_error_msgs[n];
}



/* {Defining New Procedures}
 */

#ifdef __STDC__
void
gscm_define_procedure (char * name, SCM (*fn)(), int req, int opt, int varp, char * doc)
#else
void
gscm_define_procedure (name, fn, req, opt, varp, doc)
     char * name;
     SCM (*fn)();
     int req;
     int opt;
     int varp;
     char * doc;
#endif
{
  scm_make_gsubr (name, req, opt, varp, fn);
}

#ifdef __STDC__
SCM
gscm_make_subr (SCM (*fn)(), int req, int opt, int varp, char * doc)
#else
SCM
gscm_make_subr (fn, req, opt, varp, doc)
     SCM (*fn)();
     int req;
     int opt;
     int varp;
     char * doc;
#endif
{
  return scm_make_gsubr ("*anonymous*", req, opt, varp, fn);
}

#define CURRY_PROC(cclo) (VELTS(cclo)[1])
#define CURRY_ARG1(cclo) (VELTS(cclo)[2])
static SCM curry_apply_fn;

#ifdef __STDC__
static SCM 
curry_apply (SCM self, SCM rest)
#else
static SCM 
curry_apply (self, rest)
     SCM self;
     SCM rest;
#endif
{
  return scm_apply (CURRY_PROC (self),
		    scm_cons (CURRY_ARG1 (self), rest),
		    EOL);
}

#ifdef __STDC__
SCM
gscm_curry (SCM procedure, SCM first_arg)
#else
SCM
gscm_curry (procedure, first_arg)
     SCM procedure;
     SCM first_arg;
#endif
{
  SCM answer;

  answer = scm_makcclo (curry_apply_fn, 3L);
  CURRY_ARG1(answer) = first_arg;
  CURRY_PROC(answer) = procedure;
  return answer;
}


#ifdef __STDC__
int
gscm_2_char (SCM c)
#else
int
gscm_2_char (c)
     SCM c;
#endif
{
  ASSERT (ICHRP (c), c, ARG1, "gscm_2_char");
  return ICHR (c);
}



#ifdef __STDC__
void
gscm_2_str (char ** out, int * len_out, SCM * objp)
#else
void
gscm_2_str (out, len_out, objp)
     char ** out;
     int * len_out;
     SCM * objp;
#endif
{
  ASSERT (NIMP (*objp) && STRINGP (*objp), *objp, ARG3, "gscm_2_str");
  if (out)
    *out = CHARS (*objp);
  if (len_out)
    *len_out = LENGTH (*objp);
}


#ifdef __STDC__
void
gscm_error (char * message, SCM args)
#else
void
gscm_error (message, args)
     char * message;
     SCM args;
#endif
{
  SCM errfn;
  SCM str;

  errfn = CDR (scm_intern ("error", 5));
  str = makfrom0str (message);
  scm_apply (errfn, scm_cons (str, args), EOL);
}


#define GSCM_SET_SIZE(OBJ, SIZE)	(CAR(OBJ) = (((SIZE) << 16) | tc16_gscm_obj))
#define GSCM_SIZE(OBJ)		((CAR (OBJ) >> 16) & 0x7f)
#define GSCM_MEM(OBJ)		((struct gscm_type **)CDR(OBJ))
#define GSCM_UMEM(OBJ)		((char *)(1 + GSCM_MEM(OBJ)))
#define GSCM_UTYPE(OBJ)		(* GSCM_MEM(OBJ))

#ifdef __STDC__
static SCM
mark_gscm (SCM obj)
#else
static SCM
mark_gscm (obj)
     SCM obj;
#endif
{
  if (!GC8MARKP (obj))
    {
      STACKITEM * start;
      sizet size;

      SETGC8MARK (obj);
      start = (STACKITEM *)GSCM_UMEM (obj);
      size = ((GSCM_SIZE (obj) - sizeof (void *)) / sizeof (*start));
      scm_mark_locations (start, size);
    }
  return BOOL_F;
}

#ifdef __STDC__
static sizet
free_gscm (SCM obj)
#else
static sizet
free_gscm (obj)
     SCM obj;
#endif
{
  struct gscm_type * type;

  type = GSCM_UTYPE (obj);
  if (type->die)
    type->die (obj);
  {
    int size;
    size = GSCM_SIZE (obj);
    scm_must_free ((char *)GSCM_MEM (obj));
    return size;
  }
}

#ifdef __STDC__
static int
print_gscm (SCM exp, SCM port, int writingp)
#else
static int
print_gscm (exp, port, writingp)
     SCM exp;
     SCM port;
     int writingp;
#endif
{
  struct gscm_type * type;

  type = GSCM_UTYPE (exp);
  if (   !type->print
      || !(type->print (exp, port, writingp)))
    {
      scm_lputs ("#<", port);
      scm_lputs (type->name ? type->name : "unknown", port);
      scm_putc (' ', port);
      scm_intprint (exp, 16, port);
      scm_putc ('>', port);
    }
  return 1;
}

#ifdef __STDC__
static SCM
equal_gscm (SCM a, SCM b)
#else
static SCM
equal_gscm (a, b)
     SCM a;
     SCM b;
#endif
{
  struct gscm_type * type;

  if (a == b)
    return BOOL_T;

  type = GSCM_UTYPE (a);
  if (type != GSCM_UTYPE (b))
    return BOOL_F;

  if (type->equal)
    return (type->equal (a, b) ? BOOL_T: BOOL_F);
  else
    return BOOL_F;
}


static int tc16_gscm_obj;
static struct scm_smobfuns gscm_obj_smob
= { mark_gscm, free_gscm, print_gscm, equal_gscm };

#ifdef __STDC__
SCM
gscm_alloc (struct gscm_type * type, int size)
#else
SCM
gscm_alloc (type, size)
     struct gscm_type * type;
     int size;
#endif
{
  SCM answer;
  char * mem;
  
  size = 1 + ((size + sizeof (void *) - 1) / sizeof (void *));
  size *= sizeof (void *);

  NEWCELL (answer);
  DEFER_INTS;
  mem = (char *)scm_must_malloc (size, type->name);
  bzero (mem, size);
  CDR (answer) = (SCM)mem;
  GSCM_UTYPE (answer) = type;
  GSCM_SET_SIZE (answer, size);
  ALLOW_INTS;
  return answer;
}
     
#ifdef __STDC__
char *
gscm_unwrap_obj (struct gscm_type * type, SCM * objp)
#else
char *
gscm_unwrap_obj (type, objp)
     struct gscm_type * type;
     SCM * objp;
#endif
{
  SCM obj;
  obj = *objp;
  ASSERT (   NIMP (obj)
	  && (TYP16 (obj) == tc16_gscm_obj)
	  && (type == GSCM_UTYPE (obj)),
	  obj, ARG2, "gscm_unwrap_obj");

  return GSCM_UMEM (obj);
}

#ifdef __STDC__
struct gscm_type * 
gscm_get_type (SCM * objp)
#else
struct gscm_type * 
gscm_get_type (objp)
     SCM * objp;
#endif
{
  SCM obj;
  obj = *objp;
  ASSERT (   NIMP (obj)
	  && (TYP16 (obj) == tc16_gscm_obj),
	  obj, ARG1, "gscm_get_type");

  return GSCM_UTYPE (obj);
}





static SCM
scm_stand_in_proc (proc)
     SCM proc;
{
  SCM answer;
  answer = scm_assoc (proc, scm_stand_in_procs);
  if (answer == BOOL_F)
    {
      answer = scm_closure (scm_listify (EOL, BOOL_F, SCM_UNDEFINED),
			    EOL);
      scm_stand_in_procs = scm_cons (scm_cons (proc, answer),
				     scm_stand_in_procs);
    }
  else
    answer = CDR (answer);
  return answer;
}

PROC (s_procedure_properties, "procedure-properties", 1, 0, 0, gscm_procedure_properties);
#ifdef __STDC__
SCM
gscm_procedure_properties (SCM proc)
#else
SCM
gscm_procedure_properties (proc)
     SCM proc;
#endif
{
  ASSERT (scm_procedure_p (proc), proc, ARG1, s_procedure_properties);
  if (!(NIMP (proc) && CLOSUREP (proc)))
    proc = scm_stand_in_proc (proc);
  return PROCPROPS (proc);
}

PROC (s_set_procedure_properties_x, "set-procedure-properties!", 2, 0, 0, gscm_set_procedure_properties_x);
#ifdef __STDC__
SCM
gscm_set_procedure_properties_x (SCM proc, SCM new)
#else
SCM
gscm_set_procedure_properties_x (proc, new)
     SCM proc;
     SCM new;
#endif
{
  if (!(NIMP (proc) && CLOSUREP (proc)))
    proc = scm_stand_in_proc (proc);
  ASSERT (NIMP (proc) && CLOSUREP (proc), proc, ARG1, s_set_procedure_properties_x);
  PROCPROPS (proc) = new;
  return UNSPECIFIED;
}


PROC (s_procedure_assoc, "procedure-assoc", 2, 0, 0, gscm_procedure_assoc);
#ifdef __STDC__
SCM 
gscm_procedure_assoc (SCM p, SCM k)
#else
SCM 
gscm_procedure_assoc (p, k)
     SCM p;
     SCM k;
#endif
{
  if (!(NIMP (p) && CLOSUREP (p)))
    p = scm_stand_in_proc (p);
  ASSERT (scm_procedure_p (p), p, ARG1, s_procedure_assoc);
  return scm_assoc (k, PROCPROPS (p));
}

PROC (s_procedure_property, "procedure-property", 2, 0, 0, gscm_procedure_property);
#ifdef __STDC__
SCM
gscm_procedure_property (SCM p, SCM k)
#else
SCM
gscm_procedure_property (p, k)
     SCM p;
     SCM k;
#endif
{
  SCM assoc;
  if (!(NIMP (p) && CLOSUREP (p)))
    p = scm_stand_in_proc (p);
  ASSERT (scm_procedure_p (p), p, ARG1, s_procedure_property);
  assoc = scm_assoc (k, PROCPROPS (p));
  return (NIMP (assoc) ? CDR (assoc) : BOOL_F);
}

PROC (s_set_procedure_property_x, "set-procedure-property!", 3, 0, 0, gscm_set_procedure_property_x);
#ifdef __STDC__
SCM
gscm_set_procedure_property_x (SCM p, SCM k, SCM v)
#else
SCM
gscm_set_procedure_property_x (p, k, v)
     SCM p;
     SCM k;
     SCM v;
#endif
{
  SCM assoc;
  if (!(NIMP (p) && CLOSUREP (p)))
    p = scm_stand_in_proc (p);
  ASSERT (NIMP (p) && CLOSUREP (p), p, ARG1, s_set_procedure_property_x);
  assoc = scm_assoc (k, PROCPROPS (p));
  if (NIMP (assoc))
    SETCDR (assoc, v);
  else
    PROCPROPS (p) = scm_acons (k, v, PROCPROPS (p));
  return UNSPECIFIED;
}


#ifdef __STDC__
GSCM_status
guile_ks (void)
#else
GSCM_status
guile_ks ()
#endif
{
  return 0;
}


#ifdef __STDC__
GSCM_status
gscm_run_scm (int argc, char ** argv, FILE * in, FILE * out, FILE * err, GSCM_status (*initfn)(), char * initfile, char * initcmd)
#else
GSCM_status
gscm_run_scm (argc, argv, in, out, err, initfn, initfile, initcmd)
     int argc;
     char ** argv;
     FILE * in;
     FILE * out;
     FILE * err;
     GSCM_status (*initfn)();
     char * initfile;
     char * initcmd;
#endif
{
  SCM_STACKITEM i;
  GSCM_status status;
  GSCM_top_level top;

  scm_ports_prehistory ();
  scm_smob_prehistory ();
  scm_tables_prehistory ();
  scm_init_storage (&i, init_heap_size, in, out, err);	/* BASE (rootcont) gets set here */
  scm_init_gsubr ();
  scm_init_arbiters ();
  scm_init_boolean ();
  scm_init_chars ();
  scm_init_continuations ();
  scm_init_dynwind ();
  scm_init_eq ();
  scm_init_error ();
  scm_init_feature ();
  scm_init_fports ();
  scm_init_files ();
  scm_init_gc ();
  scm_init_hash ();
  scm_init_kw ();
  scm_init_lvectors ();
  scm_init_numbers ();
  scm_init_pairs ();
  scm_init_ports ();
  scm_init_procs ();
  scm_init_record ();
  scm_init_repl (gscm_default_verbosity);
  scm_init_scmsigs ();
  scm_init_stackchk ();
  scm_init_strports ();
  scm_init_struct ();
  scm_init_symbols ();
  scm_init_time ();
  scm_init_strings ();
  scm_init_strop ();
  scm_init_throw ();
  scm_init_variable ();
  scm_init_vectors ();
  scm_init_vports ();
  scm_init_eval ();
  scm_init_ramap ();
  scm_init_unif ();
  scm_init_simpos ();
  scm_init_guile ();
  initfn ();

  /* Save the argument list to be the return value of (program-arguments).
   */
  progargs = scm_makfromstrs (argc, argv);

  scm_exitval = MAKINUM (EXIT_SUCCESS);
  scm_errjmp_bad = 0;
  errno = 0;
  scm_alrm_deferred = 0;
  scm_sig_deferred = 0;
  scm_ints_disabled = 1;

  if (initfile == NULL)
    {
      initfile = getenv ("SCM_INIT_PATH");
      if (initfile == NULL)
	initfile = IMPLINIT;
    }

  if (initfile == NULL)
    {
      init_file_processed = 1;
      status = GSCM_OK;
    }
  else
    {
      int verb;
      SCM answer;

      gscm_last_attempted_init_file = initfile;
      verb = scm_verbose;
      scm_verbose = -1;
      init_file_processed = 0;
      strncpy (init_file_name, initfile, MAXPATHLEN);
      status = gscm_seval_file (&answer, -1, initfile);
      if ((status == GSCM_OK) && (answer == BOOL_F))
	status = GSCM_ERROR_OPENING_INIT_FILE;
      scm_verbose = verb;
    }

  if (status == GSCM_OK)
    status = gscm_create_top_level (&top);

  if (status == GSCM_OK)
    {
      scm_verbose = -1;
      status = gscm_seval_str (0, top, initcmd);
    }
  return status;
}



#ifdef __STDC__
SCM
gscm_malloc_2_uve (int type, int k, int size, char * data)
#else
SCM
gscm_malloc_2_uve (type, k, size, data)
     int type;
     int k;
     int size;
     char * data;
#endif
{
  SCM v;
  NEWCELL (v);
  DEFER_INTS;
  scm_mallocated += size;
  SETCHARS (v, data);
  SETLENGTH (v, (k < LENGTH_MAX ? k : LENGTH_MAX), type);
  ALLOW_INTS;
  return v;
}




#ifdef __STDC__
int
gscm_is_gscm_obj (SCM obj)
#else
int
gscm_is_gscm_obj (obj)
     SCM obj;
#endif
{
  return (NIMP (obj) && TYP16 (obj) == tc16_gscm_obj);
}





void
scm_init_guile ()
{
  curry_apply_fn = scm_make_gsubr (" curry-apply", 0, 0, 1, curry_apply);
  tc16_gscm_obj = scm_newsmob (&gscm_obj_smob);
#include "gscm.x"
}

