/*	Copyright (C) 1995 Free Software Foundation, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, the Free Software Foundation gives permission
 * for additional uses of the text contained in its release of GUILE.
 *
 * The exception is that, if you link the GUILE library with other files
 * to produce an executable, this does not by itself cause the
 * resulting executable to be covered by the GNU General Public License.
 * Your use of that executable is in no way restricted on account of
 * linking the GUILE library code into it.
 *
 * This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU General Public License.
 *
 * This exception applies only to the code released by the
 * Free Software Foundation under the name GUILE.  If you copy
 * code from other Free Software Foundation releases into a copy of
 * GUILE, as the General Public License permits, the exception does
 * not apply to the code that you add in this way.  To avoid misleading
 * anyone as to the status of such modified files, you must delete
 * this exception notice from them.
 *
 * If you write modifications of your own for GUILE, it is your choice
 * whether to permit this exception to apply to your modifications.
 * If you do not wish that, delete this exception notice.  
 */


#include <stdio.h>
#include <math.h>
#include "_scm.h"







PROC (s_exact_p, "exact?", 1, 0, 0, scm_exact_p);
PROC (s_integer_p, "integer?", 1, 0, 0, scm_exact_p);
#ifdef __STDC__
SCM
scm_exact_p(SCM x)
#else
SCM
scm_exact_p(x)
     SCM x;
#endif
{
  if INUMP(x) return BOOL_T;
#ifdef BIGDIG
  if (NIMP(x) && BIGP(x)) return BOOL_T;
#endif
  return BOOL_F;
}

PROC (s_odd_p, "odd?", 1, 0, 0, scm_odd_p);
#ifdef __STDC__
SCM
scm_odd_p(SCM n)
#else
SCM
scm_odd_p(n)
     SCM n;
#endif
{
#ifdef BIGDIG
  if NINUMP(n) {
    ASSERT(NIMP(n) && BIGP(n), n, ARG1, s_odd_p);
    return (1 & BDIGITS(n)[0]) ? BOOL_T : BOOL_F;
  }
#else
  ASSERT(INUMP(n), n, ARG1, s_odd_p);
#endif
  return (4 & (int)n) ? BOOL_T : BOOL_F;
}

PROC (s_even_p, "even?", 1, 0, 0, scm_even_p);
#ifdef __STDC__
SCM
scm_even_p(SCM n)
#else
SCM
scm_even_p(n)
     SCM n;
#endif
{
#ifdef BIGDIG
  if NINUMP(n) {
    ASSERT(NIMP(n) && BIGP(n), n, ARG1, s_even_p);
    return (1 & BDIGITS(n)[0]) ? BOOL_F : BOOL_T;
  }
#else
  ASSERT(INUMP(n), n, ARG1, s_even_p);
#endif
  return (4 & (int)n) ? BOOL_F : BOOL_T;
}

PROC (s_abs, "abs", 1, 0, 0, scm_abs);
#ifdef __STDC__
SCM
scm_abs(SCM x)
#else
SCM
scm_abs(x)
     SCM x;
#endif
{
#ifdef BIGDIG
  if NINUMP(x) {
    ASSERT(NIMP(x) && BIGP(x), x, ARG1, s_abs);
    if (TYP16(x)==tc16_bigpos) return x;
    return scm_copybig(x, 0);
  }
#else
  ASSERT(INUMP(x), x, ARG1, s_abs);
#endif
  if (INUM(x) >= 0) return x;
  x = -INUM(x);
  if (!POSFIXABLE(x))
#ifdef BIGDIG
    return scm_long2big(x);
#else
  scm_wta(MAKINUM(-x), (char *)OVFLOW, s_abs);
#endif
  return MAKINUM(x);
}

PROC (s_quotient, "quotient", 2, 0, 0, scm_quotient);
#ifdef __STDC__
SCM
scm_quotient(SCM x, SCM y)
#else
SCM
scm_quotient(x, y)
     SCM x;
     SCM y;
#endif
{
  register long z;
#ifdef BIGDIG
  if NINUMP(x) {
    long w;
    ASSERT(NIMP(x) && BIGP(x), x, ARG1, s_quotient);
    if NINUMP(y) {
      ASRTGO(NIMP(y) && BIGP(y), bady);
      return scm_divbigbig(BDIGITS(x),
			   NUMDIGS(x),
			   BDIGITS(y),
			   NUMDIGS(y),
			   BIGSIGN(x) ^ BIGSIGN(y),
			   2);
    }
    z = INUM(y);
    ASRTGO(z, ov);
    if (1==z) return x;
    if (z < 0) z = -z;
    if (z < BIGRAD) {
      w = scm_copybig(x, BIGSIGN(x) ? (y>0) : (y<0));
      scm_divbigdig(BDIGITS(w), NUMDIGS(w), (BIGDIG)z);
      return scm_normbig(w);
    }
#ifndef DIGSTOOBIG
    w = scm_pseudolong(z);
    return scm_divbigbig(BDIGITS(x), NUMDIGS(x), (BIGDIG *)&w, DIGSPERLONG,
			 BIGSIGN(x) ? (y>0) : (y<0), 2);
#else
    { BIGDIG zdigs[DIGSPERLONG];
      scm_longdigs(z, zdigs);
      return scm_divbigbig(BDIGITS(x), NUMDIGS(x), zdigs, DIGSPERLONG,
			   BIGSIGN(x) ? (y>0) : (y<0), 2);
    }
#endif
  }
  if NINUMP(y) {
# ifndef RECKLESS
    if (!(NIMP(y) && BIGP(y)))
    bady: scm_wta(y, (char *)ARG2, s_quotient);
# endif
    return INUM0;
  }
#else
  ASSERT(INUMP(x), x, ARG1, s_quotient);
  ASSERT(INUMP(y), y, ARG2, s_quotient);
#endif
  if ((z = INUM(y))==0)
    ov: scm_wta(y, (char *)OVFLOW, s_quotient);
  z = INUM(x)/z;
#ifdef BADIVSGNS
  {
#if (__TURBOC__==1)
    long t = ((y<0) ? -INUM(x) : INUM(x))%INUM(y);
#else
    long t = INUM(x)%INUM(y);
#endif
    if (t==0) ;
    else if (t < 0)
      if (x < 0) ;
      else z--;
    else if (x < 0) z++;
  }
#endif
  if (!FIXABLE(z))
#ifdef BIGDIG
    return scm_long2big(z);
#else
  scm_wta(x, (char *)OVFLOW, s_quotient);
#endif
  return MAKINUM(z);
}

PROC (s_remainder, "remainder", 2, 0, 0, scm_remainder);
#ifdef __STDC__
SCM
scm_remainder(SCM x, SCM y)
#else
SCM
scm_remainder(x, y)
     SCM x;
     SCM y;
#endif
{
  register long z;
#ifdef BIGDIG
  if NINUMP(x) {
    ASSERT(NIMP(x) && BIGP(x), x, ARG1, s_remainder);
    if NINUMP(y) {
      ASRTGO(NIMP(y) && BIGP(y), bady);
      return scm_divbigbig(BDIGITS(x), NUMDIGS(x), BDIGITS(y), NUMDIGS(y),
			   BIGSIGN(x), 0);
    }
    if (!(z = INUM(y))) goto ov;
    return scm_divbigint(x, z, BIGSIGN(x), 0);
  }
  if NINUMP(y) {
# ifndef RECKLESS
    if (!(NIMP(y) && BIGP(y)))
    bady: scm_wta(y, (char *)ARG2, s_remainder);
# endif
    return x;
  }
#else
  ASSERT(INUMP(x), x, ARG1, s_remainder);
  ASSERT(INUMP(y), y, ARG2, s_remainder);
#endif
  if (!(z = INUM(y)))
    ov: scm_wta(y, (char *)OVFLOW, s_remainder);
#if (__TURBOC__==1)
  if (z < 0) z = -z;
#endif
  z = INUM(x)%z;
#ifdef BADIVSGNS
  if (!z) ;
  else if (z < 0)
    if (x < 0) ;
    else z += INUM(y);
  else if (x < 0) z -= INUM(y);
#endif
  return MAKINUM(z);
}

PROC (s_modulo, "modulo", 2, 0, 0, scm_modulo);
#ifdef __STDC__
SCM
scm_modulo(SCM x, SCM y)
#else
SCM
scm_modulo(x, y)
     SCM x;
     SCM y;
#endif
{
  register long yy, z;
#ifdef BIGDIG
  if NINUMP(x) {
    ASSERT(NIMP(x) && BIGP(x), x, ARG1, s_modulo);
    if NINUMP(y) {
      ASRTGO(NIMP(y) && BIGP(y), bady);
      return scm_divbigbig(BDIGITS(x), NUMDIGS(x), BDIGITS(y), NUMDIGS(y),
			   BIGSIGN(y), (BIGSIGN(x) ^ BIGSIGN(y)) ? 1 : 0);
    }
    if (!(z = INUM(y))) goto ov;
    return scm_divbigint(x, z, y < 0, (BIGSIGN(x) ? (y > 0) : (y < 0)) ? 1 : 0);
  }
  if NINUMP(y) {
# ifndef RECKLESS
    if (!(NIMP(y) && BIGP(y)))
    bady: scm_wta(y, (char *)ARG2, s_modulo);
# endif
    return (BIGSIGN(y) ? (x>0) : (x<0)) ? scm_sum(x, y) : x;
  }
#else
  ASSERT(INUMP(x), x, ARG1, s_modulo);
  ASSERT(INUMP(y), y, ARG2, s_modulo);
#endif
  if (!(yy = INUM(y)))
    ov: scm_wta(y, (char *)OVFLOW, s_modulo);
#if (__TURBOC__==1)
  z = INUM(x);
  z = ((yy<0) ? -z : z)%yy;
#else
  z = INUM(x)%yy;
#endif
  return MAKINUM(((yy<0) ? (z>0) : (z<0)) ? z+yy : z);
}

PROC1 (s_gcd, "gcd", tc7_asubr, scm_gcd);
#ifdef __STDC__
SCM
scm_gcd(SCM x, SCM y)
#else
SCM
scm_gcd(x, y)
     SCM x;
     SCM y;
#endif
{
  register long u, v, k, t;
  if UNBNDP(y) return UNBNDP(x) ? INUM0 : x;
 tailrec:
#ifdef BIGDIG
  if NINUMP(x) {
  big_gcd:
    ASSERT(NIMP(x) && BIGP(x), x, ARG1, s_gcd);
    if BIGSIGN(x) x = scm_copybig(x, 0);
  newy:
    if NINUMP(y) {
      ASSERT(NIMP(y) && BIGP(y), y, ARG2, s_gcd);
      if BIGSIGN(y) y = scm_copybig(y, 0);
      switch (scm_bigcomp(x, y)) {
      case -1:
      swaprec: t = scm_remainder(x, y); x = y; y = t; goto tailrec;
      case  0: return x;
      case  1: y = scm_remainder(y, x); goto newy;
      }
      /* instead of the switch, we could just return scm_gcd(y, scm_modulo(x, y)); */
    }
    if (INUM0==y) return x; goto swaprec;
  }
  if NINUMP(y) { t=x; x=y; y=t; goto big_gcd;}
#else
  ASSERT(INUMP(x), x, ARG1, s_gcd);
  ASSERT(INUMP(y), y, ARG2, s_gcd);
#endif
  u = INUM(x);
  if (u<0) u = -u;
  v = INUM(y);
  if (v<0) v = -v;
  else if (0==v) goto getout;
  if (0==u) {u = v; goto getout;}
  for (k = 1;!(1 & ((int)u|(int)v));k <<= 1, u >>= 1, v >>= 1);
  if (1 & (int)u) t = -v;
  else {
    t = u;
  b3:
    t = SRS(t, 1);
  }
  if (!(1 & (int)t)) goto b3;
  if (t>0) u = t;
  else v = -t;
  if ((t = u-v)) goto b3;
  u = u*k;
 getout:
  if (!POSFIXABLE(u))
#ifdef BIGDIG
    return scm_long2big(u);
#else
  scm_wta(x, (char *)OVFLOW, s_gcd);
#endif
  return MAKINUM(u);
}

PROC1 (s_lcm, "lcm", tc7_asubr, scm_lcm);
#ifdef __STDC__
SCM
scm_lcm(SCM n1, SCM n2)
#else
SCM
scm_lcm(n1, n2)
     SCM n1;
     SCM n2;
#endif
{
  SCM d;
  if UNBNDP(n2) {
    n2 = MAKINUM(1L);
    if UNBNDP(n1) return n2;
  }
  d = scm_gcd(n1, n2);
  if (INUM0==d) return d;
  return scm_abs(scm_product(n1, scm_quotient(n2, d)));
}

#ifndef BIGDIG
# ifndef FLOATS
#  define long2num MAKINUM
# endif
#endif

#ifndef long2num
PROC1 (s_logand, "logand", tc7_asubr, scm_logand);
#ifdef __STDC__
SCM
scm_logand(SCM n1, SCM n2)
#else
SCM
scm_logand(n1, n2)
     SCM n1;
     SCM n2;
#endif
{
  return scm_long2num(scm_num2long(n1, (char *)ARG1, s_logand)
		      & scm_num2long(n2, (char *)ARG2, s_logand));
}

PROC1 (s_logior, "logior", tc7_asubr, scm_logior);
#ifdef __STDC__
SCM
scm_logior(SCM n1, SCM n2)
#else
SCM
scm_logior(n1, n2)
     SCM n1;
     SCM n2;
#endif
{
  return scm_long2num(scm_num2long(n1, (char *)ARG1, s_logior)
		      | scm_num2long(n2, (char *)ARG2, s_logior));
}

PROC1 (s_logxor, "logxor", tc7_asubr, scm_logxor);
#ifdef __STDC__
SCM
scm_logxor(SCM n1, SCM n2)
#else
SCM
scm_logxor(n1, n2)
     SCM n1;
     SCM n2;
#endif
{
  return scm_long2num(scm_num2long(n1, (char *)ARG1, s_logxor)
		      ^ scm_num2long(n2, (char *)ARG2, s_logxor));
}

PROC (s_logtest, "logtest", 2, 0, 0, scm_logtest);
#ifdef __STDC__
SCM
scm_logtest(SCM n1, SCM n2)
#else
SCM
scm_logtest(n1, n2)
     SCM n1;
     SCM n2;
#endif
{
  return ((scm_num2long (n1, (char *)ARG1, s_logtest)
	   & scm_num2long (n2, (char *)ARG2, s_logtest))
	  ? BOOL_T : BOOL_F);
}


PROC (s_logbit_p, "logbit?", 2, 0, 0, scm_logbit_p);
#ifdef __STDC__
SCM
scm_logbit_p(SCM n1, SCM n2)
#else
SCM
scm_logbit_p(n1, n2)
     SCM n1;
     SCM n2;
#endif
{
  return (((1 << scm_num2long (n1, (char *)ARG1, s_logtest))
	   & scm_num2long (n2, (char *)ARG2, s_logtest))
	  ? BOOL_T : BOOL_F);
}

#else

PROC1 (s_logand, "logand", tc7_asubr, scm_logand);
#ifdef __STDC__
SCM
scm_logand(SCM n1, SCM n2)
#else
SCM
scm_logand(n1, n2)
     SCM n1;
     SCM n2;
#endif
{
  ASSERT(INUMP(n1), n1, ARG1, s_logand);
  ASSERT(INUMP(n2), n2, ARG2, s_logand);
  return MAKINUM(INUM(n1) & INUM(n2));
}

PROC1 (s_logior, "logior", tc7_asubr, scm_logior);
#ifdef __STDC__
SCM
scm_logior(SCM n1, SCM n2)
#else
SCM
scm_logior(n1, n2)
     SCM n1;
     SCM n2;
#endif
{
  ASSERT(INUMP(n1), n1, ARG1, s_logior);
  ASSERT(INUMP(n2), n2, ARG2, s_logior);
  return MAKINUM(INUM(n1) | INUM(n2));
}

PROC1 (s_logxor, "logxor", tc7_asubr, scm_logxor);
#ifdef __STDC__
SCM
scm_logxor(SCM n1, SCM n2)
#else
SCM
scm_logxor(n1, n2)
     SCM n1;
     SCM n2;
#endif
{
  ASSERT(INUMP(n1), n1, ARG1, s_logxor);
  ASSERT(INUMP(n2), n2, ARG2, s_logxor);
  return MAKINUM(INUM(n1) ^ INUM(n2));
}

PROC (s_logtest, "logtest", 2, 0, 0, scm_logtest);
#ifdef __STDC__
SCM
scm_logtest(SCM n1, SCM n2)
#else
SCM
scm_logtest(n1, n2)
     SCM n1;
     SCM n2;
#endif
{
  ASSERT(INUMP(n1), n1, ARG1, s_logtest);
  ASSERT(INUMP(n2), n2, ARG2, s_logtest);
  return (INUM(n1) & INUM(n2)) ? BOOL_T : BOOL_F;
}

PROC (s_logbit_p, "logbit?", 2, 0, 0, scm_logbit_p);
#ifdef __STDC__
SCM
scm_logbit_p(SCM n1, SCM n2)
#else
SCM
scm_logbit_p(n1, n2)
     SCM n1;
     SCM n2;
#endif
{
  ASSERT(INUMP(n1) && INUM(n1) >= 0, n1, ARG1, s_logbit_p);
  ASSERT(INUMP(n2), n2, ARG2, s_logbit_p);
  return ((1 << INUM(n1)) & INUM(n2)) ? BOOL_T : BOOL_F;
}
#endif

PROC (s_lognot, "lognot", 1, 0, 0, scm_lognot);
#ifdef __STDC__
SCM
scm_lognot(SCM n)
#else
SCM
scm_lognot(n)
     SCM n;
#endif
{
  ASSERT(INUMP(n), n, ARG1, s_lognot);
  return scm_difference(MAKINUM(-1L), n);
}

PROC (s_integer_expt, "integer-expt", 2, 0, 0, scm_integer_expt);
#ifdef __STDC__
SCM
scm_integer_expt(SCM z1, SCM z2)
#else
SCM
scm_integer_expt(z1, z2)
     SCM z1;
     SCM z2;
#endif
{
  SCM acc = MAKINUM(1L);
#ifdef BIGDIG
  if (INUM0==z1 || acc==z1) return z1;
  else if (MAKINUM(-1L)==z1) return BOOL_F==scm_even_p(z2)?z1:acc;
#endif
  ASSERT(INUMP(z2), z2, ARG2, s_integer_expt);
  z2 = INUM(z2);
  if (z2 < 0) {
    z2 = -z2;
    z1 = scm_divide(z1, SCM_UNDEFINED);
  }
  while(1) {
    if (0==z2) return acc;
    if (1==z2) return scm_product(acc, z1);
    if (z2 & 1) acc = scm_product(acc, z1);
    z1 = scm_product(z1, z1);
    z2 >>= 1;
  }
}

PROC (s_ash, "ash", 2, 0, 0, scm_ash);
#ifdef __STDC__
SCM
scm_ash(SCM n, SCM cnt)
#else
SCM
scm_ash(n, cnt)
     SCM n;
     SCM cnt;
#endif
{
  SCM res = INUM(n);
  ASSERT(INUMP(cnt), cnt, ARG2, s_ash);
#ifdef BIGDIG
  if(cnt < 0) {
    res = scm_integer_expt(MAKINUM(2), MAKINUM(-INUM(cnt)));
    if (NFALSEP(scm_negative_p(n)))
      return scm_sum(MAKINUM(-1L), scm_quotient(scm_sum(MAKINUM(1L), n), res));
    else return scm_quotient(n, res);
  }
  else return scm_product(n, scm_integer_expt(MAKINUM(2), cnt));
#else
  ASSERT(INUMP(n), n, ARG1, s_ash);
  cnt = INUM(cnt);
  if (cnt < 0) return MAKINUM(SRS(res, -cnt));
  res = MAKINUM(res<<cnt);
  if (INUM(res)>>cnt != INUM(n)) scm_wta(n, (char *)OVFLOW, s_ash);
  return res;
#endif
}

PROC (s_bit_extract, "bit-extract", 3, 0, 0, scm_bit_extract);
#ifdef __STDC__
SCM
scm_bit_extract(SCM n, SCM start, SCM end)
#else
SCM
scm_bit_extract(n, start, end)
     SCM n;
     SCM start;
     SCM end;
#endif
{
  ASSERT(INUMP(start), start, ARG2, s_bit_extract);
  ASSERT(INUMP(end), end, ARG3, s_bit_extract);
  start = INUM(start); end = INUM(end);
  ASSERT(end >= start, MAKINUM(end), OUTOFRANGE, s_bit_extract);
#ifdef BIGDIG
  if NINUMP(n)
    return
      scm_logand(scm_difference(scm_integer_expt(MAKINUM(2), MAKINUM(end - start)),
				MAKINUM(1L)),
		 scm_ash(n, MAKINUM(-start)));
#else
  ASSERT(INUMP(n), n, ARG1, s_bit_extract);
#endif
  return MAKINUM((INUM(n)>>start) & ((1L<<(end-start))-1));
}

char scm_logtab[] = {0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4};
PROC (s_logcount, "logcount", 1, 0, 0, scm_logcount);
#ifdef __STDC__
SCM
scm_logcount (SCM n)
#else
SCM
scm_logcount(n)
     SCM n;
#endif
{
  register unsigned long c = 0;
  register long nn;
#ifdef BIGDIG
  if NINUMP(n) {
    sizet i; BIGDIG *ds, d;
    ASSERT(NIMP(n) && BIGP(n), n, ARG1, s_logcount);
    if BIGSIGN(n) return scm_logcount(scm_difference(MAKINUM(-1L), n));
    ds = BDIGITS(n);
    for(i = NUMDIGS(n); i--; )
      for(d = ds[i]; d; d >>= 4) c += scm_logtab[15 & d];
    return MAKINUM(c);
  }
#else
  ASSERT(INUMP(n), n, ARG1, s_logcount);
#endif
  if ((nn = INUM(n)) < 0) nn = -1 - nn;
  for(; nn; nn >>= 4) c += scm_logtab[15 & nn];
  return MAKINUM(c);
}

char scm_ilentab[] = {0, 1, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4};
PROC (s_integer_length, "integer-length", 1, 0, 0, scm_integer_length);
#ifdef __STDC__
SCM
scm_integer_length(SCM n)
#else
SCM
scm_integer_length(n)
     SCM n;
#endif
{
  register unsigned long c = 0;
  register long nn;
  unsigned int l = 4;
#ifdef BIGDIG
  if NINUMP(n) {
    BIGDIG *ds, d;
    ASSERT(NIMP(n) && BIGP(n), n, ARG1, s_integer_length);
    if BIGSIGN(n) return scm_integer_length(scm_difference(MAKINUM(-1L), n));
    ds = BDIGITS(n);
    d = ds[c = NUMDIGS(n)-1];
    for(c *= BITSPERDIG; d; d >>= 4) {c += 4; l = scm_ilentab[15 & d];}
    return MAKINUM(c - 4 + l);
  }
#else
  ASSERT(INUMP(n), n, ARG1, s_integer_length);
#endif
  if ((nn = INUM(n)) < 0) nn = -1 - nn;
  for(;nn; nn >>= 4) {c += 4; l = scm_ilentab[15 & nn];}
  return MAKINUM(c - 4 + l);
}


#ifdef BIGDIG
char s_bignum[] = "bignum";
#ifdef __STDC__
SCM
scm_mkbig(sizet nlen, int sign)
#else
SCM
scm_mkbig(nlen, sign)
     sizet nlen;
     int sign;
#endif
{
  SCM v = nlen;
  if (((v << 16) >> 16) != nlen)
    scm_wta(MAKINUM(nlen), (char *)NALLOC, s_bignum);
  NEWCELL(v);
  DEFER_INTS;
  SETCHARS(v, scm_must_malloc((long)(nlen*sizeof(BIGDIG)), s_bignum));
  SETNUMDIGS(v, nlen, sign?tc16_bigneg:tc16_bigpos);
  ALLOW_INTS;
  return v;
}

#ifdef __STDC__
SCM
scm_big2inum(SCM b, sizet l)
#else
SCM
scm_big2inum(b, l)
     SCM b;
     sizet l;
#endif
{
  unsigned long num = 0;
  BIGDIG *tmp = BDIGITS(b);
  while (l--) num = BIGUP(num) + tmp[l];
  if (TYP16(b)==tc16_bigpos) {
    if POSFIXABLE(num) return MAKINUM(num);
  }
  else if UNEGFIXABLE(num) return MAKINUM(-num);
  return b;
}


char s_adjbig[] = "scm_adjbig";
#ifdef __STDC__
SCM
scm_adjbig(SCM b, sizet nlen)
#else
SCM
scm_adjbig(b, nlen)
     SCM b;
     sizet nlen;
#endif
{
  long nsiz = nlen;
  if (((nsiz << 16) >> 16) != nlen) scm_wta(MAKINUM(nsiz), (char *)NALLOC, s_adjbig);
  DEFER_INTS;
  SETCHARS(b, (BIGDIG *)scm_must_realloc((char *)CHARS(b),
					 (long)(NUMDIGS(b)*sizeof(BIGDIG)),
					 (long)(nsiz*sizeof(BIGDIG)), s_adjbig));
  SETNUMDIGS(b, nsiz, TYP16(b));
  ALLOW_INTS;
  return b;
}


#ifdef __STDC__
SCM
scm_normbig(SCM b)
#else
SCM
scm_normbig(b)
     SCM b;
#endif
{
#ifndef _UNICOS  
  sizet nlen = NUMDIGS(b);
#else
  int nlen = NUMDIGS(b);	/* unsigned nlen breaks on Cray when nlen => 0 */
#endif
  BIGDIG *zds = BDIGITS(b);
  while (nlen-- && !zds[nlen]); nlen++;
  if (nlen * BITSPERDIG/CHAR_BIT <= sizeof(SCM))
    if INUMP(b = scm_big2inum(b, (sizet)nlen)) return b;
  if (NUMDIGS(b)==nlen) return b;
  return scm_adjbig(b, (sizet)nlen);
}


#ifdef __STDC__
SCM
scm_copybig(SCM b, int sign)
#else
SCM
scm_copybig(b, sign)
     SCM b;
     int sign;
#endif
{
  sizet i = NUMDIGS(b);
  SCM ans = scm_mkbig(i, sign);
  BIGDIG *src = BDIGITS(b), *dst = BDIGITS(ans);
  while (i--) dst[i] = src[i];
  return ans;
}


#ifdef __STDC__
SCM
scm_long2big(long n)
#else
SCM
scm_long2big(n)
     long n;
#endif
{
  sizet i = 0;
  BIGDIG *digits;
  SCM ans = scm_mkbig(DIGSPERLONG, n<0);
  digits = BDIGITS(ans);
  if (n < 0) n = -n;
  while (i < DIGSPERLONG) {
    digits[i++] = BIGLO(n);
    n = BIGDN((unsigned long)n);
  }
  return ans;
}


#ifdef __STDC__
SCM
scm_2ulong2big(unsigned long * np)
#else
SCM
scm_2ulong2big(np)
     unsigned long * np;
#endif
{
  unsigned long n;
  sizet i;
  BIGDIG *digits;
  SCM ans;

  ans = scm_mkbig(2 * DIGSPERLONG, 0);
  digits = BDIGITS(ans);

  n = np[0];
  for (i = 0; i < DIGSPERLONG; ++i)
    {
      digits[i] = BIGLO(n);
      n = BIGDN((unsigned long)n);
    }
  n = np[1];
  for (i = 0; i < DIGSPERLONG; ++i)
    {
      digits[i + DIGSPERLONG] = BIGLO(n);
      n = BIGDN((unsigned long)n);
    }
  return ans;
}


#ifdef __STDC__
SCM
scm_ulong2big(unsigned long n)
#else
SCM
scm_ulong2big(n)
     unsigned long n;
#endif
{
  sizet i = 0;
  BIGDIG *digits;
  SCM ans = scm_mkbig(DIGSPERLONG, 0);
  digits = BDIGITS(ans);
  while (i < DIGSPERLONG) {
    digits[i++] = BIGLO(n);
    n = BIGDN(n);
  }
  return ans;
}


#ifdef __STDC__
int
scm_bigcomp(SCM x, SCM y)
#else
int
scm_bigcomp(x, y)
     SCM x;
     SCM y;
#endif
{
  int xsign = BIGSIGN(x);
  int ysign = BIGSIGN(y);
  sizet xlen, ylen;
  if (ysign < xsign) return 1;
  if (ysign > xsign) return -1;
  if ((ylen = NUMDIGS(y)) > (xlen = NUMDIGS(x))) return (xsign) ? -1 : 1;
  if (ylen < xlen) return (xsign) ? 1 : -1;
  while(xlen-- && (BDIGITS(y)[xlen]==BDIGITS(x)[xlen]));
  if (-1==xlen) return 0;
  return (BDIGITS(y)[xlen] > BDIGITS(x)[xlen]) ?
    (xsign ? -1 : 1) : (xsign ? 1 : -1);
}

#ifndef DIGSTOOBIG
long
scm_pseudolong(x)
     long x;
{
  union {
    long l;
    BIGDIG bd[DIGSPERLONG];
  } p;
  sizet i = 0;
  if (x < 0) x = -x;
  while (i < DIGSPERLONG) {p.bd[i++] = BIGLO(x); x = BIGDN(x);}
  /*  p.bd[0] = BIGLO(x); p.bd[1] = BIGDN(x); */
  return p.l;
}

#else

#ifdef __STDC__
void
scm_longdigs(long x, SCM_BIGDIG digs[])
#else
void
scm_longdigs(x, digs)
     long x;
     SCM_BIGDIG digs[];
#endif
{
  sizet i = 0;
  if (x < 0) x = -x;
  while (i < DIGSPERLONG) {digs[i++] = BIGLO(x); x = BIGDN(x);}
}
#endif


#ifdef __STDC__
SCM
scm_addbig(SCM_BIGDIG *x, sizet nx, int xsgn, SCM bigy, int sgny)
#else
SCM
scm_addbig(x, nx, xsgn, bigy, sgny)
     SCM_BIGDIG *x;
     sizet nx;
     int xsgn;
     SCM bigy;
     int sgny;
#endif
{
  /* Assumes nx <= NUMDIGS(bigy) */
  /* Assumes xsgn and sgny scm_equal either 0 or 0x0100 */
  long num = 0;
  sizet i = 0, ny = NUMDIGS(bigy);
  SCM z = scm_copybig(bigy, BIGSIGN(bigy) ^ sgny);
  BIGDIG *zds = BDIGITS(z);
  if (xsgn ^ BIGSIGN(z)) {
    do {
      num += (long) zds[i] - x[i];
      if (num < 0) {zds[i] = num + BIGRAD; num = -1;}
      else {zds[i] = BIGLO(num); num = 0;}
    } while (++i < nx);
    if (num && nx==ny) {
      num = 1; i = 0;
      CAR(z) ^= 0x0100;
      do {
	num += (BIGRAD-1) - zds[i];
	zds[i++] = BIGLO(num);
	num = BIGDN(num);
      } while (i < ny);
    }
    else while (i < ny) {
      num += zds[i];
      if (num < 0) {zds[i++] = num + BIGRAD; num = -1;}
      else {zds[i++] = BIGLO(num); num = 0;}
    }
  } else {
    do {
      num += (long) zds[i] + x[i];
      zds[i++] = BIGLO(num);
      num = BIGDN(num);
    } while (i < nx);
    if (!num) return z;
    while (i < ny) {
      num += zds[i];
      zds[i++] = BIGLO(num);
      num = BIGDN(num);
      if (!num) return z;
    }
    if (num) {z = scm_adjbig(z, ny+1); BDIGITS(z)[ny] = num; return z;}
  }
  return scm_normbig(z);
}

#ifdef __STDC__
SCM
scm_mulbig(SCM_BIGDIG *x, sizet nx, SCM_BIGDIG *y, sizet ny, int sgn)
#else
SCM
scm_mulbig(x, nx, y, ny, sgn)
     SCM_BIGDIG *x;
     sizet nx;
     SCM_BIGDIG *y;
     sizet ny;
     int sgn;
#endif
{
  sizet i = 0, j = nx + ny;
  unsigned long n = 0;
  SCM z = scm_mkbig(j, sgn);
  BIGDIG *zds = BDIGITS(z);
  while (j--) zds[j] = 0;
  do {
    j = 0;
    if (x[i]) {
      do {
	n += zds[i + j] + ((unsigned long) x[i] * y[j]);
	zds[i + j++] = BIGLO(n);
	n = BIGDN(n);
      } while (j < ny);
      if (n) {zds[i + j] = n; n = 0;}
    }
  } while (++i < nx);
  return scm_normbig(z);
}

#ifdef __STDC__
unsigned int
scm_divbigdig(SCM_BIGDIG *ds, sizet h, SCM_BIGDIG div)
#else
unsigned int
scm_divbigdig(ds, h, div)
     SCM_BIGDIG *ds;
     sizet h;
     SCM_BIGDIG div;
#endif
{
  register unsigned long t2 = 0;
  while(h--) {
    t2 = BIGUP(t2) + ds[h];
    ds[h] = t2 / div;
    t2 %= div;
  }
  return t2;
}


#ifdef __STDC__
SCM
scm_divbigint(SCM x, long z, int sgn, int mode)
#else
SCM
scm_divbigint(x, z, sgn, mode)
     SCM x;
     long z;
     int sgn;
     int mode;
#endif
{
  if (z < 0) z = -z;
  if (z < BIGRAD) {
    register unsigned long t2 = 0;
    register BIGDIG *ds = BDIGITS(x);
    sizet nd = NUMDIGS(x);
    while(nd--) t2 = (BIGUP(t2) + ds[nd]) % z;
    if (mode) t2 = z - t2;
    return MAKINUM(sgn ? -t2 : t2);
  }
  {
#ifndef DIGSTOOBIG
    unsigned long t2 = scm_pseudolong(z);
    return scm_divbigbig(BDIGITS(x), NUMDIGS(x), (BIGDIG *)&t2,
			 DIGSPERLONG, sgn, mode); 
#else
    BIGDIG t2[DIGSPERLONG];
    scm_longdigs(z, t2);
    return scm_divbigbig(BDIGITS(x), NUMDIGS(x), t2, DIGSPERLONG, sgn, mode);
#endif
  }
}

#ifdef __STDC__
SCM
scm_divbigbig(SCM_BIGDIG *x, sizet nx, SCM_BIGDIG *y, sizet ny, int sgn, int modes)
#else
SCM
scm_divbigbig(x, nx, y, ny, sgn, modes)
     SCM_BIGDIG *x;
     sizet nx;
     SCM_BIGDIG *y;
     sizet ny;
     int sgn;
     int modes;
#endif
{
  /* modes description
     0	remainder
     1	scm_modulo
     2	quotient
     3	quotient but returns 0 if division is not exact. */
  sizet i = 0, j = 0;
  long num = 0;
  unsigned long t2 = 0;
  SCM z, newy;
  BIGDIG  d = 0, qhat, *zds, *yds;
  /* algorithm requires nx >= ny */
  if (nx < ny)
    switch (modes) {
    case 0:			/* remainder -- just return x */
      z = scm_mkbig(nx, sgn); zds = BDIGITS(z);
      do {zds[i] = x[i];} while (++i < nx);
      return z;
    case 1:			/* scm_modulo -- return y-x */
      z = scm_mkbig(ny, sgn); zds = BDIGITS(z);
      do {
	num += (long) y[i] - x[i];
	if (num < 0) {zds[i] = num + BIGRAD; num = -1;}
	else {zds[i] = num; num = 0;}
      } while (++i < nx);
      while (i < ny) {
	num += y[i];
	if (num < 0) {zds[i++] = num + BIGRAD; num = -1;}
	else {zds[i++] = num; num = 0;}
      }
      goto doadj;
    case 2: return INUM0;	/* quotient is zero */
    case 3: return 0;		/* the division is not exact */
    }

  z = scm_mkbig(nx==ny ? nx+2 : nx+1, sgn); zds = BDIGITS(z);
  if (nx==ny) zds[nx+1] = 0;
  while(!y[ny-1]) ny--;		/* in case y came in as a psuedolong */
  if (y[ny-1] < (BIGRAD>>1)) {  /* normalize operands */
    d = BIGRAD/(y[ny-1]+1);
    newy = scm_mkbig(ny, 0); yds = BDIGITS(newy);
    while(j < ny)
      {t2 += (unsigned long) y[j]*d; yds[j++] = BIGLO(t2); t2 = BIGDN(t2);}
    y = yds; j = 0; t2 = 0;
    while(j < nx)
      {t2 += (unsigned long) x[j]*d; zds[j++] = BIGLO(t2); t2 = BIGDN(t2);}
    zds[j] = t2;
  }
  else {zds[j = nx] = 0; while (j--) zds[j] = x[j];}
  j = nx==ny ? nx+1 : nx;	/* dividend needs more digits than divisor */
  do {				/* loop over digits of quotient */
    if (zds[j]==y[ny-1]) qhat = BIGRAD-1;
    else qhat = (BIGUP(zds[j]) + zds[j-1])/y[ny-1];
    if (!qhat) continue;
    i = 0; num = 0; t2 = 0;
    do {			/* multiply and subtract */
      t2 += (unsigned long) y[i] * qhat;
      num += zds[j - ny + i] - BIGLO(t2);
      if (num < 0) {zds[j - ny + i] = num + BIGRAD; num = -1;}
      else {zds[j - ny + i] = num; num = 0;}
      t2 = BIGDN(t2);
    } while (++i < ny);
    num += zds[j - ny + i] - t2; /* borrow from high digit; don't update */
    while (num) {		/* "add back" required */
      i = 0; num = 0; qhat--;
      do {
	num += (long) zds[j - ny + i] + y[i];
	zds[j - ny + i] = BIGLO(num);
	num = BIGDN(num);
      } while (++i < ny);
      num--;
    }
    if (modes & 2) zds[j] = qhat;
  } while (--j >= ny);
  switch (modes) {
  case 3:			/* check that remainder==0 */
    for(j = ny;j && !zds[j-1];--j) ; if (j) return 0;
  case 2:			/* move quotient down in z */
    j = (nx==ny ? nx+2 : nx+1) - ny;
    for (i = 0;i < j;i++) zds[i] = zds[i+ny];
    ny = i;
    break;
  case 1:			/* subtract for scm_modulo */
    i = 0; num = 0; j = 0;
    do {num += y[i] - zds[i];
	j = j | zds[i];
	if (num < 0) {zds[i] = num + BIGRAD; num = -1;}
	else {zds[i] = num; num = 0;}
      } while (++i < ny);
    if (!j) return INUM0;
  case 0:			/* just normalize remainder */
    if (d) scm_divbigdig(zds, ny, d);
  }
 doadj:
  for(j = ny;j && !zds[j-1];--j) ;
  if (j * BITSPERDIG <= sizeof(SCM)*CHAR_BIT)
    if INUMP(z = scm_big2inum(z, j)) return z;
  return scm_adjbig(z, j);
}
#endif





/*** NUMBERS -> STRINGS ***/
#ifdef FLOATS
int scm_dblprec;
static double fx[] = {0.0, 5e-1, 5e-2, 5e-3, 5e-4, 5e-5,
			5e-6, 5e-7, 5e-8, 5e-9, 5e-10,
			5e-11,5e-12,5e-13,5e-14,5e-15,
			5e-16,5e-17,5e-18,5e-19,5e-20};



#ifdef __STDC__
static sizet
idbl2str(double f, char *a)
#else
static sizet
idbl2str(f, a)
     double f;
     char *a;
#endif
{
  int efmt, dpt, d, i, wp = scm_dblprec;
  sizet ch = 0;
  int exp = 0;

  if (f == 0.0) goto zero;	/*{a[0]='0'; a[1]='.'; a[2]='0'; return 3;}*/
  if (f < 0.0) {f = -f;a[ch++]='-';}
  else if (f > 0.0) ;
  else goto funny;
  if (IS_INF(f))
    {
      if (ch == 0) a[ch++]='+';
    funny: a[ch++]='#'; a[ch++]='.'; a[ch++]='#'; return ch;
    }
# ifdef DBL_MIN_10_EXP		/* Prevent unnormalized values, as from 
				   make-uniform-vector, from causing infinite loops. */
  while (f < 1.0) {f *= 10.0;  if (exp-- < DBL_MIN_10_EXP) goto funny;}
  while (f > 10.0) {f *= 0.10; if (exp++ > DBL_MAX_10_EXP) goto funny;}
# else
  while (f < 1.0) {f *= 10.0; exp--;}
  while (f > 10.0) {f /= 10.0; exp++;}
# endif
  if (f+fx[wp] >= 10.0) {f = 1.0; exp++;}
 zero:
# ifdef ENGNOT
  dpt = (exp+9999)%3;
  exp -= dpt++;
  efmt = 1;
# else
  efmt = (exp < -3) || (exp > wp+2);
  if (!efmt)
    if (exp < 0) {
      a[ch++] = '0';
      a[ch++] = '.';
      dpt = exp;
      while (++dpt)  a[ch++] = '0';
    } else
      dpt = exp+1;
  else
    dpt = 1;
# endif

  do {
    d = f;
    f -= d;
    a[ch++] = d+'0';
    if (f < fx[wp])  break;
    if (f+fx[wp] >= 1.0) {
      a[ch-1]++;
      break;
    }
    f *= 10.0;
    if (!(--dpt))  a[ch++] = '.';
  } while (wp--);

  if (dpt > 0)
# ifndef ENGNOT
    if ((dpt > 4) && (exp > 6)) {
      d = (a[0]=='-'?2:1);
      for (i = ch++; i > d; i--)
	a[i] = a[i-1];
      a[d] = '.';
      efmt = 1;
    } else
# endif
      {
	while (--dpt)  a[ch++] = '0';
	a[ch++] = '.';
      }
  if (a[ch-1]=='.')  a[ch++]='0'; /* trailing zero */
  if (efmt && exp) {
    a[ch++] = 'e';
    if (exp < 0) {
      exp = -exp;
      a[ch++] = '-';
    }
    for (i = 10; i <= exp; i *= 10);
    for (i /= 10; i; i /= 10) {
      a[ch++] = exp/i + '0';
      exp %= i;
    }
  }
  return ch;
}

#ifdef __STDC__
static sizet
iflo2str(SCM flt, char *str)
#else
static sizet
iflo2str(flt, str)
     SCM flt;
     char *str;
#endif
{
  sizet i;
# ifdef SINGLES
  if SINGP(flt) i = idbl2str(FLO(flt), str);
  else
# endif
    i = idbl2str(REAL(flt), str);
  if CPLXP(flt) {
    if(0 <= IMAG(flt))		/* jeh */
      str[i++] = '+';		/* jeh */
    i += idbl2str(IMAG(flt), &str[i]);
    str[i++] = 'i';
  }
  return i;
}
#endif				/* FLOATS */

#ifdef __STDC__
sizet
scm_iint2str(long num, int rad, char *p)
#else
sizet
scm_iint2str(num, rad, p)
     long num;
     int rad;
     char *p;
#endif
{
  sizet j;
  register int i = 1, d;
  register long n = num;
  if (n < 0) {n = -n; i++;}
  for (n /= rad;n > 0;n /= rad) i++;
  j = i;
  n = num;
  if (n < 0) {n = -n; *p++ = '-'; i--;}
  while (i--) {
    d = n % rad;
    n /= rad;
    p[i] = d + ((d < 10) ? '0' : 'a' - 10);
  }
  return j;
}


#ifdef BIGDIG
#ifdef __STDC__
static SCM
big2str(SCM b, register unsigned int radix)
#else
static SCM
big2str(b, radix)
     SCM b;
     register unsigned int radix;
#endif
{
  SCM t = scm_copybig(b, 0);	/* sign of temp doesn't matter */
  register BIGDIG *ds = BDIGITS(t);
  sizet i = NUMDIGS(t);
  sizet j = radix==16 ? (BITSPERDIG*i)/4+2
    : radix >= 10 ? (BITSPERDIG*i*241L)/800+2
      : (BITSPERDIG*i)+2;
  sizet k = 0;
  sizet radct = 0;
  sizet ch;			/* jeh */
  BIGDIG radpow = 1, radmod = 0;
  SCM ss = scm_makstr((long)j, 0);
  char *s = CHARS(ss), c;
  while ((long) radpow * radix < BIGRAD) {
    radpow *= radix;
    radct++;
  }
  s[0] = tc16_bigneg==TYP16(b) ? '-' : '+';
  while ((i || radmod) && j) {
    if (k == 0) {
      radmod = (BIGDIG)scm_divbigdig(ds, i, radpow);
      k = radct;
      if (!ds[i-1]) i--;
    }
    c = radmod % radix; radmod /= radix; k--;
    s[--j] = c < 10 ? c + '0' : c + 'a' - 10;
  }
  ch = s[0] == '-' ? 1 : 0;	/* jeh */
  if (ch < j) {			/* jeh */
    for(i = j;j < LENGTH(ss);j++) s[ch+j-i] = s[j]; /* jeh */
    scm_resizuve(ss, (SCM)MAKINUM(ch+LENGTH(ss)-i)); /* jeh */
  }
  return ss;
}
#endif


PROC (s_number_to_string, "number->string", 1, 1, 0, scm_number_to_string);
#ifdef __STDC__
SCM
scm_number_to_string(SCM x, SCM radix)
#else
SCM
scm_number_to_string(x, radix)
     SCM x;
     SCM radix;
#endif
{
  if UNBNDP(radix) radix=MAKINUM(10L);
  else ASSERT(INUMP(radix), radix, ARG2, s_number_to_string);
#ifdef FLOATS
  if NINUMP(x) {
    char num_buf[FLOBUFLEN];
# ifdef BIGDIG
    ASRTGO(NIMP(x), badx);
    if BIGP(x) return big2str(x, (unsigned int)INUM(radix));
#  ifndef RECKLESS
    if (!(INEXP(x)))
    badx: scm_wta(x, (char *)ARG1, s_number_to_string);
#  endif
# else
    ASSERT(NIMP(x) && INEXP(x), x, ARG1, s_number_to_string);
# endif
    return scm_makfromstr(num_buf, iflo2str(x, num_buf), 0);
  }
#else
# ifdef BIGDIG
  if NINUMP(x) {
    ASSERT(NIMP(x) && BIGP(x), x, ARG1, s_number_to_string);
    return big2str(x, (unsigned int)INUM(radix));
  }
# else
  ASSERT(INUMP(x), x, ARG1, s_number_to_string);
# endif
#endif
  {
    char num_buf[INTBUFLEN];
    return scm_makfromstr(num_buf,
			  scm_iint2str(INUM(x), (int)INUM(radix), num_buf), 0);
  }
}


/* These print routines are stubbed here so that scm_repl.c doesn't need
   FLOATS or BIGDIGs conditionals */
#ifdef __STDC__
int
scm_floprint(SCM sexp, SCM port, int writing)
#else
int
scm_floprint(sexp, port, writing)
     SCM sexp;
     SCM port;
     int writing;
#endif
{
#ifdef FLOATS
  char num_buf[FLOBUFLEN];
  scm_lfwrite(num_buf, (sizet)sizeof(char), iflo2str(sexp, num_buf), port);
#else
  scm_ipruk("float", sexp, port);
#endif
  return !0;
}


#ifdef __STDC__
int
scm_bigprint(SCM exp, SCM port, int writing)
#else
int
scm_bigprint(exp, port, writing)
     SCM exp;
     SCM port;
     int writing;
#endif
{
#ifdef BIGDIG
  exp = big2str(exp, (unsigned int)10);
  scm_lfwrite(CHARS(exp), (sizet)sizeof(char), (sizet)LENGTH(exp), port);
#else
  scm_ipruk("bignum", exp, port);
#endif
  return !0;
}
/*** END nums->strs ***/

/*** STRINGS -> NUMBERS ***/
#ifdef BIGDIG
#ifdef __STDC__
SCM
scm_istr2int(char *str, long len, long radix)
#else
SCM
scm_istr2int(str, len, radix)
     char *str;
     long len;
     long radix;
#endif
{
  sizet j;
  register sizet k, blen = 1;
  sizet i = 0;
  int c;
  SCM res;
  register BIGDIG *ds;
  register unsigned long t2;

  if (0 >= len) return BOOL_F;	/* zero scm_length */
  if (16==radix) j = 1+(4*len*sizeof(char))/(BITSPERDIG);
  else if (10 <= radix)
    j = 1+(84*len*sizeof(char))/(BITSPERDIG*25);
  else j = 1+(len*sizeof(char))/(BITSPERDIG);
  switch (str[0]) {		/* leading sign */
  case '-':
  case '+': if (++i==len) return BOOL_F; /* bad if lone `+' or `-' */
  }
  res = scm_mkbig(j, '-'==str[0]);
  ds = BDIGITS(res);
  for (k = j;k--;) ds[k] = 0;
  do {
    switch (c = str[i++]) {
    case DIGITS:
      c = c - '0';
      goto accumulate;
    case 'A': case 'B': case 'C': case 'D': case 'E': case 'F':
      c = c-'A'+10;
      goto accumulate;
    case 'a': case 'b': case 'c': case 'd': case 'e': case 'f':
      c = c-'a'+10;
    accumulate:
      if (c >= radix) return BOOL_F; /* bad digit for radix */
      k = 0;
      t2 = c;
    moretodo:
      while(k < blen) {
	/*	printf("k = %d, blen = %d, t2 = %ld, ds[k] = %d\n", k, blen, t2, ds[k]);*/
	t2 += ds[k]*radix;
	ds[k++] = BIGLO(t2);
	t2 = BIGDN(t2);
      }
      ASSERT(blen <= j, (SCM)MAKINUM(blen), OVFLOW, "bignum");
      if (t2) {blen++; goto moretodo;}
      break;
    default:
      return BOOL_F;		/* not a digit */
    }
  } while (i < len);
  if (blen * BITSPERDIG/CHAR_BIT <= sizeof(SCM))
    if INUMP(res = scm_big2inum(res, blen)) return res;
  if (j==blen) return res;
  return scm_adjbig(res, blen);
}
#else



#ifdef __STDC__
SCM
scm_istr2int(char *str, long len, long radix)
#else
SCM
scm_istr2int(str, len, radix)
     char *str;
     long len;
     long radix;
#endif
{
  register long n = 0, ln;
  register int c;
  register int i = 0;
  int lead_neg = 0;
  if (0 >= len) return BOOL_F;	/* zero scm_length */
  switch (*str) {		/* leading sign */
  case '-': lead_neg = 1;
  case '+': if (++i==len) return BOOL_F; /* bad if lone `+' or `-' */
  }

  do {
    switch (c = str[i++]) {
    case DIGITS:
      c = c - '0';
      goto accumulate;
    case 'A': case 'B': case 'C': case 'D': case 'E': case 'F':
      c = c-'A'+10;
      goto accumulate;
    case 'a': case 'b': case 'c': case 'd': case 'e': case 'f':
      c = c-'a'+10;
    accumulate:
      if (c >= radix) return BOOL_F; /* bad digit for radix */
      ln = n;
      n = n * radix - c;
      /* Negation is a workaround for HP700 cc bug */
      if (n > ln || (-n > -MOST_NEGATIVE_FIXNUM)) goto ovfl;
      break;
    default:
      return BOOL_F;		/* not a digit */
    }
  } while (i < len);
  if (!lead_neg) if ((n = -n) > MOST_POSITIVE_FIXNUM) goto ovfl;
  return MAKINUM(n);
 ovfl:				/* overflow scheme integer */
  return BOOL_F;
}
#endif

#ifdef FLOATS
#ifdef __STDC__
SCM
scm_istr2flo(char *str, long len, long radix)
#else
SCM
scm_istr2flo(str, len, radix)
     char *str;
     long len;
     long radix;
#endif
{
  register int c, i = 0;
  double lead_sgn;
  double res = 0.0, tmp = 0.0;
  int flg = 0;
  int point = 0;
  SCM second;

  if (i >= len) return BOOL_F;	/* zero scm_length */

  switch (*str) {		/* leading sign */
  case '-': lead_sgn = -1.0; i++; break;
  case '+': lead_sgn = 1.0; i++; break;
  default : lead_sgn = 0.0;
  }
  if (i==len) return BOOL_F;	/* bad if lone `+' or `-' */

  if (str[i]=='i' || str[i]=='I') { /* handle `+i' and `-i'   */
    if (lead_sgn==0.0) return BOOL_F; /* must have leading sign */
    if (++i < len) return BOOL_F; /* `i' not last character */
    return scm_makdbl(0.0, lead_sgn);
  }
  do {				/* check initial digits */
    switch (c = str[i]) {
    case DIGITS:
      c = c - '0';
      goto accum1;
    case 'D': case 'E': case 'F':
      if (radix==10) goto out1; /* must be exponent */
    case 'A': case 'B': case 'C':
      c = c-'A'+10;
      goto accum1;
    case 'd': case 'e': case 'f':
      if (radix==10) goto out1;
    case 'a': case 'b': case 'c':
      c = c-'a'+10;
    accum1:
      if (c >= radix) return BOOL_F; /* bad digit for radix */
      res = res * radix + c;
      flg = 1;			/* res is valid */
      break;
    default:
      goto out1;
    }
  } while (++i < len);
 out1:

  /* if true, then we did see a digit above, and res is valid */
  if (i==len) goto done;

  /* By here, must have seen a digit,
     or must have next char be a `.' with radix==10 */
  if (!flg)
    if (!(str[i]=='.' && radix==10))
      return BOOL_F;

  while (str[i]=='#') {		/* optional sharps */
    res *= radix;
    if (++i==len) goto done;
  }

  if (str[i]=='/') {
    while (++i < len) {
      switch (c = str[i]) {
      case DIGITS:
	c = c - '0';
	goto accum2;
      case 'A': case 'B': case 'C': case 'D': case 'E': case 'F':
	c = c-'A'+10;
	goto accum2;
      case 'a': case 'b': case 'c': case 'd': case 'e': case 'f':
	c = c-'a'+10;
      accum2:
	if (c >= radix) return BOOL_F;
	tmp = tmp * radix + c;
	break;
      default:
	goto out2;
      }
    }
  out2:
    if (tmp==0.0) return BOOL_F; /* `slash zero' not allowed */
    if (i < len)
      while (str[i]=='#') {	/* optional sharps */
	tmp *= radix;
	if (++i==len) break;
      }
    res /= tmp;
    goto done;
  }

  if (str[i]=='.') {		/* decimal point notation */
    if (radix != 10) return BOOL_F; /* must be radix 10 */
    while (++i < len) {
      switch (c = str[i]) {
      case DIGITS:
	point--;
	res = res*10.0 + c-'0';
	flg = 1;
	break;
      default:
	goto out3;
      }
    }
  out3:
    if (!flg) return BOOL_F;	/* no digits before or after decimal point */
    if (i==len) goto adjust;
    while (str[i]=='#') {	/* ignore remaining sharps */
      if (++i==len) goto adjust;
    }
  }

  switch (str[i]) {		/* exponent */
  case 'd': case 'D':
  case 'e': case 'E':
  case 'f': case 'F':
  case 'l': case 'L':
  case 's': case 'S': {
    int expsgn = 1, expon = 0;
    if (radix != 10) return BOOL_F; /* only in radix 10 */
    if (++i==len) return BOOL_F; /* bad exponent */
    switch (str[i]) {
    case '-':  expsgn=(-1);
    case '+':  if (++i==len) return BOOL_F; /* bad exponent */
    }
    if (str[i] < '0' || str[i] > '9') return BOOL_F; /* bad exponent */
    do {
      switch (c = str[i]) {
      case DIGITS:
	expon = expon*10 + c-'0';
	if (expon > MAXEXP)  return BOOL_F; /* exponent too large */
	break;
      default:
	goto out4;
      }
    } while (++i < len);
  out4:
    point += expsgn*expon;
  }
  }

 adjust:
  if (point >= 0)
    while (point--)  res *= 10.0;
  else
# ifdef _UNICOS
    while (point++)  res *= 0.1; 
# else
  while (point++)  res /= 10.0;
# endif

 done:
  /* at this point, we have a legitimate floating point result */
  if (lead_sgn==-1.0)  res = -res;
  if (i==len) return scm_makdbl(res, 0.0);

  if (str[i]=='i' || str[i]=='I') { /* pure imaginary number  */
    if (lead_sgn==0.0) return BOOL_F; /* must have leading sign */
    if (++i < len) return BOOL_F; /* `i' not last character */
    return scm_makdbl(0.0, res);
  }

  switch (str[i++]) {
  case '-':  lead_sgn = -1.0; break;
  case '+':  lead_sgn = 1.0;  break;
  case '@': {			/* polar input for complex number */
    /* get a `real' for scm_angle */
    second = scm_istr2flo(&str[i], (long)(len-i), radix);
    if (!(INEXP(second))) return BOOL_F; /* not `real' */
    if (CPLXP(second))    return BOOL_F; /* not `real' */
    tmp = REALPART(second);
    return scm_makdbl(res*cos(tmp), res*sin(tmp));
  }
  default: return BOOL_F;
  }

  /* at this point, last char must be `i' */
  if (str[len-1] != 'i' && str[len-1] != 'I') return BOOL_F;
  /* handles `x+i' and `x-i' */
  if (i==(len-1))  return scm_makdbl(res, lead_sgn);
  /* get a `ureal' for complex part */
  second = scm_istr2flo(&str[i], (long)((len-i)-1), radix);
  if (!(INEXP(second))) return BOOL_F; /* not `ureal' */
  if (CPLXP(second))    return BOOL_F; /* not `ureal' */
  tmp = REALPART(second);
  if (tmp < 0.0)	return BOOL_F; /* not `ureal' */
  return scm_makdbl(res, (lead_sgn*tmp));
}
#endif				/* FLOATS */


#ifdef __STDC__
SCM
scm_istring2number(char *str, long len, long radix)
#else
SCM
scm_istring2number(str, len, radix)
     char *str;
     long len;
     long radix;
#endif
{
  int i = 0;
  char ex = 0;
  char ex_p = 0, rx_p = 0;	/* Only allow 1 exactness and 1 radix prefix */
  SCM res;
  if (len==1)
    if (*str=='+' || *str=='-') /* Catches lone `+' and `-' for speed */
      return BOOL_F;

  while ((len-i) >= 2  &&  str[i]=='#' && ++i)
    switch (str[i++]) {
    case 'b': case 'B':  if (rx_p++) return BOOL_F; radix = 2;  break;
    case 'o': case 'O':  if (rx_p++) return BOOL_F; radix = 8;  break;
    case 'd': case 'D':  if (rx_p++) return BOOL_F; radix = 10; break;
    case 'x': case 'X':  if (rx_p++) return BOOL_F; radix = 16; break;
    case 'i': case 'I':  if (ex_p++) return BOOL_F; ex = 2;     break;
    case 'e': case 'E':  if (ex_p++) return BOOL_F; ex = 1;     break;
    default:  return BOOL_F;
    }

  switch (ex) {
  case 1:
    return scm_istr2int(&str[i], len-i, radix);
  case 0:
    res = scm_istr2int(&str[i], len-i, radix);
    if NFALSEP(res) return res;
#ifdef FLOATS
  case 2: return scm_istr2flo(&str[i], len-i, radix);
#endif
  }
  return BOOL_F;
}


PROC (s_string_to_number, "string->number", 1, 1, 0, scm_string_to_number);
#ifdef __STDC__
SCM
scm_string_to_number(SCM str, SCM radix)
#else
SCM
scm_string_to_number(str, radix)
     SCM str;
     SCM radix;
#endif
{
  if UNBNDP(radix) radix=MAKINUM(10L);
  else ASSERT(INUMP(radix), radix, ARG2, s_string_to_number);
  ASSERT(NIMP(str) && STRINGP(str), str, ARG1, s_string_to_number);
  return scm_istring2number(CHARS(str), LENGTH(str), INUM(radix));
}
/*** END strs->nums ***/

#ifdef FLOATS
#ifdef __STDC__
SCM
scm_makdbl (double x, double y)
#else
SCM
scm_makdbl (x, y)
     double x;
     double y;
#endif
{
  SCM z;
  if ((y==0.0) && (x==0.0)) return flo0;
  NEWCELL(z);
  DEFER_INTS;
  if (y==0.0) {
# ifdef SINGLES
    float fx = x;
#  ifndef SINGLESONLY
    if ((-FLTMAX < x) && (x < FLTMAX) && (fx==x))
#  endif
      {
	CAR(z) = tc_flo;
	FLO(z) = x;
	ALLOW_INTS;
	return z;
      }
# endif/* def SINGLES */
    CDR(z) = (SCM)scm_must_malloc(1L*sizeof(double), "real");
    CAR(z) = tc_dblr;
  }
  else {
    CDR(z) = (SCM)scm_must_malloc(2L*sizeof(double), "complex");
    CAR(z) = tc_dblc;
    IMAG(z) = y;
  }
  REAL(z) = x;
  ALLOW_INTS;
  return z;
}
#endif


#ifdef __STDC__
SCM
scm_bigequal(SCM x, SCM y)
#else
SCM
scm_bigequal(x, y)
     SCM x;
     SCM y;
#endif
{
#ifdef BIGDIG
  if (0==scm_bigcomp(x, y)) return BOOL_T;
#endif
  return BOOL_F;
}


#ifdef __STDC__
SCM
scm_floequal(SCM x, SCM y)
#else
SCM
scm_floequal(x, y)
     SCM x;
     SCM y;
#endif
{
#ifdef FLOATS
  if (REALPART(x) != REALPART(y)) return BOOL_F;
  if (!(CPLXP(x) && (IMAG(x) != IMAG(y)))) return BOOL_T;
#endif
  return BOOL_F;
}




PROC (s_number_p, "number?", 1, 0, 0, scm_number_p);
PROC (s_complex_p, "complex?", 1, 0, 0, scm_number_p);
#ifdef __STDC__
SCM
scm_number_p(SCM x)
#else
SCM
scm_number_p(x)
     SCM x;
#endif
{
  if INUMP(x) return BOOL_T;
#ifdef FLOATS
  if (NIMP(x) && NUMP(x)) return BOOL_T;
#else
# ifdef BIGDIG
  if (NIMP(x) && NUMP(x)) return BOOL_T;
# endif
#endif
  return BOOL_F;
}



#ifdef FLOATS
PROC (s_real_p, "real?", 1, 0, 0, scm_real_p);
PROC (s_rational_p, "rational?", 1, 0, 0, scm_real_p);
#ifdef __STDC__
SCM
scm_real_p(SCM x)
#else
SCM
scm_real_p(x)
     SCM x;
#endif
{
  if (INUMP(x))
    return BOOL_T;
  if (IMP(x))
    return BOOL_F;
  if (REALP(x))
    return BOOL_T;
# ifdef BIGDIG
  if (BIGP(x))
    return BOOL_T;
# endif
  return BOOL_F;
}



PROC (s_int_p, "int?", 1, 0, 0, scm_int_p);
#ifdef __STDC__
SCM
scm_int_p(SCM x)
#else
SCM
scm_int_p(x)
     SCM x;
#endif
{
  double r;
  if INUMP(x) return BOOL_T;
  if IMP(x) return BOOL_F;
# ifdef BIGDIG
  if BIGP(x) return BOOL_T;
# endif
  if (!INEXP(x)) return BOOL_F;
  if CPLXP(x) return BOOL_F;
  r = REALPART(x);
  if (r==floor(r)) return BOOL_T;
  return BOOL_F;
}



#endif				/* FLOATS */

PROC (s_inexact_p, "inexact?", 1, 0, 0, scm_inexact_p);
#ifdef __STDC__
SCM
scm_inexact_p(SCM x)
#else
SCM
scm_inexact_p(x)
     SCM x;
#endif
{
#ifdef FLOATS
  if (NIMP(x) && INEXP(x)) return BOOL_T;
#endif
  return BOOL_F;
}




PROC1 (s_eq_p, "=?", tc7_rpsubr, scm_num_eq_p);
#ifdef __STDC__
SCM
scm_num_eq_p(SCM x, SCM y)
#else
SCM
scm_equal_p(x, y)
     SCM x;
     SCM y;
#endif
{
#ifdef FLOATS
  SCM t;
  if NINUMP(x) {
# ifdef BIGDIG
#  ifndef RECKLESS
    if (!(NIMP(x)))
    badx: scm_wta(x, (char *)ARG1, s_eq_p);
#  endif
    if BIGP(x) {
      if INUMP(y) return BOOL_F;
      ASRTGO(NIMP(y), bady);
      if BIGP(y) return (0==scm_bigcomp(x, y)) ? BOOL_T : BOOL_F;
      ASRTGO(INEXP(y), bady);
    bigreal:
      return (REALP(y) && (scm_big2dbl(x)==REALPART(y))) ? BOOL_T : BOOL_F;
    }
    ASRTGO(INEXP(x), badx);
# else
    ASSERT(NIMP(x) && INEXP(x), x, ARG1, s_eq_p);
# endif
    if INUMP(y) {t = x; x = y; y = t; goto realint;}
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(y) {t = x; x = y; y = t; goto bigreal;}
    ASRTGO(INEXP(y), bady);
# else
    ASRTGO(NIMP(y) && INEXP(y), bady);
# endif
    if (REALPART(x) != REALPART(y)) return BOOL_F;
    if CPLXP(x)
      return (CPLXP(y) && (IMAG(x)==IMAG(y))) ? BOOL_T : BOOL_F;
    return CPLXP(y) ? BOOL_F : BOOL_T;
  }
  if NINUMP(y) {
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(y) return BOOL_F;
#  ifndef RECKLESS
    if (!(INEXP(y)))
    bady: scm_wta(y, (char *)ARG2, s_eq_p);
#  endif
# else
#  ifndef RECKLESS
    if (!(NIMP(y) && INEXP(y)))
    bady: scm_wta(y, (char *)ARG2, s_eq_p);
#  endif
# endif
  realint:
    return (REALP(y) && (((double)INUM(x))==REALPART(y))) ? BOOL_T : BOOL_F;
  }
#else
# ifdef BIGDIG
  if NINUMP(x) {
    ASSERT(NIMP(x) && BIGP(x), x, ARG1, s_eq_p);
    if INUMP(y) return BOOL_F;
    ASRTGO(NIMP(y) && BIGP(y), bady);
    return (0==scm_bigcomp(x, y)) ? BOOL_T : BOOL_F;
  }
  if NINUMP(y) {
#  ifndef RECKLESS
    if (!(NIMP(y) && BIGP(y)))
    bady: scm_wta(y, (char *)ARG2, s_eq_p);
#  endif
    return BOOL_F;
  }
# else
  ASSERT(INUMP(x), x, ARG1, s_eq_p);
  ASSERT(INUMP(y), y, ARG2, s_eq_p);
# endif
#endif
  return ((long)x==(long)y) ? BOOL_T : BOOL_F;
}



PROC1 (s_less_p, "<?", tc7_rpsubr, scm_less_p);
#ifdef __STDC__
SCM
scm_less_p(SCM x, SCM y)
#else
SCM
scm_less_p(x, y)
     SCM x;
     SCM y;
#endif
{
#ifdef FLOATS
  if NINUMP(x) {
# ifdef BIGDIG
#  ifndef RECKLESS
    if (!(NIMP(x)))
    badx: scm_wta(x, (char *)ARG1, s_less_p);
#  endif
    if BIGP(x) {
      if INUMP(y) return BIGSIGN(x) ? BOOL_T : BOOL_F;
      ASRTGO(NIMP(y), bady);
      if BIGP(y) return (1==scm_bigcomp(x, y)) ? BOOL_T : BOOL_F;
      ASRTGO(REALP(y), bady);
      return (scm_big2dbl(x) < REALPART(y)) ? BOOL_T : BOOL_F;
    }
    ASRTGO(REALP(x), badx);
# else
    ASSERT(NIMP(x) && REALP(x), x, ARG1, s_less_p);
# endif
    if (INUMP(y))
      return (REALPART(x) < ((double)INUM(y))) ? BOOL_T : BOOL_F;
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(y) return (REALPART(x) < scm_big2dbl(y)) ? BOOL_T : BOOL_F;
    ASRTGO(REALP(y), bady);
# else
    ASRTGO(NIMP(y) && REALP(y), bady);
# endif
    return (REALPART(x) < REALPART(y)) ? BOOL_T : BOOL_F;
  }
  if NINUMP(y) {
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(y) return BIGSIGN(y) ? BOOL_F : BOOL_T;
#  ifndef RECKLESS
    if (!(REALP(y)))
    bady: scm_wta(y, (char *)ARG2, s_less_p);
#  endif
# else
#  ifndef RECKLESS
    if (!(NIMP(y) && REALP(y)))
    bady: scm_wta(y, (char *)ARG2, s_less_p);
#  endif
# endif
    return (((double)INUM(x)) < REALPART(y)) ? BOOL_T : BOOL_F;
  }
#else
# ifdef BIGDIG
  if NINUMP(x) {
    ASSERT(NIMP(x) && BIGP(x), x, ARG1, s_less_p);
    if INUMP(y) return BIGSIGN(x) ? BOOL_T : BOOL_F;
    ASRTGO(NIMP(y) && BIGP(y), bady);
    return (1==scm_bigcomp(x, y)) ? BOOL_T : BOOL_F;
  }
  if NINUMP(y) {
#  ifndef RECKLESS
    if (!(NIMP(y) && BIGP(y)))
    bady: scm_wta(y, (char *)ARG2, s_less_p);
#  endif
    return BIGSIGN(y) ? BOOL_F : BOOL_T;
  }
# else
  ASSERT(INUMP(x), x, ARG1, s_less_p);
  ASSERT(INUMP(y), y, ARG2, s_less_p);
# endif
#endif
  return ((long)x < (long)y) ? BOOL_T : BOOL_F;
}


PROC1 (s_gr_p, ">?", tc7_rpsubr, scm_gr_p);
#ifdef __STDC__
SCM
scm_gr_p(SCM x, SCM y)
#else
SCM
scm_gr_p(x, y)
     SCM x;
     SCM y;
#endif
{
  return scm_less_p(y, x);
}



PROC1 (s_leq_p, "<=?", tc7_rpsubr, scm_leq_p);
#ifdef __STDC__
SCM
scm_leq_p(SCM x, SCM y)
#else
SCM
scm_leq_p(x, y)
     SCM x;
     SCM y;
#endif
{
  return BOOL_NOT(scm_less_p(y, x));
}



PROC1 (s_geq_p, ">=?", tc7_rpsubr, scm_geq_p);
#ifdef __STDC__
SCM
scm_geq_p(SCM x, SCM y)
#else
SCM
scm_geq_p(x, y)
     SCM x;
     SCM y;
#endif
{
  return BOOL_NOT(scm_less_p(x, y));
}



PROC (s_zero_p, "zero?", 1, 0, 0, scm_zero_p);
#ifdef __STDC__
SCM
scm_zero_p(SCM z)
#else
SCM
scm_zero_p(z)
     SCM z;
#endif
{
#ifdef FLOATS
  if NINUMP(z) {
# ifdef BIGDIG
    ASRTGO(NIMP(z), badz);
    if BIGP(z) return BOOL_F;
#  ifndef RECKLESS
    if (!(INEXP(z)))
    badz: scm_wta(z, (char *)ARG1, s_zero_p);
#  endif
# else
    ASSERT(NIMP(z) && INEXP(z), z, ARG1, s_zero_p);
# endif
    return (z==flo0) ? BOOL_T : BOOL_F;
  }
#else
# ifdef BIGDIG
  if NINUMP(z) {
    ASSERT(NIMP(z) && BIGP(z), z, ARG1, s_zero_p);
    return BOOL_F;
  }
# else
  ASSERT(INUMP(z), z, ARG1, s_zero_p);
# endif
#endif
  return (z==INUM0) ? BOOL_T: BOOL_F;
}



PROC (s_positive_p, "positive?", 1, 0, 0, scm_positive_p);
#ifdef __STDC__
SCM
scm_positive_p(SCM x)
#else
SCM
scm_positive_p(x)
     SCM x;
#endif
{
#ifdef FLOATS
  if NINUMP(x) {
# ifdef BIGDIG
    ASRTGO(NIMP(x), badx);
    if BIGP(x) return TYP16(x)==tc16_bigpos ? BOOL_T : BOOL_F;
#  ifndef RECKLESS
    if (!(REALP(x)))
    badx: scm_wta(x, (char *)ARG1, s_positive_p);
#  endif
# else
    ASSERT(NIMP(x) && REALP(x), x, ARG1, s_positive_p);
# endif
    return (REALPART(x) > 0.0) ? BOOL_T : BOOL_F;
  }
#else
# ifdef BIGDIG
  if NINUMP(x) {
    ASSERT(NIMP(x) && BIGP(x), x, ARG1, s_positive_p);
    return TYP16(x)==tc16_bigpos ? BOOL_T : BOOL_F;
  }
# else
  ASSERT(INUMP(x), x, ARG1, s_positive_p);
# endif
#endif
  return (x > INUM0) ? BOOL_T : BOOL_F;
}



PROC (s_negative_p, "negative?", 1, 0, 0, scm_negative_p);
#ifdef __STDC__
SCM
scm_negative_p(SCM x)
#else
SCM
scm_negative_p(x)
     SCM x;
#endif
{
#ifdef FLOATS
  if NINUMP(x) {
# ifdef BIGDIG
    ASRTGO(NIMP(x), badx);
    if BIGP(x) return TYP16(x)==tc16_bigpos ? BOOL_F : BOOL_T;
#  ifndef RECKLESS
    if (!(REALP(x)))
    badx: scm_wta(x, (char *)ARG1, s_negative_p);
#  endif
# else
    ASSERT(NIMP(x) && REALP(x), x, ARG1, s_negative_p);
# endif
    return (REALPART(x) < 0.0) ? BOOL_T : BOOL_F;
  }
#else
# ifdef BIGDIG
  if NINUMP(x) {
    ASSERT(NIMP(x) && BIGP(x), x, ARG1, s_negative_p);
    return (TYP16(x)==tc16_bigneg) ? BOOL_T : BOOL_F;
  }
# else
  ASSERT(INUMP(x), x, ARG1, s_negative_p);
# endif
#endif
  return (x < INUM0) ? BOOL_T : BOOL_F;
}


PROC1 (s_max, "max", tc7_asubr, scm_max);
#ifdef __STDC__
SCM
scm_max(SCM x, SCM y)
#else
SCM
scm_max(x, y)
     SCM x;
     SCM y;
#endif
{
#ifdef FLOATS
  double z;
#endif
  if UNBNDP(y) {
#ifndef RECKLESS
    if (!(NUMBERP(x)))
      badx: scm_wta(x, (char *)ARG1, s_max);
#endif
    return x;
  }
#ifdef FLOATS
  if NINUMP(x) {
# ifdef BIGDIG
    ASRTGO(NIMP(x), badx);
    if BIGP(x) {
      if INUMP(y) return BIGSIGN(x) ? y : x;
      ASRTGO(NIMP(y), bady);
      if BIGP(y) return (1==scm_bigcomp(x, y)) ? y : x;
      ASRTGO(REALP(y), bady);
      z = scm_big2dbl(x);
      return (z < REALPART(y)) ? y : scm_makdbl(z, 0.0);
    }
    ASRTGO(REALP(x), badx);
# else
    ASSERT(NIMP(x) && REALP(x), x, ARG1, s_max);
# endif
    if (INUMP(y))
      return (REALPART(x) < (z = INUM(y))) ? scm_makdbl(z, 0.0) : x;
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if (BIGP(y))
      return (REALPART(x) < (z = scm_big2dbl(y))) ? scm_makdbl(z, 0.0) : x;
    ASRTGO(REALP(y), bady);
# else
    ASRTGO(NIMP(y) && REALP(y), bady);
# endif
    return (REALPART(x) < REALPART(y)) ? y : x;
  }
  if NINUMP(y) {
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(y) return BIGSIGN(y) ? x : y;
#  ifndef RECKLESS
    if (!(REALP(y)))
    bady: scm_wta(y, (char *)ARG2, s_max);
#  endif
# else
#  ifndef RECKLESS
    if (!(NIMP(y) && REALP(y)))
    bady: scm_wta(y, (char *)ARG2, s_max);
#  endif
# endif
    return ((z = INUM(x)) < REALPART(y)) ? y : scm_makdbl(z, 0.0);
  }
#else
# ifdef BIGDIG
  if NINUMP(x) {
    ASSERT(NIMP(x) && BIGP(x), x, ARG1, s_max);
    if INUMP(y) return BIGSIGN(x) ? y : x;
    ASRTGO(NIMP(y) && BIGP(y), bady);
    return (1==scm_bigcomp(x, y)) ? y : x;
  }
  if NINUMP(y) {
#  ifndef RECKLESS
    if (!(NIMP(y) && BIGP(y)))
    bady: scm_wta(y, (char *)ARG2, s_max);
#  endif
    return BIGSIGN(y) ? x : y;
  }
# else
  ASSERT(INUMP(x), x, ARG1, s_max);
  ASSERT(INUMP(y), y, ARG2, s_max);
# endif
#endif
  return ((long)x < (long)y) ? y : x;
}




PROC1 (s_min, "min", tc7_asubr, scm_min);
#ifdef __STDC__
SCM
scm_min(SCM x, SCM y)
#else
SCM
scm_min(x, y)
     SCM x;
     SCM y;
#endif
{
#ifdef FLOATS
  double z;
#endif
  if UNBNDP(y) {
#ifndef RECKLESS
    if (!(NUMBERP(x)))
      badx:scm_wta(x, (char *)ARG1, s_min);
#endif
    return x;
  }
#ifdef FLOATS
  if NINUMP(x) {
# ifdef BIGDIG
    ASRTGO(NIMP(x), badx);
    if BIGP(x) {
      if INUMP(y) return BIGSIGN(x) ? x : y;
      ASRTGO(NIMP(y), bady);
      if BIGP(y) return (-1==scm_bigcomp(x, y)) ? y : x;
      ASRTGO(REALP(y), bady);
      z = scm_big2dbl(x);
      return (z > REALPART(y)) ? y : scm_makdbl(z, 0.0);
    }
    ASRTGO(REALP(x), badx);
# else
    ASSERT(NIMP(x) && REALP(x), x, ARG1, s_min);
# endif
    if INUMP(y) return (REALPART(x) > (z = INUM(y))) ? scm_makdbl(z, 0.0) : x;
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(y) return (REALPART(x) > (z = scm_big2dbl(y))) ? scm_makdbl(z, 0.0) : x;
    ASRTGO(REALP(y), bady);
# else
    ASRTGO(NIMP(y) && REALP(y), bady);
# endif
    return (REALPART(x) > REALPART(y)) ? y : x;
  }
  if NINUMP(y) {
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(y) return BIGSIGN(y) ? y : x;
#  ifndef RECKLESS
    if (!(REALP(y)))
    bady: scm_wta(y, (char *)ARG2, s_min);
#  endif
# else
#  ifndef RECKLESS
    if (!(NIMP(y) && REALP(y)))
    bady: scm_wta(y, (char *)ARG2, s_min);
#  endif
# endif
    return ((z = INUM(x)) > REALPART(y)) ? y : scm_makdbl(z, 0.0);
  }
#else
# ifdef BIGDIG
  if NINUMP(x) {
    ASSERT(NIMP(x) && BIGP(x), x, ARG1, s_min);
    if INUMP(y) return BIGSIGN(x) ? x : y;
    ASRTGO(NIMP(y) && BIGP(y), bady);
    return (-1==scm_bigcomp(x, y)) ? y : x;
  }
  if NINUMP(y) {
#  ifndef RECKLESS
    if (!(NIMP(y) && BIGP(y)))
    bady: scm_wta(y, (char *)ARG2, s_min);
#  endif
    return BIGSIGN(y) ? y : x;
  }
# else
  ASSERT(INUMP(x), x, ARG1, s_min);
  ASSERT(INUMP(y), y, ARG2, s_min);
# endif
#endif
  return ((long)x > (long)y) ? y : x;
}




PROC1 (s_sum, "+", tc7_asubr, scm_sum);
#ifdef __STDC__
SCM
scm_sum(SCM x, SCM y)
#else
SCM
scm_sum(x, y)
     SCM x;
     SCM y;
#endif
{
  if UNBNDP(y) {
    if UNBNDP(x) return INUM0;
#ifndef RECKLESS
    if (!(NUMBERP(x)))
    badx: scm_wta(x, (char *)ARG1, s_sum);
#endif
    return x;
  }
#ifdef FLOATS
  if NINUMP(x) {
    SCM t;
# ifdef BIGDIG
    ASRTGO(NIMP(x), badx);
    if BIGP(x) {
      if INUMP(y) {t = x; x = y; y = t; goto intbig;}
      ASRTGO(NIMP(y), bady);
      if BIGP(y) {
	if (NUMDIGS(x) > NUMDIGS(y)) {t = x; x = y; y = t;}
	return scm_addbig(BDIGITS(x), NUMDIGS(x), BIGSIGN(x), y, 0);
      }
      ASRTGO(INEXP(y), bady);
    bigreal: return scm_makdbl(scm_big2dbl(x)+REALPART(y), CPLXP(y)?IMAG(y):0.0);
    }
    ASRTGO(INEXP(x), badx);
# else
    ASRTGO(NIMP(x) && INEXP(x), badx);
# endif
    if INUMP(y) {t = x; x = y; y = t; goto intreal;}
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(y) {t = x; x = y; y = t; goto bigreal;}
#  ifndef RECKLESS
    else if (!(INEXP(y)))
    bady: scm_wta(y, (char *)ARG2, s_sum);
#  endif
# else
#  ifndef RECKLESS
    if (!(NIMP(y) && INEXP(y)))
    bady: scm_wta(y, (char *)ARG2, s_sum);
#  endif
# endif
    { double i = 0.0;
      if CPLXP(x) i = IMAG(x);
      if CPLXP(y) i += IMAG(y);
      return scm_makdbl(REALPART(x)+REALPART(y), i); }
  }
  if NINUMP(y) {
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(y)
    intbig: {
#  ifndef DIGSTOOBIG
      long z = scm_pseudolong(INUM(x));
      return scm_addbig((BIGDIG *)&z, DIGSPERLONG, (x < 0) ? 0x0100 : 0, y, 0);
#  else
      BIGDIG zdigs[DIGSPERLONG];
      scm_longdigs(INUM(x), zdigs);
      return scm_addbig(zdigs, DIGSPERLONG, (x < 0) ? 0x0100 : 0, y, 0);
#  endif
    }
    ASRTGO(INEXP(y), bady);
# else
    ASRTGO(NIMP(y) && INEXP(y), bady);
# endif
  intreal: return scm_makdbl(INUM(x)+REALPART(y), CPLXP(y)?IMAG(y):0.0);
  }
#else
# ifdef BIGDIG
  if NINUMP(x) {
    SCM t;
    ASRTGO(NIMP(x) && BIGP(x), badx);
    if INUMP(y) {t = x; x = y; y = t; goto intbig;}
    ASRTGO(NIMP(y) && BIGP(y), bady);
    if (NUMDIGS(x) > NUMDIGS(y)) {t = x; x = y; y = t;}
    return scm_addbig(BDIGITS(x), NUMDIGS(x), BIGSIGN(x), y, 0);
  }
  if NINUMP(y) {
#  ifndef RECKLESS
    if (!(NIMP(y) && BIGP(y)))
    bady: scm_wta(y, (char *)ARG2, s_sum);
#  endif
  intbig: {
#  ifndef DIGSTOOBIG
    long z = scm_pseudolong(INUM(x));
    return scm_addbig(&z, DIGSPERLONG, (x < 0) ? 0x0100 : 0, y, 0);
#  else
    BIGDIG zdigs[DIGSPERLONG];
    scm_longdigs(INUM(x), zdigs);
    return scm_addbig(zdigs, DIGSPERLONG, (x < 0) ? 0x0100 : 0, y, 0);
#  endif
  }
  }
# else
  ASRTGO(INUMP(x), badx);
  ASSERT(INUMP(y), y, ARG2, s_sum);
# endif
#endif
  x = INUM(x)+INUM(y);
  if FIXABLE(x) return MAKINUM(x);
#ifdef BIGDIG
  return scm_long2big(x);
#else
# ifdef FLOATS
  return scm_makdbl((double)x, 0.0);
# else
  scm_wta(y, (char *)OVFLOW, s_sum);
  return UNSPECIFIED;
# endif
#endif
}




PROC1 (s_difference, "-", tc7_asubr, scm_difference);
#ifdef __STDC__
SCM
scm_difference(SCM x, SCM y)
#else
SCM
scm_difference(x, y)
     SCM x;
     SCM y;
#endif
{
#ifdef FLOATS
  if NINUMP(x) {
# ifndef RECKLESS
    if (!(NIMP(x)))
    badx: scm_wta(x, (char *)ARG1, s_difference);
# endif
    if UNBNDP(y) {
# ifdef BIGDIG
      if BIGP(x) {
	x = scm_copybig(x, !BIGSIGN(x));
	return NUMDIGS(x) * BITSPERDIG/CHAR_BIT <= sizeof(SCM) ?
	  scm_big2inum(x, NUMDIGS(x)) : x;
      }
# endif
      ASRTGO(INEXP(x), badx);
      return scm_makdbl(-REALPART(x), CPLXP(x)?-IMAG(x):0.0);
    }
    if INUMP(y) return scm_sum(x, MAKINUM(-INUM(y)));
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(x) {
      if BIGP(y) return (NUMDIGS(x) < NUMDIGS(y)) ?
	scm_addbig(BDIGITS(x), NUMDIGS(x), BIGSIGN(x), y, 0x0100) :
      scm_addbig(BDIGITS(y), NUMDIGS(y), BIGSIGN(y) ^ 0x0100, x, 0);
      ASRTGO(INEXP(y), bady);
      return scm_makdbl(scm_big2dbl(x)-REALPART(y), CPLXP(y)?-IMAG(y):0.0);
    }
    ASRTGO(INEXP(x), badx);
    if BIGP(y) return scm_makdbl(REALPART(x)-scm_big2dbl(y), CPLXP(x)?IMAG(x):0.0);
    ASRTGO(INEXP(y), bady);
# else
    ASRTGO(INEXP(x), badx);
    ASRTGO(NIMP(y) && INEXP(y), bady);
# endif
    if CPLXP(x)
      if CPLXP(y)
	return scm_makdbl(REAL(x)-REAL(y), IMAG(x)-IMAG(y));
      else
	return scm_makdbl(REAL(x)-REALPART(y), IMAG(x));
    return scm_makdbl(REALPART(x)-REALPART(y), CPLXP(y)?-IMAG(y):0.0);
  }
  if UNBNDP(y) {x = -INUM(x); goto checkx;}
  if NINUMP(y) {
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(y) {
#  ifndef DIGSTOOBIG
      long z = scm_pseudolong(INUM(x));
      return scm_addbig((BIGDIG *)&z, DIGSPERLONG, (x < 0) ? 0x0100 : 0, y, 0x0100);
#  else
      BIGDIG zdigs[DIGSPERLONG];
      scm_longdigs(INUM(x), zdigs);
      return scm_addbig(zdigs, DIGSPERLONG, (x < 0) ? 0x0100 : 0, y, 0x0100);
#  endif
    }
#  ifndef RECKLESS
    if (!(INEXP(y)))
    bady: scm_wta(y, (char *)ARG2, s_difference);
#  endif
# else
#  ifndef RECKLESS
    if (!(NIMP(y) && INEXP(y)))
    bady: scm_wta(y, (char *)ARG2, s_difference);
#  endif
# endif
    return scm_makdbl(INUM(x)-REALPART(y), CPLXP(y)?-IMAG(y):0.0);
  }
#else
# ifdef BIGDIG
  if NINUMP(x) {
    ASSERT(NIMP(x) && BIGP(x), x, ARG1, s_difference);
    if UNBNDP(y) {
      x = scm_copybig(x, !BIGSIGN(x));
      return NUMDIGS(x) * BITSPERDIG/CHAR_BIT <= sizeof(SCM) ?
	scm_big2inum(x, NUMDIGS(x)) : x;
    }
    if INUMP(y) {
#  ifndef DIGSTOOBIG
      long z = scm_pseudolong(INUM(y));
      return scm_addbig(&z, DIGSPERLONG, (y < 0) ? 0 : 0x0100, x, 0);
#  else
      BIGDIG zdigs[DIGSPERLONG];
      scm_longdigs(INUM(x), zdigs);
      return scm_addbig(zdigs, DIGSPERLONG, (y < 0) ? 0 : 0x0100, x, 0);
#  endif
    }
    ASRTGO(NIMP(y) && BIGP(y), bady);
    return (NUMDIGS(x) < NUMDIGS(y)) ?
      scm_addbig(BDIGITS(x), NUMDIGS(x), BIGSIGN(x), y, 0x0100) :
    scm_addbig(BDIGITS(y), NUMDIGS(y), BIGSIGN(y) ^ 0x0100, x, 0);
  }
  if UNBNDP(y) {x = -INUM(x); goto checkx;}
  if NINUMP(y) {
#  ifndef RECKLESS
    if (!(NIMP(y) && BIGP(y)))
    bady: scm_wta(y, (char *)ARG2, s_difference);
#  endif
    {
#  ifndef DIGSTOOBIG
      long z = scm_pseudolong(INUM(x));
      return scm_addbig(&z, DIGSPERLONG, (x < 0) ? 0x0100 : 0, y, 0x0100);
#  else
      BIGDIG zdigs[DIGSPERLONG];
      scm_longdigs(INUM(x), zdigs);
      return scm_addbig(zdigs, DIGSPERLONG, (x < 0) ? 0x0100 : 0, y, 0x0100);
#  endif
    }
  }
# else
  ASSERT(INUMP(x), x, ARG1, s_difference);
  if UNBNDP(y) {x = -INUM(x); goto checkx;}
  ASSERT(INUMP(y), y, ARG2, s_difference);
# endif
#endif
  x = INUM(x)-INUM(y);
 checkx:
  if FIXABLE(x) return MAKINUM(x);
#ifdef BIGDIG
  return scm_long2big(x);
#else
# ifdef FLOATS
  return scm_makdbl((double)x, 0.0);
# else
  scm_wta(y, (char *)OVFLOW, s_difference);
  return UNSPECIFIED;
# endif
#endif
}




PROC1 (s_product, "*", tc7_asubr, scm_product);
#ifdef __STDC__
SCM
scm_product(SCM x, SCM y)
#else
SCM
scm_product(x, y)
     SCM x;
     SCM y;
#endif
{
  if UNBNDP(y) {
    if UNBNDP(x) return MAKINUM(1L);
#ifndef RECKLESS
    if (!(NUMBERP(x)))
    badx: scm_wta(x, (char *)ARG1, s_product);
#endif
    return x;
  }
#ifdef FLOATS
  if NINUMP(x) {
    SCM t;
# ifdef BIGDIG
    ASRTGO(NIMP(x), badx);
    if BIGP(x) {
      if INUMP(y) {t = x; x = y; y = t; goto intbig;}
      ASRTGO(NIMP(y), bady);
      if BIGP(y) return scm_mulbig(BDIGITS(x), NUMDIGS(x), BDIGITS(y), NUMDIGS(y),
				   BIGSIGN(x) ^ BIGSIGN(y));
      ASRTGO(INEXP(y), bady);
    bigreal: {
      double bg = scm_big2dbl(x);
      return scm_makdbl(bg*REALPART(y), CPLXP(y)?bg*IMAG(y):0.0); }
    }
    ASRTGO(INEXP(x), badx);
# else
    ASRTGO(NIMP(x) && INEXP(x), badx);
# endif
    if INUMP(y) {t = x; x = y; y = t; goto intreal;}
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(y) {t = x; x = y; y = t; goto bigreal;}
#  ifndef RECKLESS
    else if (!(INEXP(y)))
    bady: scm_wta(y, (char *)ARG2, s_product);
#  endif
# else
#  ifndef RECKLESS
    if (!(NIMP(y) && INEXP(y)))
    bady: scm_wta(y, (char *)ARG2, s_product);
#  endif
# endif
    if CPLXP(x)
      if CPLXP(y)
	return scm_makdbl(REAL(x)*REAL(y)-IMAG(x)*IMAG(y),
			  REAL(x)*IMAG(y)+IMAG(x)*REAL(y));
      else
	return scm_makdbl(REAL(x)*REALPART(y), IMAG(x)*REALPART(y));
    return scm_makdbl(REALPART(x)*REALPART(y),
		      CPLXP(y)?REALPART(x)*IMAG(y):0.0);
  }
  if NINUMP(y) {
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(y) {
    intbig: if (INUM0==x) return x; if (MAKINUM(1L)==x) return y;
      {
#  ifndef DIGSTOOBIG
	long z = scm_pseudolong(INUM(x));
	return scm_mulbig((BIGDIG *)&z, DIGSPERLONG, BDIGITS(y), NUMDIGS(y),
			  BIGSIGN(y) ? (x>0) : (x<0));
#  else
	BIGDIG zdigs[DIGSPERLONG];
	scm_longdigs(INUM(x), zdigs);
	return scm_mulbig(zdigs, DIGSPERLONG, BDIGITS(y), NUMDIGS(y),
			  BIGSIGN(y) ? (x>0) : (x<0));
#  endif
      }
    }
    ASRTGO(INEXP(y), bady);
# else
    ASRTGO(NIMP(y) && INEXP(y), bady);
# endif
  intreal: return scm_makdbl(INUM(x)*REALPART(y), CPLXP(y)?INUM(x)*IMAG(y):0.0);
  }
#else
# ifdef BIGDIG
  if NINUMP(x) {
    ASRTGO(NIMP(x) && BIGP(x), badx);
    if INUMP(y) {SCM t = x; x = y; y = t; goto intbig;}
    ASRTGO(NIMP(y) && BIGP(y), bady);
    return scm_mulbig(BDIGITS(x), NUMDIGS(x), BDIGITS(y), NUMDIGS(y),
		      BIGSIGN(x) ^ BIGSIGN(y));
  }
  if NINUMP(y) {
#  ifndef RECKLESS
    if (!(NIMP(y) && BIGP(y)))
    bady: scm_wta(y, (char *)ARG2, s_product);
#  endif
  intbig: if (INUM0==x) return x; if (MAKINUM(1L)==x) return y;
    {
#  ifndef DIGSTOOBIG
      long z = scm_pseudolong(INUM(x));
      return scm_mulbig(&z, DIGSPERLONG, BDIGITS(y), NUMDIGS(y),
			BIGSIGN(y) ? (x>0) : (x<0));
#  else
      BIGDIG zdigs[DIGSPERLONG];
      scm_longdigs(INUM(x), zdigs);
      return scm_mulbig(zdigs, DIGSPERLONG, BDIGITS(y), NUMDIGS(y),
			BIGSIGN(y) ? (x>0) : (x<0));
#  endif
    }
  }
# else
  ASRTGO(INUMP(x), badx);
  ASSERT(INUMP(y), y, ARG2, s_product);
# endif
#endif
  {
    long i, j, k;
    i = INUM(x);
    if (0==i) return x;
    j = INUM(y);
    k = i * j;
    y = MAKINUM(k);
    if (k != INUM(y) || k/i != j)
#ifdef BIGDIG
      { int sgn = (i < 0) ^ (j < 0);
# ifndef DIGSTOOBIG
	i = scm_pseudolong(i);
	j = scm_pseudolong(j);
	return scm_mulbig((BIGDIG *)&i, DIGSPERLONG,
			  (BIGDIG *)&j, DIGSPERLONG, sgn);
# else /* DIGSTOOBIG */
	BIGDIG idigs[DIGSPERLONG];
	BIGDIG jdigs[DIGSPERLONG];
	scm_longdigs(i, idigs);
	scm_longdigs(j, jdigs);
	return scm_mulbig(idigs, DIGSPERLONG, jdigs, DIGSPERLONG, sgn);
# endif
      }
#else
# ifdef FLOATS
    return scm_makdbl(((double)i)*((double)j), 0.0);
# else
    scm_wta(y, (char *)OVFLOW, s_product);
# endif
#endif
    return y;
  }
}


#ifdef __STDC__
double
scm_num2dbl (SCM a, char * why)
#else
double
scm_num2dbl (a, why)
     SCM a;
     char * why;
#endif
{
  if (INUMP (a))
    return (double) INUM (a);
#ifdef FLOATS
  ASSERT (NIMP (a), a, "wrong type argument", why);
  if (REALP (a))
    return (REALPART (a));
#endif
#ifdef BIGDIG
  return scm_big2dbl (a);
#endif
  ASSERT (0, a, "wrong type argument", why);
  return UNSPECIFIED;
}


PROC (s_fuck, "fuck", 1, 0, 0, scm_fuck);
#ifdef __STDC__
SCM
scm_fuck (SCM a)
#else
SCM
scm_fuck (a)
     SCM a;
#endif
{
  return scm_makdbl (scm_num2dbl (a, "just because"), 0.0);
}

PROC1 (s_divide, "/", tc7_asubr, scm_divide);
#ifdef __STDC__
SCM
scm_divide(SCM x, SCM y)
#else
SCM
scm_divide(x, y)
     SCM x;
     SCM y;
#endif
{
#ifdef FLOATS
  double d, r, i, a;
  if NINUMP(x) {
# ifndef RECKLESS
    if (!(NIMP(x)))
    badx: scm_wta(x, (char *)ARG1, s_divide);
# endif
    if UNBNDP(y) {
# ifdef BIGDIG
      if BIGP(x) return scm_makdbl(1.0/scm_big2dbl(x), 0.0);
# endif
      ASRTGO(INEXP(x), badx);
      if REALP(x) return scm_makdbl(1.0/REALPART(x), 0.0);
      r = REAL(x);  i = IMAG(x);  d = r*r+i*i;
      return scm_makdbl(r/d, -i/d);
    }
# ifdef BIGDIG
    if BIGP(x) {
      SCM z;
      if INUMP(y) {
        z = INUM(y);
        ASSERT(z, y, OVFLOW, s_divide);
	if (1==z) return x;
        if (z < 0) z = -z;
        if (z < BIGRAD) {
          SCM w = scm_copybig(x, BIGSIGN(x) ? (y>0) : (y<0));
          return scm_divbigdig(BDIGITS(w), NUMDIGS(w), (BIGDIG)z) ?
	    scm_makdbl(scm_big2dbl(x)/INUM(y), 0.0) : scm_normbig(w);
	}
#  ifndef DIGSTOOBIG
        z = scm_pseudolong(z);
        z = scm_divbigbig(BDIGITS(x), NUMDIGS(x), (BIGDIG *)&z, DIGSPERLONG,
			  BIGSIGN(x) ? (y>0) : (y<0), 3);
#  else
	{ BIGDIG zdigs[DIGSPERLONG];
	  scm_longdigs(z, zdigs);
	  z = scm_divbigbig(BDIGITS(x), NUMDIGS(x), zdigs, DIGSPERLONG,
			    BIGSIGN(x) ? (y>0) : (y<0), 3);}
#  endif
        return z ? z : scm_makdbl(scm_big2dbl(x)/INUM(y), 0.0);
      }
      ASRTGO(NIMP(y), bady);
      if BIGP(y) {
	z = scm_divbigbig(BDIGITS(x), NUMDIGS(x), BDIGITS(y), NUMDIGS(y),
			  BIGSIGN(x) ^ BIGSIGN(y), 3);
	return z ? z : scm_makdbl(scm_big2dbl(x)/scm_big2dbl(y), 0.0);
      }
      ASRTGO(INEXP(y), bady);
      if REALP(y) return scm_makdbl(scm_big2dbl(x)/REALPART(y), 0.0);
      a = scm_big2dbl(x);
      goto complex_div;
    }
# endif
    ASRTGO(INEXP(x), badx);
    if INUMP(y) {d = INUM(y); goto basic_div;}
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(y) {d = scm_big2dbl(y); goto basic_div;}
    ASRTGO(INEXP(y), bady);
# else
    ASRTGO(NIMP(y) && INEXP(y), bady);
# endif
    if REALP(y) {
      d = REALPART(y);
    basic_div: return scm_makdbl(REALPART(x)/d, CPLXP(x)?IMAG(x)/d:0.0);
    }
    a = REALPART(x);
    if REALP(x) goto complex_div;
    r = REAL(y);  i = IMAG(y);  d = r*r+i*i;
    return scm_makdbl((a*r+IMAG(x)*i)/d, (IMAG(x)*r-a*i)/d);
  }
  if UNBNDP(y) {
    if ((MAKINUM(1L)==x) || (MAKINUM(-1L)==x)) return x;
    return scm_makdbl(1.0/((double)INUM(x)), 0.0);
  }
  if NINUMP(y) {
# ifdef BIGDIG
    ASRTGO(NIMP(y), bady);
    if BIGP(y) return scm_makdbl(INUM(x)/scm_big2dbl(y), 0.0);
#  ifndef RECKLESS
    if (!(INEXP(y)))
    bady: scm_wta(y, (char *)ARG2, s_divide);
#  endif
# else
#  ifndef RECKLESS
    if (!(NIMP(y) && INEXP(y)))
    bady: scm_wta(y, (char *)ARG2, s_divide);
#  endif
# endif
    if (REALP(y))
      return scm_makdbl(INUM(x)/REALPART(y), 0.0);
    a = INUM(x);
  complex_div:
    r = REAL(y);  i = IMAG(y);  d = r*r+i*i;
    return scm_makdbl((a*r)/d, (-a*i)/d);
  }
#else
# ifdef BIGDIG
  if NINUMP(x) {
    SCM z;
    ASSERT(NIMP(x) && BIGP(x), x, ARG1, s_divide);
    if UNBNDP(y) goto ov;
    if INUMP(y) {
      z = INUM(y);
      if (!z) goto ov;
      if (1==z) return x;
      if (z < 0) z = -z;
      if (z < BIGRAD) {
        SCM w = scm_copybig(x, BIGSIGN(x) ? (y>0) : (y<0));
        if (scm_divbigdig(BDIGITS(w), NUMDIGS(w), (BIGDIG)z)) goto ov;
        return w;
      }
#  ifndef DIGSTOOBIG
      z = scm_pseudolong(z);
      z = scm_divbigbig(BDIGITS(x), NUMDIGS(x), &z, DIGSPERLONG,
			BIGSIGN(x) ? (y>0) : (y<0), 3);
#  else
      { BIGDIG zdigs[DIGSPERLONG];
	scm_longdigs(z, zdigs);
	z = scm_divbigbig(BDIGITS(x), NUMDIGS(x), zdigs, DIGSPERLONG,
			  BIGSIGN(x) ? (y>0) : (y<0), 3);}
#  endif
    } else {
      ASRTGO(NIMP(y) && BIGP(y), bady);
      z = scm_divbigbig(BDIGITS(x), NUMDIGS(x), BDIGITS(y), NUMDIGS(y),
			BIGSIGN(x) ^ BIGSIGN(y), 3);
    }
    if (!z) goto ov;
    return z;
  }
  if UNBNDP(y) {
    if ((MAKINUM(1L)==x) || (MAKINUM(-1L)==x)) return x;
    goto ov;
  }
  if NINUMP(y) {
#  ifndef RECKLESS
    if (!(NIMP(y) && BIGP(y)))
    bady: scm_wta(y, (char *)ARG2, s_divide);
#  endif
    goto ov;
  }
# else
  ASSERT(INUMP(x), x, ARG1, s_divide);
  if UNBNDP(y) {
    if ((MAKINUM(1L)==x) || (MAKINUM(-1L)==x)) return x;
    goto ov;
  }
  ASSERT(INUMP(y), y, ARG2, s_divide);
# endif
#endif
  {
    long z = INUM(y);
    if ((0==z) || INUM(x)%z) goto ov;
    z = INUM(x)/z;
    if FIXABLE(z) return MAKINUM(z);
#ifdef BIGDIG
    return scm_long2big(z);
#endif
#ifdef FLOATS
  ov: return scm_makdbl(((double)INUM(x))/((double)INUM(y)), 0.0);
#else
  ov: scm_wta(x, (char *)OVFLOW, s_divide);
    return UNSPECIFIED;
#endif
  }
}




#ifdef FLOATS
PROC1 (s_asinh, "$asinh", tc7_cxr, (SCM (*)()) scm_asinh);
#ifdef __STDC__
double
scm_asinh(double x)
#else
double
scm_asinh(x)
     double x;
#endif
{
  return log(x+sqrt(x*x+1));
}




PROC1 (s_acosh, "$acosh", tc7_cxr, (SCM (*)()) scm_acosh);
#ifdef __STDC__
double
scm_acosh(double x)
#else
double
scm_acosh(x)
     double x;
#endif
{
  return log(x+sqrt(x*x-1));
}




PROC1 (s_atanh, "$atanh", tc7_cxr, (SCM (*)()) scm_atanh);
#ifdef __STDC__
double
scm_atanh(double x)
#else
double
scm_atanh(x)
     double x;
#endif
{
  return 0.5*log((1+x)/(1-x));
}




PROC1 (s_truncate, "truncate", tc7_cxr, (SCM (*)()) scm_truncate);
#ifdef __STDC__
double
scm_truncate(double x)
#else
double
scm_truncate(x)
     double x;
#endif
{
  if (x < 0.0) return -floor(-x);
  return floor(x);
}



PROC1 (s_round, "round", tc7_cxr, (SCM (*)()) scm_round);
#ifdef __STDC__
double
scm_round(double x)
#else
double
scm_round(x)
     double x;
#endif
{
  double plus_half = x + 0.5;
  double result = floor(plus_half);
  /* Adjust so that the scm_round is towards even.  */
  return (plus_half == result && plus_half / 2 != floor(plus_half / 2))
    ? result - 1 : result;
}



PROC1 (s_exact_to_inexact, "exact->inexact", tc7_cxr, (SCM (*)()) scm_exact_to_inexact);
#ifdef __STDC__
double
scm_exact_to_inexact(double z)
#else
double
scm_exact_to_inexact(z)
     double z;
#endif
{
  return z;
}


PROC1 (s_i_floor, "floor", tc7_cxr, (SCM (*)()) floor);
PROC1 (s_i_ceil, "ceiling", tc7_cxr, (SCM (*)()) ceil);
PROC1 (s_i_sqrt, "$sqrt", tc7_cxr, (SCM (*)())sqrt);
PROC1 (s_i_abs, "$abs", tc7_cxr, (SCM (*)())fabs);
PROC1 (s_i_exp, "$exp", tc7_cxr, (SCM (*)())exp);
PROC1 (s_i_log, "$log", tc7_cxr, (SCM (*)())log);
PROC1 (s_i_sin, "$sin", tc7_cxr, (SCM (*)())sin);
PROC1 (s_i_cos, "$cos", tc7_cxr, (SCM (*)())cos);
PROC1 (s_i_tan, "$tan", tc7_cxr, (SCM (*)())tan);
PROC1 (s_i_asin, "$asin", tc7_cxr, (SCM (*)())asin);
PROC1 (s_i_acos, "$acos", tc7_cxr, (SCM (*)())acos);
PROC1 (s_i_atan, "$atan", tc7_cxr, (SCM (*)())atan);
PROC1 (s_i_sinh, "$sinh", tc7_cxr, (SCM (*)())sinh);
PROC1 (s_i_cosh, "$cosh", tc7_cxr, (SCM (*)())cosh);
PROC1 (s_i_tanh, "$tanh", tc7_cxr, (SCM (*)())tanh);

struct dpair {double x, y;};

void scm_two_doubles(z1, z2, sstring, xy)
     SCM z1, z2;
     char *sstring;
     struct dpair *xy;
{
  if INUMP(z1) xy->x = INUM(z1);
  else {
# ifdef BIGDIG
    ASRTGO(NIMP(z1), badz1);
    if BIGP(z1) xy->x = scm_big2dbl(z1);
    else {
#  ifndef RECKLESS
      if (!(REALP(z1)))
      badz1: scm_wta(z1, (char *)ARG1, sstring);
#  endif
      xy->x = REALPART(z1);}
# else
    {ASSERT(NIMP(z1) && REALP(z1), z1, ARG1, sstring);
     xy->x = REALPART(z1);}
# endif
  }
  if INUMP(z2) xy->y = INUM(z2);
  else {
# ifdef BIGDIG
    ASRTGO(NIMP(z2), badz2);
    if BIGP(z2) xy->y = scm_big2dbl(z2);
    else {
#  ifndef RECKLESS
      if (!(REALP(z2)))
      badz2: scm_wta(z2, (char *)ARG2, sstring);
#  endif
      xy->y = REALPART(z2);}
# else
    {ASSERT(NIMP(z2) && REALP(z2), z2, ARG2, sstring);
     xy->y = REALPART(z2);}
# endif
  }
}




PROC (s_sys_expt, "%expt", 2, 0, 0, scm_sys_expt);
#ifdef __STDC__
SCM
scm_sys_expt(SCM z1, SCM z2)
#else
SCM
scm_sys_expt(z1, z2)
     SCM z1;
     SCM z2;
#endif
{
  struct dpair xy;
  scm_two_doubles(z1, z2, s_sys_expt, &xy);
  return scm_makdbl(pow(xy.x, xy.y), 0.0);
}



PROC (s_sys_atan2, "%atan2", 2, 0, 0, scm_sys_atan2);
#ifdef __STDC__
SCM
scm_sys_atan2(SCM z1, SCM z2)
#else
SCM
scm_sys_atan2(z1, z2)
     SCM z1;
     SCM z2;
#endif
{
  struct dpair xy;
  scm_two_doubles(z1, z2, s_sys_atan2, &xy);
  return scm_makdbl(atan2(xy.x, xy.y), 0.0);
}



PROC (s_make_rectangular, "make-rectangular", 2, 0, 0, scm_make_rectangular);
#ifdef __STDC__
SCM
scm_make_rectangular(SCM z1, SCM z2)
#else
SCM
scm_make_rectangular(z1, z2)
     SCM z1;
     SCM z2;
#endif
{
  struct dpair xy;
  scm_two_doubles(z1, z2, s_make_rectangular, &xy);
  return scm_makdbl(xy.x, xy.y);
}



PROC (s_make_polar, "make-polar", 2, 0, 0, scm_make_polar);
#ifdef __STDC__
SCM
scm_make_polar(SCM z1, SCM z2)
#else
SCM
scm_make_polar(z1, z2)
     SCM z1;
     SCM z2;
#endif
{
  struct dpair xy;
  scm_two_doubles(z1, z2, s_make_polar, &xy);
  return scm_makdbl(xy.x*cos(xy.y), xy.x*sin(xy.y));
}




PROC (s_realpart, "real-part", 1, 0, 0, scm_realpart);
#ifdef __STDC__
SCM
scm_realpart(SCM z)
#else
SCM
scm_realpart(z)
     SCM z;
#endif
{
  if NINUMP(z) {
# ifdef BIGDIG
    ASRTGO(NIMP(z), badz);
    if BIGP(z) return z;
#  ifndef RECKLESS
    if (!(INEXP(z)))
    badz: scm_wta(z, (char *)ARG1, s_realpart);
#  endif
# else
    ASSERT(NIMP(z) && INEXP(z), z, ARG1, s_realpart);
# endif
    if CPLXP(z) return scm_makdbl(REAL(z), 0.0);
  }
  return z;
}



PROC (s_imag_part, "imag-part", 1, 0, 0, scm_imag_part);
#ifdef __STDC__
SCM
scm_imag_part(SCM z)
#else
SCM
scm_imag_part(z)
     SCM z;
#endif
{
  if INUMP(z) return INUM0;
# ifdef BIGDIG
  ASRTGO(NIMP(z), badz);
  if BIGP(z) return INUM0;
#  ifndef RECKLESS
  if (!(INEXP(z)))
  badz: scm_wta(z, (char *)ARG1, s_imag_part);
#  endif
# else
  ASSERT(NIMP(z) && INEXP(z), z, ARG1, s_imag_part);
# endif
  if CPLXP(z) return scm_makdbl(IMAG(z), 0.0);
  return flo0;
}



PROC (s_magnitude, "magnitude", 1, 0, 0, scm_magnitude);
#ifdef __STDC__
SCM
scm_magnitude(SCM z)
#else
SCM
scm_magnitude(z)
     SCM z;
#endif
{
  if INUMP(z) return scm_abs(z);
# ifdef BIGDIG
  ASRTGO(NIMP(z), badz);
  if BIGP(z) return scm_abs(z);
#  ifndef RECKLESS
  if (!(INEXP(z)))
  badz: scm_wta(z, (char *)ARG1, s_magnitude);
#  endif
# else
  ASSERT(NIMP(z) && INEXP(z), z, ARG1, s_magnitude);
# endif
  if CPLXP(z)
    {
      double i = IMAG(z), r = REAL(z);
      return scm_makdbl(sqrt(i*i+r*r), 0.0);
    }
  return scm_makdbl(fabs(REALPART(z)), 0.0);
}




PROC (s_angle, "angle", 1, 0, 0, scm_angle);
#ifdef __STDC__
SCM
scm_angle(SCM z)
#else
SCM
scm_angle(z)
     SCM z;
#endif
{
  double x, y = 0.0;
  if INUMP(z) {x = (z>=INUM0) ? 1.0 : -1.0; goto do_angle;}
# ifdef BIGDIG
  ASRTGO(NIMP(z), badz);
  if BIGP(z) {x = (TYP16(z)==tc16_bigpos) ? 1.0 : -1.0; goto do_angle;}
#  ifndef RECKLESS
  if (!(INEXP(z))) {
  badz: scm_wta(z, (char *)ARG1, s_angle);}
#  endif
# else
  ASSERT(NIMP(z) && INEXP(z), z, ARG1, s_angle);
# endif
  if (REALP(z))
    {
      x = REALPART(z);
      goto do_angle;
    }
  x = REAL(z); y = IMAG(z);
 do_angle:
  return scm_makdbl(atan2(y, x), 0.0);
}


PROC (s_inexact_to_exact, "inexact->exact", 1, 0, 0, scm_inexact_to_exact);
#ifdef __STDC__
SCM
scm_inexact_to_exact(SCM z)
#else
SCM
scm_inexact_to_exact(z)
     SCM z;
#endif
{
  if INUMP(z) return z;
# ifdef BIGDIG
  ASRTGO(NIMP(z), badz);
  if BIGP(z) return z;
#  ifndef RECKLESS
  if (!(REALP(z)))
  badz: scm_wta(z, (char *)ARG1, s_inexact_to_exact);
#  endif
# else
  ASSERT(NIMP(z) && REALP(z), z, ARG1, s_inexact_to_exact);
# endif
# ifdef BIGDIG
  {
    double u = floor(REALPART(z)+0.5);
    if ((u <= MOST_POSITIVE_FIXNUM) && (-u <= -MOST_NEGATIVE_FIXNUM)) {
      /* Negation is a workaround for HP700 cc bug */
      SCM ans = MAKINUM((long)u);
      if (INUM(ans)==(long)u) return ans;
    }
    ASRTGO(!IS_INF(u), badz);	/* problem? */
    return scm_dbl2big(u);
  }
# else
  return MAKINUM((long)floor(REALPART(z)+0.5));
# endif
}



#else				/* ~FLOATS */
PROC (s_trunc, "truncate", 1, 0, 0, scm_trunc);
#ifdef __STDC__
SCM
scm_trunc(SCM x)
#else
SCM
scm_trunc(x)
     SCM x;
#endif
{
  ASSERT(INUMP(x), x, ARG1, s_truncate);
  return x;
}



#endif				/* FLOATS */

#ifdef BIGDIG
# ifdef FLOATS
/* d must be integer */
#ifdef __STDC__
SCM
scm_dbl2big(double d)
#else
SCM
scm_dbl2big(d)
     double d;
#endif
{
  sizet i = 0;
  long c;
  BIGDIG *digits;
  SCM ans;
  double u = (d < 0)?-d:d;
  while (0 != floor(u)) {u /= BIGRAD;i++;}
  ans = scm_mkbig(i, d < 0);
  digits = BDIGITS(ans);
  while (i--) {
    u *= BIGRAD;
    c = floor(u);
    u -= c;
    digits[i] = c;
  }
  ASSERT(0==u, INUM0, OVFLOW, "dbl2big");
  return ans;
}



#ifdef __STDC__
double
scm_big2dbl(SCM b)
#else
double
scm_big2dbl(b)
     SCM b;
#endif
{
  double ans = 0.0;
  sizet i = NUMDIGS(b);
  BIGDIG *digits = BDIGITS(b);
  while (i--) ans = digits[i] + BIGRAD*ans;
  if (tc16_bigneg==TYP16(b)) return -ans;
  return ans;
}
# endif
#endif

#ifdef __STDC__
SCM
scm_long2num(long sl)
#else
SCM
scm_long2num(sl)
     long sl;
#endif
{
  if (!FIXABLE(sl)) {
#ifdef BIGDIG
    return scm_long2big(sl);
#else
# ifdef FLOATS
    return scm_makdbl((double) sl, 0.0);
# else
    return BOOL_F;
# endif
#endif
  }
  return MAKINUM(sl);
}



#ifdef __STDC__
SCM
scm_ulong2num(unsigned long sl)
#else
SCM
scm_ulong2num(sl)
     unsigned long sl;
#endif
{
  if (!POSFIXABLE(sl)) {
#ifdef BIGDIG
    return scm_ulong2big(sl);
#else
# ifdef FLOATS
    return scm_makdbl((double) sl, 0.0);
# else
    return BOOL_F;
# endif
#endif
  }
  return MAKINUM(sl);
}

#ifdef __STDC__
long
scm_num2long(SCM num, char *pos, char *s_caller)
#else
long
scm_num2long(num, pos, s_caller)
     SCM num;
     char *pos;
     char *s_caller;
#endif
{
  long res;
  if (INUMP(num))
    {
      res = INUM(num);
      return res;
    }
  ASRTGO(NIMP(num), errout);
#ifdef FLOATS
  if (REALP(num))
    {
      double u = REALPART(num);
      if ((0 <= u) && (u <= (long)~0L))
	{
	  res = u;
	  return res;
	}
    }
#endif
#ifdef BIGDIG
  if (BIGP(num)) {
    long oldres;
    sizet l;
    res = 0;
    oldres = 0;
    for(l = NUMDIGS(num);l--;)
      {
	res = BIGUP(res) + BDIGITS(num)[l];
	if (res < oldres)
	  goto errout;
	oldres = res;
      }
    if (TYP16 (num) == tc16_bigpos)
      return res;
    else
      return -res;
  }
#endif
 errout: scm_wta(num, pos, s_caller);
  return UNSPECIFIED;
}




#ifdef __STDC__
long
num2long(SCM num, char *pos, char *s_caller)
#else
long
num2long(num, pos, s_caller)
     SCM num;
     char *pos;
     char *s_caller;
#endif
{
  long res;
  if INUMP(num) {
    res = INUM((long)num);
    return res;
  }
  ASRTGO(NIMP(num), errout);
#ifdef FLOATS
  if REALP(num) {
    double u = REALPART(num);
    if (((MOST_NEGATIVE_FIXNUM * 4) <= u)
	&& (u <= (MOST_POSITIVE_FIXNUM * 4 + 3))) {
      res = u;
      return res;
    }
  }
#endif
#ifdef BIGDIG
  if BIGP(num) {
    sizet l = NUMDIGS(num);
    ASRTGO(DIGSPERLONG >= l, errout);
    res = 0;
    for(;l--;) res = BIGUP(res) + BDIGITS(num)[l];
    return res;
  }
#endif
 errout: scm_wta(num, pos, s_caller);
  return UNSPECIFIED;
}



#ifdef __STDC__
unsigned long
scm_num2ulong(SCM num, char *pos, char *s_caller)
#else
unsigned long
scm_num2ulong(num, pos, s_caller)
     SCM num;
     char *pos;
     char *s_caller;
#endif
{
  unsigned long res;
  if (INUMP(num))
    {
      res = INUM((unsigned long)num);
      return res;
    }
  ASRTGO(NIMP(num), errout);
#ifdef FLOATS
  if (REALP(num))
    {
      double u = REALPART(num);
      if ((0 <= u) && (u <= (unsigned long)~0L))
	{
	  res = u;
	  return res;
	}
    }
#endif
#ifdef BIGDIG
  if (BIGP(num)) {
    unsigned long oldres;
    sizet l;
    res = 0;
    oldres = 0;
    for(l = NUMDIGS(num);l--;)
      {
	res = BIGUP(res) + BDIGITS(num)[l];
	if (res < oldres)
	  goto errout;
	oldres = res;
      }
    return res;
  }
#endif
 errout: scm_wta(num, pos, s_caller);
  return UNSPECIFIED;
}


#ifdef FLOATS
# ifndef DBL_DIG
static void add1(f, fsum)
     double f, *fsum;
{
  *fsum = f + 1.0;
}
# endif
#endif


#ifdef __STDC__
void
scm_init_numbers (void)
#else
void
scm_init_numbers ()
#endif
{
#ifdef FLOATS
  NEWCELL(flo0);
# ifdef SINGLES
  CAR(flo0) = tc_flo;
  FLO(flo0) = 0.0;
# else
  CDR(flo0) = (SCM)scm_must_malloc(1L*sizeof(double), "real");
  REAL(flo0) = 0.0;
  CAR(flo0) = tc_dblr;
# endif
# ifdef DBL_DIG
  scm_dblprec = (DBL_DIG > 20) ? 20 : DBL_DIG;
# else
  {				/* determine floating point precision */
    double f = 0.1;
    double fsum = 1.0+f;
    while (fsum != 1.0) {
      f /= 10.0;
      if (++scm_dblprec > 20) break;
      add1(f, &fsum);
    }
    scm_dblprec = scm_dblprec-1;
  }
# endif /* DBL_DIG */
#endif
#include "numbers.x"
}

