// -*- C++ -*-
// Copyright (C) 2000 Red Hat, Inc.
// Example code using the Inti main loop
/*
  The following license applies to this example code, but not to
  the Inti library itself:
  
    Permission is hereby granted, free of charge, to any person obtaining
    a copy of this software and associated documentation files (the
    "Software"), to deal in the Software without restriction, including
    without limitation the rights to use, copy, modify, merge, publish,
    distribute, sublicense, and/or sell copies of the Software, and to
    permit persons to whom the Software is furnished to do so, subject to
    the following conditions:

    The above copyright notice and this permission notice shall be
    included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
    NONINFRINGEMENT.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
    DISTRIBUTOR OF THE SOFTWARE BE LIABLE FOR ANY CLAIM, DAMAGES OR
    OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
    OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
    OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <inti/main.h>
#include <vector>
#include <string>
#include <iostream>

using namespace Inti;

vector<string> strings_to_print;

// An event source that handles the event "strings_to_print
// contains strings". It's useless, but shows you
// how a custom event source works.
class PrintoutProcessor : public Main::Source
{
public:
  // called before the event loop blocks
  virtual bool prepare (const TimeStamp& current_time,
                        int *timeout_milliseconds)
  {
    // We want to be sure we print out our strings every second. So,
    // we impose a half-second maximum time before the event loop
    // stops blocking and calls check()
    *timeout_milliseconds = 1000;
    
    // we have to return true if we know we need to do work; we need
    // to do work if there are strings to be printed.
    return !strings_to_print.empty();
  }

  // called after the event loop blocks
  virtual bool check (const TimeStamp& current_time)
  {
    // we have to return true if we know we need to do work; we need
    // to do work if there are strings to be printed.
    return !strings_to_print.empty();
  }
  
  // called if prepare() or check() returned true
  virtual bool dispatch (const TimeStamp& current_time)
  {
    // If check() or prepare() returned true, dispatch() is called to
    // do the work.  So we print our strings.
    vector<string>::iterator i;
    i = strings_to_print.begin();
    while (i != strings_to_print.end())
      {
        cout << *i << endl;
        ++i;
      }
    strings_to_print.clear();

    // keep this event source around. If we return false, the main
    // loop will remove this Source and call unref() on it.
    return true;
  }
  
};

class TimeoutHandler
{
public:
  TimeoutHandler ()
    : count_(0)
  {

  }
  
  bool timeout ()
  {
    cout << "Timeout occurred!" << endl;
    // Add some strings to the queue of strings to print out.
    strings_to_print.push_back ("String added to the string list by the timeout handler");

    ++count_;

    // Quit the main loop after 50 timeouts
    if (count_ > 50)
      Main::primary()->quit();

    // return true to keep our timeout installed.
    return true;
  }

private:
  int count_;
};

int
main (int argc, char **argv)
{
  // Create a main loop
  Main::Loop loop;

  // Create an object to respond to timeouts, and add it to the event
  // loop to be called every 300 milliseconds
  TimeoutHandler timeout_handler;

  Main::get()->connect_timeout (300,
                                slot (&timeout_handler,
                                      &TimeoutHandler::timeout));

  // Add a custom event source
  Main::get()->connect_source (new PrintoutProcessor);
  
  // block until someone calls loop.quit()
  loop.run ();

  // Control of the application returns here if loop.quit() is called.
  cout << "Main loop quit" << endl;
  
  return 0;
}

