// -*- C++ -*-
// Copyright (C) 2000 Red Hat, Inc.
// Example code using the Inti smart pointer
/*
  The following license applies to this example code, but not to
  the Inti library itself:
  
    Permission is hereby granted, free of charge, to any person obtaining
    a copy of this software and associated documentation files (the
    "Software"), to deal in the Software without restriction, including
    without limitation the rights to use, copy, modify, merge, publish,
    distribute, sublicense, and/or sell copies of the Software, and to
    permit persons to whom the Software is furnished to do so, subject to
    the following conditions:

    The above copyright notice and this permission notice shall be
    included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
    NONINFRINGEMENT.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
    DISTRIBUTOR OF THE SOFTWARE BE LIABLE FOR ANY CLAIM, DAMAGES OR
    OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
    OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
    OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <inti/ptr.h>
#include <inti/main.h>
#include <iostream.h>

using namespace Inti;

// A simple object implementing floating reference count memory model
class Floating
{
public:
  Floating ();

  void ref () { ref_count_ += 1; }
  void unref ();

  bool is_floating () const
  {
    return floating_;
  }

  void sink ();

  void frobate ();

  static ptr<Floating> create ()
  {
    return ptr<Floating>(new Floating);
  }
  
protected:
  // protected destructor, so no one tries to delete the object
  // without going through unref ()
  ~Floating ()
  {
    --outstanding_objects_;
    cout << "Deleted an object, " << outstanding_objects_
         << " remaining" << endl;
  }
  
private:
  bool floating_;
  unsigned int ref_count_;
  int use_up_some_memory_[100];

  static int outstanding_objects_;

  Floating (const Floating&);
  Floating& operator=(const Floating&);
};

int Floating::outstanding_objects_ = 0;

Floating::Floating ()
  : floating_(true), ref_count_(1)
{
  ++outstanding_objects_;
}

void
Floating::unref ()
{
  ref_count_ -= 1;
  if (ref_count_ == 0)
    delete this;
}

void
Floating::sink ()
{
  // Remove the floating reference count
  if (floating_)
    {
      floating_ = false;
      unref ();
    }
}


void
Floating::frobate ()
{


}

// trivial subclass
class SubFloating : public Floating
{
protected:
  // note that we have to keep the destructor protected,
  // or suddenly these objects can be on the stack.
  // See Item 27 in "More Effective C++"
  // on other ways to deal with this.
  ~SubFloating () {}

};

class StrictlyCounted
{
public:
  StrictlyCounted ()
    : ref_count_(1)
  {
    ++outstanding_objects_;
  }

  void ref ()
  {
    ref_count_ += 1;
  }

  void unref ()
  {
    ref_count_ -= 1;
    if (ref_count_ == 0)
      delete this;
  }
  
protected:
  // protected destructor, so no one tries to delete the object
  // without going through unref ()
  ~StrictlyCounted ()
  {
    --outstanding_objects_;
    cout << "Deleted an object, " << outstanding_objects_
         << " strictly-counted objects remaining" << endl;
  }
  
private:
  unsigned int ref_count_;
  int use_up_some_memory_[100];

  static int outstanding_objects_;

  StrictlyCounted (const StrictlyCounted&);
  StrictlyCounted& operator=(const StrictlyCounted&);
};

// pass by pointer
void func (Floating * f)
{
  
}

// pass by reference
void func_takes_reference (const Floating & f)
{


}

int
main (int argc, char **argv)
{
  // Here is how you initialize a new pointer -
  // ptr<Floating> f = new Floating won't work,
  // unfortunately, because the ptr(T* obj) constructor
  // is declared explicit to prevent insanity.
  
  ptr<Floating> f (new Floating);
  ptr<SubFloating> s (new SubFloating);

  // Of course, this works, even though the "ptr<Floating> f = new
  // Floating" you would really like to have does not work.
  // Trust C++ to have this "feature"
  f = new Floating;
  
  // See if we can compare to 0 properly
  if (f != 0)
    cout << "!= 0" << endl;

  // The following commented-out stuff shouldn't compile if uncommented
  //   delete f;   // delete instead of ->unref()
  //   s = f;      // assignment of base class ptr to subclass ptr
  //   *f = *f;    // copy by value disallowed for Floating
  
  // These things should all work
  f->frobate ();         // calling members with ->
  s->frobate ();
  f = new SubFloating;   // assignment of subclass to ptr<base class>
  f = s;                 // assignment of subclass ptr to base class ptr

  if (f == s)            // comparison between base and subclass ptr
    cout << "They are equal" << endl;
  if (f != s)
    cout << "They are not equal" << endl;
  
  func (f.get());        // .get() 
  func (s.get());
  func (f);              // implicit conversion when passing to function
  func (s);
  func_takes_reference (*f); // dereference operator
  func_takes_reference (*s);

  // This should not leak memory or segfault.
  int i = 0;
  while (i < 1000)
    {
      f = new Floating;
      f.reset (new Floating);
      f.reset ();
      f = new Floating;
      f = new Floating;
      f = 0;
      f = f;
      f = new Floating;
      f = f;
      f = f;
      ++i;
    }

  // This should work too
  ptr<Floating>* array = new ptr<Floating>[100];
  i = 0;
  while (i < 100)
    {
      array[i] = new Floating;
      ++i;
    }

  delete [] array;

  // The following, if uncommented, would be a memory leak
  // ptr<Floating> g (new Floating);
  // ptr<Floating> h (new Floating);
  // g = h.release ()               // released reference isn't floating,
  //                                // so g adds its own new reference

  // This should immediately free the pointed-to object when
  // the ptr goes out of scope
  {
    cout << "Should delete an object..." << endl;
    ptr<Floating> p (new Floating);
  }
  cout << "... object should have been deleted" << endl;

  ptr<Floating> p (new Floating);
  ptr<Floating> q (new Floating);
  ptr<Floating> r (new Floating);

  p = q = r;

  if ((p == q) && (q == r))
    cout << "All equal" << endl;

  q = r = 0;


  // Now let's try some strict_ptr stuff.
  
  strict_ptr<Floating> f_s (new Floating);
  // refcount of Floating instance is now TWO, not ONE as it would be
  // with ptr, because strict_ptr will not "adopt" the floating count.

  // drop our count, leaving strict_ptr with the only reference
  f_s->unref ();
  
  // See if we can assign between strict_ptr and ptr
  f = 0;
  f = f_s;
  f_s = f;
  
  return 0;
}

void
testing (Floating * p)
{
  p->frobate ();
}

void
ptr_testing (const ptr<Floating> & p)
{
  p->frobate ();
}

void
safe_ptr_code_size (void)
{
  ptr<Floating> p = Floating::create ();


  p = Floating::create();


  ptr_testing (p);

  
  ptr_testing (Floating::create());
}

void
unsafe_ptr_code_size (void)
{
  ptr<Floating> p (new Floating);


  p = new Floating;


  testing (p);

  
  testing (new Floating);
}


void
plain_ptr_code_size (void)
{
  Floating * p = new Floating;
  p->unref ();

  p = new Floating;

  testing (p);
  p->unref ();
  
  testing (new Floating);
}
