// -*- C++ -*-
// Copyright (C) 2000 Red Hat, Inc.
// Typing game example
/*
  The following license applies to this example code, but not to
  the Inti library itself:
  
    Permission is hereby granted, free of charge, to any person obtaining
    a copy of this software and associated documentation files (the
    "Software"), to deal in the Software without restriction, including
    without limitation the rights to use, copy, modify, merge, publish,
    distribute, sublicense, and/or sell copies of the Software, and to
    permit persons to whom the Software is furnished to do so, subject to
    the following conditions:

    The above copyright notice and this permission notice shall be
    included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
    NONINFRINGEMENT.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
    DISTRIBUTOR OF THE SOFTWARE BE LIABLE FOR ANY CLAIM, DAMAGES OR
    OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
    OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
    OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include "game.h"
#include "badguy.h"
#include "player.h"
#include <algorithm>
#include <inti/timedate.h>
#include <inti/main.h>
#include <iostream>

Game::UpdateSignalType Game::update_signal;

Game::Game (Player * player)
  : started_ (false), paused_ (false), player_ (player),
    width_ (0), height_ (0),
    score_ (0), shields_ (100)
{


}

Game::~Game ()
{
  timeout_.disconnect ();
  
  vector<Badguy*>::iterator i = badguys_.begin ();
  vector<Badguy*>::iterator end = badguys_.end ();

  while (i != end)
    {
      delete *i;
      ++i;
    }
}

void
Game::start ()
{
  if (started_)
    return;
  
  started_ = true;

  badguys_.push_back (create_badguy (this, BADGUY_EASY));
  badguys_.push_back (create_badguy (this, BADGUY_SCARY));
  
  last_stamp_ = current_timestamp ();
  
  timeout_ = Main::get ()->connect_timeout (30,
                                            slot (this, &Game::on_timeout));

  update_signal.emit (this);
}

void
Game::set_paused (bool state)
{
  if (state && !paused_)
    {
      paused_ = true;
      
      timeout_.block ();
    }
  else if (!state && paused_)
    {
      paused_ = false;

      timeout_.unblock ();
    }
}

void
Game::set_size (int width, int height)
{
  width_ = width;
  height_ = height;
}

void
Game::add_badguy (Badguy * badguy)
{
  badguys_.push_back (badguy);
}

void
Game::delete_badguy (Badguy * badguy)
{
  remove (badguys_.begin (),
          badguys_.end (),
          badguy);

  delete badguy;
}

bool
Game::process_key (const Gdk::EventKey & event)
{
  // Don't process events that are probably key accelerators
  // rather than part of the game.
  static const Gdk::EventMaskField ignore_mask =
    Gdk::CONTROL_MASK | Gdk::MOD1_MASK;

  if ((event.state () & ignore_mask) != 0)
    return false;
  
  vector<Badguy*>::iterator i = badguys_.begin ();
  vector<Badguy*>::iterator end = badguys_.end ();

  while (i != end)
    {
      (*i)->damage (event.text ());
      
      ++i;
    }

  return true;
}

bool
Game::on_timeout ()
{
  TimeStamp new_stamp = current_timestamp ();  

  if (new_stamp == last_stamp_)
    return true;
  
  long elapsed = new_stamp.microseconds_since (last_stamp_);
  
  vector<Badguy*>::iterator i = badguys_.begin ();
  vector<Badguy*>::iterator end = badguys_.end ();

  while (i != end)
    {
      (*i)->update (elapsed);
      
      ++i;
    }

  last_stamp_ = new_stamp;

  update_signal.emit (this);
  
  return true;
}

void
Game::adjust_score (int delta)
{
  score_ += delta;
}

void
Game::adjust_shields (int delta)
{
  shields_ += delta;
}
