#!/usr/bin/env python
# -*- Mode: Python; py-indent-offset: 4 -*-

from string import *
import sys
import re
import exceptions
import message

doc_segment_start = re.compile ("^ *([$@][a-z]+)", re.MULTILINE)
def split_doc_comment (comment):
    segments = [] # this is a list of tuples: ('$foo', 'value of foo')

    strings = doc_segment_start.split (comment)

    last = None
    for s in strings:
        s = strip (s)
        if s != '':
            if last:
                segments.append ((last, s))
                last = None
            else:
                last = s
                if last[0] != '$' and last[0] != '@':
                    message.die ("doc comment parser got confused\n")

    if last:
        message.warn ("no content for: " + last + "\n")

    return segments

class Doc:
    def __init__ (self, type):
        self.type = type

    def parse_comment (self, params, comment):
        message.die ("Called abstract method")

    def spew (self, indent):
        print indent + "docs with type: " + self.type + "\n"

class TopicDoc (Doc):
    def __init__ (self):
        Doc.__init__(self, "topic")
        self.name = None
        self.short = None
        self.title = None
        self.long = None

    def set_name (self, name):
        self.name = name
        strip (self.name)

    def set_short (self, short):
        self.short = short
        strip (self.short)
        
    def set_long (self, long):
        self.long = long
        strip (self.long)
        
    def set_title (self, title):
        self.title = title
        strip (self.title)

    def get_name (self):
        return self.name

    def get_short (self):
        return self.short

    def get_long (self):
        return self.long

    def get_title (self):
        return self.title

    params_re = re.compile ("([-a-zA-Z]+) +\"([^\"]+)\"")

    def parse_comment (self, params, comment):
        match = self.params_re.match (params)
        if not params:
            message.die ("topic doc needs name and title")
        self.set_name (match.group (1))
        self.set_title (match.group (2))
        segments = split_doc_comment (comment)
        for (name, content) in segments:
            if name == '$short':
                self.set_short (content)
            elif name == '$long':
                self.set_long (content)
            else:
                message.die ("don't understand markup: " + name + "for topic docs")

class HeaderDoc (Doc):
    def __init__ (self):
        Doc.__init__(self, "header")
        self.name = None
        self.short = None

    def set_name (self, name):
        self.name = name
        strip (self.name)

    def set_short (self, short):
        self.short = short
        strip (self.short)

    def get_name (self):
        return self.name

    def get_short (self):
        return self.short

    def parse_comment (self, params, comment):
        segments = split_doc_comment (comment)
        for (name, content) in segments:
            if name == '$short':
                self.set_short (content)
            elif name == '$long':
                self.set_long (content)
            else:
                message.die ("don't understand markup: " + name + "for header docs")

class MethodDoc (Doc):
    def __init__ (self):
        Doc.__init__(self, "method")
        self.params = [] # list of (name, docs) tuples
        self.short = None # string, short docs
        self.long = None # string, long docs
        self.returns = None # string, docs for return value
        self.important = 0
        
    def add_param (self, name, docs):
        self.params.append ( (name, docs) )
        
    def set_short (self, short):
        self.short = short
        strip (self.short)

    def set_long (self, long):
        self.long = long
        strip (self.long)

    def set_returns (self, returns):
        self.returns = returns

    def get_params (self):
        return self.params

    def get_short (self):
        return self.short

    def get_long (self):
        return self.long

    def get_returns (self):
        return self.returns

    def parse_comment (self, params, comment):
        params = strip (params)
        if params == '(important)':
            self.important = 1
        #        if params == '':
        #            message.die ("method doc should specify method name: " + comment)
        #        self.set_name (params)
        segments = split_doc_comment (comment)
        for (name, content) in segments:
            if name[0] == '@':
                self.add_param (name[1:], content)
            elif name == '$returns':
                self.set_returns (content)
            elif name == '$short':
                self.set_short (content)
            elif name == '$long':
                self.set_long (content)
            else:
                message.die ("don't understand markup: " + name + "for method docs")

class VariableDoc (Doc):
    def __init__ (self):
        Doc.__init__(self, "variable")
        self.name = None
        self.short = None
        self.long = None

    def set_name (self, name):
        self.name = name
        
    def set_short (self, short):
        self.short = short
        strip (self.short)

    def set_long (self, long):
        self.long = long
        strip (self.long)

    def get_name (self):
        return self.name

    def get_short (self):
        return self.short

    def get_long (self):
        return self.long

    def parse_comment (self, params, comment):
        params = strip (params)
        if params == '':
            message.die ("variable doc should specify variable name: " + comment)
        self.set_name (params)
        segments = split_doc_comment (comment)
        for (name, content) in segments:
            if name == '$short':
                self.set_short (content)
            elif name == '$long':
                self.set_long (content)
            else:
                message.die ("don't understand markup: " + name + "for variable docs")
        
class ClassDoc (Doc):
    def __init__ (self):
        Doc.__init__(self, "class")
        self.name = None # unqualified string name
        self.short = None   # short docs (string)
        self.long = None    # long docs (string)
        self.memory_model = None # string from a fixed set
        self.copy_semantics = None # string from a fixed set

    def set_copy_semantics (self, semantics):
        self.copy_semantics = semantics
        strip (self.copy_semantics)

    def set_memory_model (self, model):
        self.memory_model = model
        strip (self.memory_model)

    def set_short (self, short):
        self.short = short
        strip (self.short)

    def set_long (self, long):
        self.long = long
        strip (self.long)

    def set_name (self, name):
        self.name = name
        strip (self.name)

    def get_copy_semantics (self):
        return self.copy_semantics

    def get_memory_model (self):
        return self.memory_model

    def get_short (self):
        return self.short

    def get_long (self):
        return self.long

    def get_name (self):
        return self.name

    def parse_comment (self, params, comment):
        params = strip (params)
        if params == '':
            message.die ("class doc should specify class name: " + comment)
        self.set_name (params)
        segments = split_doc_comment (comment)
        for (name, content) in segments:
            if name == '$short':
                self.set_short (content)
            elif name == '$long':
                self.set_long (content)
            elif name == '$memory':
                self.set_memory_model (content)
            elif name == '$copy':
                self.set_copy_semantics (content)
            else:
                message.die ("don't understand markup: " + name + "for class docs")


# /*$ doc whatever params
doc_start = re.compile (" *doc +([a-z]+) *(.*)\n")

def parse_doc_comment (comment):
    match = doc_start.match (comment)
    if not match:
        message.die ("start of doc comment didn't parse properly")

    doctype = match.group (1)
    params = match.group (2)

    # strip off the start
    comment = doc_start.sub ('', comment)

    doc = None

    if doctype == 'class':
        doc = ClassDoc ()
    elif doctype == 'method' or \
         doctype == 'constructor' or \
         doctype == 'destructor':
        doc = MethodDoc ()
    elif doctype == 'variable':
        doc = VariableDoc ()
    elif doctype == 'header':
        doc = HeaderDoc ()
    elif doctype == 'topic':
        doc = TopicDoc ()
    else:
        message.die ("unknown doc comment type: " + doctype + "\n")

    doc.parse_comment (params, comment)

    split_doc_comment (comment)
        
    return doc


