#!/usr/bin/env python
# -*- Mode: Python; py-indent-offset: 4 -*-

from string import *
import os
import sys
import re
import exceptions
from docobjects import *
from declarations import *
import cgi

def output_header_docs (header_decl, target_dir):
    name = header_decl.name
    if name[-2:] != '.h':
        message.die ("header name must end with .h")
    base = os.path.basename(name[:-2])
    dbfile = open (os.path.join(target_dir, base + '.sgml'), 'w')

    output_db (header_decl, dbfile)

def sgmlconvert (str):
    return cgi.escape (str)

variable_magic = re.compile ('##([^ \n\t]+)') # must check before class_magic
class_magic = re.compile ('#([^ \n\t]+)')
param_magic = re.compile ('@([^ \n\t]+)')
type_magic = re.compile ('%%([^ \n\t]+)')
enum_val_magic = re.compile ('%([^ \n\t]+)')
signal_magic = re.compile ('\\\\([^ \n\t]+)')
xref_magic = re.compile ('\[([^ \n\t]+):([^ \n\t]+)\]')
method_magic = re.compile ('([^ \n\t]+) ?\(\)')

global_current_decl = None

link_template = "<link linkend=\"%(linkend)s\">%(linktext)s</link>" 
def make_link (decl, expected_type, text):
    linkend = decl.get_xref_id ()

    linktext = text
    return link_template % vars ()

def make_link_from_full_name (name, expected_type):
    linkend = fix_for_sgml (expected_type + '-' + \
                            join (split (name, '::'), '-') )
    linktext = name
    return link_template % vars ()

def make_link_from_name (name, scope_decl, expected_type):
    if not scope_decl:
        message.die ("need scope_decl")
        
    decl = scope_decl.lookup (name, [expected_type])
    if decl:
        return make_link (decl, expected_type, name)
    else:
        ## assume same scope as scope_decl if name is unqualified
        if find(name, '::') < 0:
            return make_link_from_full_name (scope_decl.qualifier () + name, expected_type)
        else:
            ## go up however many scopes are specified in the name
            ## (i.e. if the name is Gdk::Foo we go up one scope)
            ## won't always work, but usually will
            scopes = scope_decl.get_scopes ()
            cnt = count (name, '::')
            i = 0
            while scopes != [] and i < cnt:
                scopes.pop ()
                i = i + 1

            qual = ''
            if scopes != []:
                qual = join (scopes, '::') + '::'
                
            return make_link_from_full_name (qual + name, expected_type)
             

def make_link_from_decl (decl):
    linkend = decl.get_xref_id ()
    linktext = decl.qualifier () + decl.name

    return link_template % vars ()

def variable_magic_matched (match):
    varname = match.groups ()[-1]

    # FIXME
    return varname


def class_magic_matched (match):
    cname = match.groups ()[-1]
    punct = ''
    if cname[-1] in ',.:':
        punct = cname[-1]
        cname = cname[:-1]
    
    decl = global_current_decl.lookup (cname, ['class','struct'])

    if decl:
        return make_link (decl, 'class', cname) + punct
    else:
        return make_link_from_name (cname, global_current_decl, 'class')

def param_magic_matched (match):
    paramname = match.groups ()[-1]

    return paramname

def enum_val_matched (match):
    valname = match.groups ()[-1]

    return valname

def signal_magic_matched (match):
    signame = match.groups ()[-1]

    return signame

def xref_magic_matched (match):
    category = match.groups ()[-2]
    target = match.groups ()[-1]

    return category + "/" + target

def type_magic_matched (match):
    typename = match.groups ()[-1]

    return typename

def method_magic_matched (match):
    methodname = match.groups ()[-1]

    decl = global_current_decl.lookup (methodname, ['method', 'function'])

    if decl:
        return make_link (decl, 'method', methodname + ' ()')
    else:
        return make_link_from_name (methodname, global_current_decl, 'method') + ' ()'

def resolve_magic_links (decl, str):
    global global_current_decl
    
    global_current_decl = decl
    
    str2 = variable_magic.sub (variable_magic_matched, str)
    str2 = class_magic.sub (class_magic_matched, str2)
    str2 = param_magic.sub (param_magic_matched, str2)
    str2 = type_magic.sub (type_magic_matched, str2)
    str2 = enum_val_magic.sub (enum_val_matched, str2)
    str2 = signal_magic.sub (signal_magic_matched, str2)
    str2 = xref_magic.sub (xref_magic_matched, str2)
    str2 = method_magic.sub (method_magic_matched, str2)

    global_current_decl = None

    return str2

def get_copy_description (decl, key):
    if key == 'value':
        return 'Instances of this class may be copied by value.'
    elif key == 'none':
        return 'Instances of this class may not be copied.'
    else:
        message.warn ("don't know copy semantics: " + key)

def get_memory_description (decl, key):
    if key == 'c++':
        return 'Instances of this class are not memory managed or reference counted; they should be treated as if they were built-in types.'
    elif key == 'floating':
        return 'Instances of this class are reference counted, and are created with a "floating" reference count.'
    else:
        message.warn ("don't know memory model: " + key)

def output_namespace (decl, outfile):
    for c in decl.get_documented_children ():
        output_child (c, outfile)

class_template = """

      <refsect1 id="%(id)s">
        <title>%(title)s</title>

        <para>
        %(short)s
        </para>
        
        <para>
          <link
          linkend="%(overview_id)s">Overview</link>,
          <link linkend="%(summary_id)s">Interface summary</link>
        </para>

        <para>
        %(copy_description)s
        </para>
        
        <para>
        %(memory_description)s
        </para>
        
        <refsect2>
          <title>Types Related to %(qualified_name)s</title>

          %(superclasses)s

          %(subclasses)s

          %(inner_classes)s

          %(enumerations)s

          %(notypes)s

        </refsect2>

        <refsect2 id="%(summary_id)s">
          <title>Interface Summary for %(qualified_name)s</title>
          %(constructors)s

          %(important_methods)s

          %(important_signals)s

          %(important_functions)s

          %(virtual_methods)s

          %(other_methods)s

          %(other_signals)s

          %(other_functions)s

          </refsect2>

        <refsect2 id="%(overview_id)s">
          <title>Overview of %(qualified_name)s</title>

          %(long)s

        </refsect2>

      </refsect1>
"""

def formalpara (str, label):
    return '<formalpara><title>' + label + '</title><para>\n' + str + \
           '</para></formalpara>\n'

def make_listitem (str):
    return '<listitem><para>' + str + '</para></listitem>\n'

def funcdecl_to_listitem (c):
    short = 'FIXME write description'
    if c.doc and c.doc.get_short ():
        short = c.doc.get_short ()
    return make_listitem (make_link_from_decl (c) + ' () -- ' + short)

def decl_to_listitem (c):
    short = 'FIXME write description'
    if c.doc and c.doc.get_short ():
        short = c.doc.get_short ()
    return make_listitem (make_link_from_decl (c) + ' -- ' + short)

def listpara (str, label):
    str = '<itemizedlist>\n' + str + '</itemizedlist>\n'
    return formalpara (str, label)

def output_class_or_struct (decl, outfile):
    ## first output the class itself, with xrefs
    ## to children
    qualified_name = sgmlconvert (decl.qualifier () + decl.name)
    overview_id = 'section-' + decl.get_xref_id () + '-overview'
    summary_id = 'section-' + decl.get_xref_id () + '-summary'
    copy_description = "Copy semantics of this object are undocumented. FIXME"
    memory_description = "Memory management of this object is undocumented. FIXME"
    short = "No short description for this object. FIXME"
    long = "<para>No long description for this object. FIXME</para>"
    if decl.doc:
        sem = decl.doc.get_copy_semantics ()
        if sem:
            copy_description = get_copy_description (decl, sem)
        sem = decl.doc.get_memory_model ()
        if sem:
            memory_description = get_memory_description (decl, sem)
        tmp = decl.doc.get_short ()
        if tmp:
            short = resolve_magic_links (decl, tmp)
        tmp = decl.doc.get_long ()
        if tmp:
            long = resolve_magic_links (decl, tmp)


    superclasses = ''
    inner_classes = ''
    enumerations = ''
    constructors = ''
    important_methods = ''
    important_functions = ''
    important_signals = ''
    virtual_methods = ''
    other_methods = ''
    other_functions = ''
    other_signals = ''
    notypes = ''

    supers = decl.get_documented_supers ()
    for s in supers:
        superclasses = superclasses + make_listitem (make_link_from_name (s, decl, 'class'))

    children = decl.get_documented_children ()
    for c in children:
        if c.type == 'method':
            if c.is_constructor:
                constructors = constructors + funcdecl_to_listitem (c)
            elif c.is_important:
                important_methods = important_methods + funcdecl_to_listitem (c)
            elif c.is_virtual:
                virtual_methods = virtual_methods + funcdecl_to_listitem (c)
            else:
                other_methods = other_methods + funcdecl_to_listitem (c)
        elif c.type == 'function':
            if c.is_important:
                important_functions = important_functions + funcdecl_to_listitem (c)
            else:
                other_functions = other_functions + funcdecl_to_listitem (c)

        elif c.type == 'class' or c.type == 'struct':
            inner_classes = inner_classes + decl_to_listitem (c)

    if superclasses != '':
        superclasses = listpara (superclasses, 'Superclasses')

    if inner_classes != '':
        inner_classes = listpara (inner_classes, 'Inner Classes')

    if enumerations != '':
        enumerations = listpara (enumerations, 'Enumerations')

    notypes = ''
    if superclasses == '' and inner_classes == '' and enumerations == '':
        notypes = '<para>No types related to this class.</para>'

    if constructors != '':
        constructors = listpara (constructors, 'Constructors')

    if important_methods != '':
        important_methods = listpara (important_methods, 'Important Methods')

    if important_functions != '':
        important_functions = listpara (important_functions, 'Important Functions')

    if important_signals != '':
        important_signals = listpara (important_signals, 'Important Signals')

    if virtual_methods != '':
        virtual_methods = listpara (virtual_methods, 'Virtual Methods')

    if other_methods != '':
        other_methods = listpara (other_methods, 'Other Methods')

    if other_functions != '':
        other_functions = listpara (other_functions, 'Other Functions')

    if other_signals != '':
        other_signals = listpara (other_signals, 'Other Signals')

    vals = { 
        'id' : decl.get_xref_id (),
        'title' : decl.type + ' ' + qualified_name,
        'overview_id' : overview_id,
        'summary_id' : summary_id,
        'copy_description' : copy_description,
        'memory_description' : memory_description,
        'qualified_name' : qualified_name,
        'superclasses' : superclasses,
        'subclasses' : '', ## hard to implement
        'inner_classes' : inner_classes,
        'enumerations' : enumerations,
        'notypes' : notypes,
        'constructors' : constructors,
        'important_methods' : important_methods,
        'important_functions' : important_functions,
        'important_signals' : important_signals,
        'virtual_methods' : virtual_methods,
        'other_methods' : other_methods,
        'other_functions' : other_functions,
        'other_signals' : other_signals,
        'short' : short,
        'long' : long,
        }
    
    outfile.write (class_template % vals)

    ## output the index stuff
    for c in decl.get_documented_children ():
        pass

    ## then output the toplevel entries for each child
    for c in decl.get_documented_children ():
        output_child (c, outfile)


method_template = """

      <refsect1 id="%(id)s">
        <title>%(name)s ()</title>

        <para>
        %(short)s
        </para>

        <para>
        <programlisting>
        %(declaration)s
        </programlisting>
        </para>

        <formalpara>
        <title>Parameters</title>
        <para>
        FIXME
        </para>
        </formalpara>

        <formalpara>
        <title>Description</title>
        
        %(long)s

        </formalpara>

      </refsect1>
"""

def output_method_or_function (decl, outfile):
    qualified_name = sgmlconvert (decl.qualifier () + decl.name)
    short = 'No short description, FIXME'
    long = '<para>No long description, FIXME</para>'

    if decl.doc:
        tmp = decl.doc.get_short ()
        if tmp:
            short = resolve_magic_links (decl, tmp)
        tmp = decl.doc.get_long ()
        if tmp:
            long = resolve_magic_links (decl, tmp)

    vals = {
        'id' : decl.get_xref_id (),
        'name' : qualified_name,
        'short' : short,
        'declaration' : sgmlconvert (decl.decl),
        'long' : long
        }

    outfile.write (method_template % vals)

def output_child (decl, outfile):
    
    if decl.type == 'method' or \
       decl.type == 'function':
        output_method_or_function (decl, outfile)
    elif decl.type == 'class' or \
         decl.type == 'struct':
        output_class_or_struct (decl, outfile)
    elif decl.type == 'enum':
        #        output_enum (decl, outfile)
        pass
    elif decl.type == 'typedef':
        pass
    elif decl.type == 'variable':
        pass
    elif decl.type == 'namespace':
        output_namespace (decl, outfile)
    else:
        message.die ("didn't handle decl type " + decl.type)


header_template = """
    <refentry id="%(id)s">

      <refmeta>
        <refentrytitle>%(name)s</refentrytitle>
        <manvolnum>3</manvolnum>
      </refmeta>
      
      <refnamediv>
        <refname>%(name)s</refname>

        <refpurpose>
        %(short)s
        </refpurpose>

      </refnamediv>

      <refsect1>
      <title>Types declared in this header</title>

      %(all_types)s
      
      </refsect1>

"""

def output_db (header, outfile):
    
    vals = {}
    vals['id'] = header.get_xref_id ()
    vals['name'] = header.name
    short =  header.doc.get_short ()
    if not short:
        message.die ("header " + header.name + " lacks short desc")
    vals['short'] = resolve_magic_links (header, short)

    typelist = []
    decls = header.get_child_types ()
    for d in decls:
        typelist.append (make_link_from_decl (d))

    typelist.sort ()

    all_types = '<para>\n' + join (typelist, ', ') + '\n</para>\n'
    
    vals['all_types'] = all_types
    
    outfile.write (header_template % vals)

    for c in header.get_documented_children ():
        output_child (c, outfile)

    outfile.write ('</refentry>\n')    
    
