#!/usr/bin/env python
# -*- Mode: Python; py-indent-offset: 4 -*-

try:
    import cString
    string = cString
except:
    import string

import os
import sys
import re
import exceptions
from declarations import *
import cgi


class TemplateEntry:
    def __init__ (self):
        self.short = None
        self.long = None
        self.decl = None
        self.is_important = 0
        self.leftover = ''
        self.type = None
        self.decl_name = None
        self.decl_qualified_name = None
        self.decl_type = None
        self.decl_xref = None
        self.not_documented = 0

    def set_decl (self, decl):
        self.decl = decl
        self.decl_name = decl.name
        self.decl_type = decl.type
        self.decl_xref = decl.xref_id

class HeaderEntry (TemplateEntry):
    def __init__ (self):
        TemplateEntry.__init__(self)
        self.type = 'header'

class ClassEntry (TemplateEntry):
    def __init__ (self):
        TemplateEntry.__init__(self)
        self.memory = ''
        self.copy = ''
        self.type = 'class'

class MethodEntry (TemplateEntry):
    def __init__ (self):
        TemplateEntry.__init__(self)
        self.returns = ''
        self.params = [] # list of (name, docs) tuples
        self.type = 'method'

class VariableEntry (TemplateEntry):
    def __init__ (self):
        TemplateEntry.__init__(self)
        self.type = 'variable'
    
class TypedefEntry (TemplateEntry):
    def __init__ (self):
        TemplateEntry.__init__(self)
        self.type = 'typedef'

class EnumEntry (TemplateEntry):
    def __init__ (self):
        TemplateEntry.__init__(self)
        self.type = 'enum'
        self.vals = [] # list of (name, docs) tuples

def entry_from_decl (decl):
    entry = None
    if decl.type == 'method' or \
       decl.type == 'function':
        entry = MethodEntry ()
    elif decl.type == 'class' or \
         decl.type == 'struct':
        entry = ClassEntry ()
    elif decl.type == 'enum':
        entry = EnumEntry ()
    elif decl.type == 'typedef':
        entry = TypedefEntry ()
    elif decl.type == 'variable':
        entry = VariableEntry ()
    elif decl.type == 'header':
        entry = HeaderEntry ()
    else:
        message.die ("didn't handle decl type " + decl.type)

    entry.set_decl (decl)

    return entry

entries = {}
            
def merge_template (header_decl, target_dir, xref_id_dict):
    name = header_decl.name
    if name[-2:] != '.h':
        message.die ("header name must end with .h")
    base = os.path.basename(name[:-2])
    template_file = os.path.join(target_dir, base + '.template.sgml')
    orig_file = None
    try:
        orig_file = open (template_file, 'r')
    except:
        orig_file = None

    if orig_file:
        text = orig_file.read ()

        parse_entries (entries, text, xref_id_dict)

    merge_entries (entries, xref_id_dict)

    backup = template_file + '.bak'
    try:
        os.unlink (backup + '.bak2')
        os.rename (backup, backup + '.bak2')
    except:
        pass
    try:
        os.unlink (backup)
        os.rename (template_file, backup)
    except:
        pass

    new_filename = template_file + '.new'

    new_file = open (new_filename, 'w')

    write_entries (entries, new_file)

    try:
        os.unlink (template_file)
    except:
        pass

    os.rename (new_filename, template_file)

def load_template (entries, template_file, xref_id_dict):
    text = open (template_file, 'r').read ()
    parse_entries (entries, text, xref_id_dict)

# The magic comment has first a the type of the decl, then
# the xref ID
entry_start = re.compile ('<!--- *#### *([A-Za-z]+) *([^# ]+) *#### *--->')
short_desc = re.compile ('<!--short-->(.*?)<!--/short-->', re.DOTALL)
long_desc = re.compile ('<!--long-->(.*?)<!--/long-->', re.DOTALL)
param_desc = re.compile ('<!--param name="([^"]*)"-->(.*?)<!--/param-->', re.DOTALL)
badparam_desc = re.compile ('<!--badparam name="([^"]*)"-->(.*?)<!--/badparam-->', re.DOTALL)
enumval_desc = re.compile ('<!--enumval name="([^"]*)"-->(.*?)<!--/enumval-->', re.DOTALL)
badenumval_desc = re.compile ('<!--badenumval name="([^"]*)"-->(.*?)<!--/badenumval-->', re.DOTALL)
returns_desc = re.compile ('<!--returns-->(.*?)<!--/returns-->', re.DOTALL)
important_desc = re.compile ('<!-- *important:(.*)-->')
nodoc_desc = re.compile ('<!-- *document:(.*)-->')
name_desc = re.compile ('<!-- *name:(.*)-->')
copy_desc = re.compile ('<!-- *copy:(.*)-->')
memory_desc = re.compile ('<!-- *memory:(.*)-->')
leftovers = re.compile ('<!--leftovers-->(.*?)<!--/leftovers-->', re.DOTALL)

def parse_entries (entries, text, xref_id_dict):
    tokens = entry_start.split (text)

    # remove bogus all-whitespace tokens, and strip extra whitespace
    new_tokens = []
    for t in tokens:
        str = strip (t)
        if str != '':
            new_tokens.append (str)
    tokens = new_tokens

    while tokens:
        type = tokens[0]
        xref = tokens[1]
        docs = tokens[2]
        tokens = tokens[3:]

        entry = None
        if type == 'method' or \
           type == 'function':
            entry = MethodEntry ()
        elif type == 'class' or \
             type == 'struct':
            entry = ClassEntry ()
        elif type == 'enum':
            entry = EnumEntry ()
        elif type == 'typedef':
            entry = TypedefEntry ()
        elif type == 'variable':
            entry = VariableEntry ()
        elif type == 'header':
            entry = HeaderEntry ()
        else:
            message.die ("didn't handle decl type from template file: " + type)

        if xref_id_dict.has_key (xref):
            decl = xref_id_dict[xref]
            if decl.type != type:
                message.die ("Type mismatch for declaration and template entry")
            if decl.xref_id != xref:
                message.die ("xref ID doesn't match for decl and template entry")
            entry.set_decl (decl)
        else:
            message.warn ("no declaration in header for xref ID " + xref)
            entry.decl_type = type
            entry.decl_xref = xref

        entries[xref] = entry

        match = short_desc.search (docs)
        if match:
            entry.short = strip (match.group (1))
            docs = strip (short_desc.sub ('', docs, 1))

        match = long_desc.search (docs)
        if match:
            entry.long = strip (match.group (1))
            docs = strip (long_desc.sub ('', docs, 1))

        match = name_desc.search (docs)
        if match:
            entry.decl_qualified_name = strip (match.group (1))
            docs = name_desc.sub ('', docs, 1)

        match = important_desc.search (docs)
        if match:
            val = strip (match.group (1))
            if val == 'yes':
                entry.is_important = 1
            elif val != 'no':
                message.die ("invalid importance (should be yes/no)")
            docs = strip (important_desc.sub ('', docs, 1))

        match = nodoc_desc.search (docs)
        if match:
            val = strip (match.group (1))
            if val == 'no':
                entry.not_documented = 1
            elif val != 'yes':
                message.die ("invalid  (should be yes/no)")
            docs = strip (nodoc_desc.sub ('', docs, 1))

        if entry.type == 'class':
            match = copy_desc.search (docs)
            if match:
                entry.copy = strip (match.group (1))
                docs = strip (copy_desc.sub ('', docs, 1))

            if not entry.copy:
                entry.copy = ''

            match = memory_desc.search (docs)
            if match:
                entry.memory = strip (match.group (1))
                docs = strip (memory_desc.sub ('', docs, 1))

            if not entry.memory:
                entry.memory = ''

        elif entry.type == 'method':
            params = param_desc.findall (docs)
            if params and params != []:
                for (name, desc) in params:
                    entry.params.append ( (strip(name), strip(desc)) )
            docs = strip (param_desc.sub ('', docs))

            params = badparam_desc.findall (docs)
            if params and params != []:
                for (name, desc) in params:
                    entry.params.append ( (strip(name), strip(desc)) )
            docs = strip (badparam_desc.sub ('', docs))

            match = returns_desc.search (docs)
            if match:
                entry.returns = strip (match.group (1))
                docs = strip (returns_desc.sub ('', docs, 1))

            if not entry.returns:
                entry.returns = ''

        elif entry.type == 'enum':
            enumvals = enumval_desc.findall (docs)
            if enumvals and enumvals != []:
                for (name, desc) in enumvals:
                    entry.vals.append ( (strip(name), strip(desc)) )
            docs = strip (enumval_desc.sub ('', docs))

            enumvals = badenumval_desc.findall (docs)
            if enumvals and enumvals != []:
                for (name, desc) in enumvals:
                    entry.vals.append ( (strip(name), strip(desc)) )
            docs = strip (badenumval_desc.sub ('', docs))

        entry.leftover = strip (docs)

def merge_entries (entries, xref_id_dict):
    for xref in xref_id_dict.keys ():
        if entries.has_key (xref):
            pass 
        else:
            # create new TemplateEntry for this decl, if non-private
            decl = xref_id_dict[xref]
            if not decl.is_private and \
               decl.type != 'namespace':
                entries[xref] = entry_from_decl (decl)

entry_format = """

<!--- #### %(type)s %(xref)s #### --->
<!-- name: %(qualified_name)s -->
<!-- important: %(important)s -->
<!-- document: %(documented)s -->

<!--short-->
%(short)s
<!--/short-->

<!--long-->
%(long)s
<!--/long-->

"""

class_entry_format = """<!-- copy: %s -->
<!-- memory: %s -->
"""

method_entry_format = """<!--returns-->%s<!--/returns-->
"""

param_format = """<!--param name="%s"-->
%s
<!--/param-->
"""

bad_param_format = """<!--badparam name="%s"-->
%s
<!--/badparam-->
"""

enumval_format =  """<!--enumval name="%s"-->
%s
<!--/enumval-->
"""

bad_enumval_format = """<!--badenumval name="%s"-->
%s
<!--/badenumval-->
"""

def output_method_entry (entry, file):
    if entry.decl and \
       (entry.decl.returns == 'void' or entry.decl.returns == ''):
        if entry.returns != '':
            message.warn ("Return value for %s documented, but it returns void: %s" % (entry.decl.name, entry.returns))
            file.write (method_entry_format % (entry.returns))
    else:
        file.write (method_entry_format % (entry.returns))

    param_docs = {}
    if entry.decl:
        for (type, name, default) in entry.decl.args:
            param_docs[name] = None

    existing_bad_docs = []
    for (name, docs) in entry.params:
        if param_docs.has_key (name):
            if param_docs[name] != None:
                message.warn ("Duplicate docs for param %s of %s" % (name, entry.decl_name))
                existing_bad_docs.append (name, docs)
            else:
                param_docs[name] = docs
        else:
            existing_bad_docs.append ((name, docs))

    # entry.decl.args is supposed to be in order
    if entry.decl:
        for (type, name, default) in entry.decl.args:
            if name == '':
                message.warn ("No parameter name in %s" % entry.decl.name)
            docs = param_docs[name]
            if not docs:
                docs = ''
            file.write (param_format % (name, docs))

    for (name, docs) in existing_bad_docs:
        message.warn ("Parameter %s for %s no longer valid" % (name, entry.decl_name))
        file.write (bad_param_format % (name, docs))

def output_enum_entry (entry, file):
    enumval_docs = {}
    if entry.decl:
        for name in entry.decl.vals:
            enumval_docs[name] = None

    existing_bad_docs = []
    for (name, docs) in entry.vals:
        if enumval_docs.has_key (name):
            if enumval_docs[name] != None:
                message.warn ("Duplicate docs for enumval %s of %s" % (name, entry.decl_name))
                existing_bad_docs.append (name, docs)
            else:
                enumval_docs[name] = docs
        else:
            existing_bad_docs.append ((name, docs))

    # entry.decl.vals is supposed to be in order
    if entry.decl:
        for name in entry.decl.vals:
            if name == '':
                message.warn ("No enum value name in %s" % entry.decl.name)
            docs = enumval_docs[name]
            if not docs:
                docs = ''
            file.write (enumval_format % (name, docs))

    for (name, docs) in existing_bad_docs:
        message.warn ("Enum value %s for %s no longer valid" % (name, entry.decl_name))
        file.write (bad_enumval_format % (name, docs))


def write_entries (entries, file):
    sorted = entries.values ()
    def sort_by_serial (a, b):
        if a.decl and b.decl:
            return cmp (a.decl.serial, b.decl.serial)
        # without decl is always less than with decl
        elif a.decl:
            return 1
        elif b.decl:
            return -1
        else:
            return 0
        
    sorted.sort (sort_by_serial)
    for entry in sorted:        
        if not entry.long:
            entry.long = '<para>\n\n</para>'

        if not entry.short:
            entry.short = ''

        important = 'no'
        if entry.is_important:
            important = 'yes'

        documented = 'yes'
        if entry.not_documented:
            documented = 'no'

        if not entry.decl_qualified_name:
            if entry.decl and entry.decl.qualified_name:
                entry.decl_qualified_name = entry.decl.qualified_name
            else:
                entry.decl_qualified_name = ''
            
        vals = {
            'qualified_name' : entry.decl_qualified_name,
            'type' : entry.decl_type,
            'xref' : entry.decl_xref,
            'short' : entry.short,
            'long' : entry.long,
            'important' : important,
            'documented' : documented
            }
        file.write (entry_format % vals)

        if entry.type == 'class':
            file.write (class_entry_format % (entry.copy, entry.memory))
        elif entry.type == 'method':
            output_method_entry (entry, file)
        elif entry.type == 'enum':
            output_enum_entry (entry, file)
                
        if entry.leftover != '':
            message.warn ("leftover: %s" % entry.leftover)
            file.write (entry.leftover)
