#!/usr/bin/env python
# -*- Mode: Python; py-indent-offset: 4 -*-

# as always, my Python is ripped off of James Henstridge 


try:
    from cString import *
except:
    from string import *

import sys
import os
import re
import exceptions
import errno
import getopt
import message
#import docobjects
#import docoutput
from declarations import *
import doctemplate

verbose = 0
def verbosity (str):
    if verbose:
        print str

class Block:
    def __init__ (self):
        # each element of the list is a pair, where either the
        # car or cdr is always None. If the car is not None,
        # it's some text; if the cdr is not None, it's
        # a sub-block.
        self.contents = []

    def append_line (self, line):
        self.contents.append ( (line, None) )

    def append_block (self, block):
        self.contents.append ( (None, block) )

    def spew (self, indent):
        for (line, block) in self.contents:
            if line:
                print indent + line
            else:
                block.spew (indent + "  ")

class BlockStack:
    def __init__ (self):
        self.root_block = Block ()
        self.blocks = [self.root_block]
        self.current_top = self.root_block

    def add_line (self, line):
        self.current_top.append_line (line)

    def open_block (self):
        self.blocks.append (Block ())
        self.current_top = self.blocks[-1]

    def close_block (self):
        top_block = self.blocks.pop ()
        self.current_top = self.blocks[-1]
        self.current_top.append_block (top_block)

    def verbosity (self):
        if verbose:
            self.root_block.spew ("")


class DeclProcessContext:
    def __init__ (self):
        self.decl_stack = []
        self.access = ['global']
        self.name_pending = None
        self.pending_docs = []

    def push_decl (self, decl):
        self.decl_stack.append (decl)

    def pop_decl (self):
        return self.decl_stack.pop ()

    def root_decl (self):
        return self.decl_stack[0]

    def top_decl (self):
        return self.decl_stack[-1]

    def top_class (self):
        c = None
        for d in self.decl_stack:
            if d.type == 'class':
                c = d

        return c

    def top_namespace (self):
        n = None
        for d in self.decl_stack:
            if d.type == 'namespace':
                n = d

        return n

    def set_access (self, access):
        self.access[-1] = access

    def push_access (self, access):
        self.access.append (access)

    def pop_access (self):
        self.access.pop ()

    def top_access (self):
        return self.access[-1]

    def set_name_pending (self, decl):
        if self.name_pending:
            message.die ("set new name pending without resolving old one")
        self.name_pending = decl

    def is_name_pending (self):
        return self.name_pending != None

    def found_name (self, name):
        self.name_pending.set_name (name)
        self.name_pending = None

    def get_pending_doc (self, type):
        i = 0
        while i < len (self.pending_docs):
            d = self.pending_docs[i]
            if d.type == type:
                self.pending_docs.pop (i)
                alternative = self.get_pending_doc (type)
                if alternative != None:
                    message.warn ("Two candidate docs found!")
                return d

        return None
    
    def add_pending_doc (self, doc):
        self.pending_docs.append (doc)

up_to_semicolon = re.compile ('(.*);', re.DOTALL)
semicolon = re.compile (r';', re.MULTILINE)
namespace_header = re.compile ('namespace ([a-zA-Z_0-9]+)', re.MULTILINE)
class_header = re.compile ('(template.*\n)?.*class ([a-zA-Z_<>0-9, ]+)', re.MULTILINE)
struct_header = re.compile ('struct ([a-zA-Z_<>0-9]+)', re.MULTILINE)
enum_header = re.compile ('typedef enum', re.MULTILINE)
access_line = re.compile ('(protected|public|private):')
static_function_header = re.compile ('((template.*\n)?[^(;{}]*static [^(]+\(.*\))', re.MULTILINE)
function_header = re.compile ('((template.*\n)?.*\))', re.MULTILINE)

function_decl = re.compile ('([^(]*\([^)]*\).*;)', re.DOTALL)
variable_decl = re.compile ('.*;', re.DOTALL)
typedef_decl = re.compile ('typedef .*;', re.DOTALL)
docs_decl = re.compile ('doc ([0-9]+);', re.MULTILINE)

constructor_initializers = re.compile ('[^:]:[^:].*', re.MULTILINE)

def process_doc (block, context, docnum):
    index = atoi (docnum)
    doc = docs[index]
    
    # class docs are inside the class, other docs
    # are before the corresponding declaration
    if doc.type == 'class':
        c = context.top_class ()
        c.set_doc (doc)
    elif doc.type == 'header':
        context.root_decl ().set_doc (doc)
    elif doc.type == 'topic':
        context.root_decl ().add_topic_doc (doc)
    else:
        context.add_pending_doc (doc)

def process_statement (statement, block, context):
    #    verbosity ("statement: " + statement + '\n')
    if context.is_name_pending ():
        match = up_to_semicolon.search (statement)
        if match:
            context.found_name (match.group (1))
            return
        else:
            message.die ("pending name not found") 

    match = docs_decl.search (statement)
    if match:
        docnum = match.group (1)
        process_doc (block, context, docnum)
        return

    match = typedef_decl.search (statement)
    if match:
        t = Typedef ()
        t.set_decl (statement)
        t.set_access (context.top_access ())
        context.top_decl ().add_child (t, context)

        return
        
    match = function_decl.search (statement)
    if match:
        d = None
        has_static = (find (match.group (1), 'static') >= 0)
        if not has_static and context.top_decl ().type == 'class':
            d = Method ()
        else:
            d = Function ()
        d.set_decl (match.group (1))
        d.set_access (context.top_access ())
        context.top_decl ().add_child (d, context)
        # don't care about contents of the function block
        
        return
        
    match = variable_decl.search (statement)
    if match:
        v = Variable ()
        v.set_decl (statement)
        v.set_access (context.top_access ())
        context.top_decl ().add_child (v, context)

        return

def recurse_block (block, context):
    child_header = None
    for (line, child) in block.contents:
        if line:
            #           verbosity ("line: " + line + '\n')
            if line == ';':
                ## child_header so far is a statement, not
                ## the start of a block.
                if child_header:
                    process_statement (child_header + ';', block, context)
                    child_header = None
            else:
                match = access_line.search (line)
                if match:
                    context.set_access (match.group (1))
                else:
                    if child_header:
                        child_header = child_header + '\n' + line
                    else:
                        child_header = line
        else:
            process_block (child_header, child, context)
            child_header = None    

def process_block (header, block, context):

#    verbosity ("header: " + header + '\n')

    match = docs_decl.search (header)
    if match:
        docnum = match.group (1)
        process_doc (block, context, docnum)        
        return
    
    match = namespace_header.search (header)
    if match:
        name = match.group (1)
        space = Namespace ()
        context.top_decl ().add_child (space, context)
        space.set_name (name)
        context.push_decl (space)
        recurse_block (block, context)
        context.pop_decl ()
        return

    match = static_function_header.search (header)
    if match:
        f = Function ()
        header = constructor_initializers.sub ('', header)
        f.set_decl (header)
        f.set_access (context.top_access ())
        context.top_decl ().add_child (f, context)
        # don't care about contents of the function block
        
        return

    match = function_header.search (header)
    if match:
        d = None
        if context.top_decl ().type == 'class':
            d = Method ()
        else:
            d = Function ()
        header = constructor_initializers.sub ('', header)
        d.set_decl (header)
        d.set_access (context.top_access ())
        context.top_decl ().add_child (d, context)
        # don't care about contents of the function block
        
        return

    match = class_header.search (header)
    if match:
        name = match.group (2)
        c = Class ()
        context.top_decl ().add_child (c, context)
        c.set_name (name)
        c.set_decl (header)
        c.set_access (context.top_access ())
        context.push_access ('private')
        context.push_decl (c)
        recurse_block (block, context)
        context.pop_decl ()
        context.pop_access ()
        return

    match = struct_header.search (header)
    if match:
        name = match.group (1)
        c = Struct ()
        context.top_decl ().add_child (c, context)
        c.set_name (name)
        c.set_decl (header)
        context.push_access ('public')
        context.push_decl (c)
        recurse_block (block, context)
        context.pop_decl ()
        context.pop_access ()
        return
    
    match = enum_header.search (header)
    if match:
        e = Enum ()
        e.set_access (context.top_access ())
        context.top_decl ().add_child (e, context)
        context.set_name_pending (e)
        text = ""
        for (line, child) in block.contents:
            if line:
                text = text + line
            else:
                message.die ("found child block inside an enum?")

        e.set_decl (text)
        return

    ## if all else fails, just ignore this block
    ## and look at its sub-blocks
    recurse_block (block, context)
    
        
c_style_comments = re.compile (r"""/[*](.|\n)*?[*]/""", re.MULTILINE)
cpp_style_comments = re.compile (r"""//.*\n""", re.MULTILINE)
def strip_comments (str):
    str = cpp_style_comments.sub ('', str)
    str = c_style_comments.sub ('\n', str)
#    verbosity ("after comments stripped:\n" + str);
    return str

preprocessor_directive = re.compile(r"""^[#].*$""", re.MULTILINE)
spaces =  re.compile('[ \t]+', re.MULTILINE)
trailing_space = re.compile(r"""\s+\n""", re.MULTILINE)
leading_space = re.compile ('^\s+', re.MULTILINE)
left_brace = re.compile (r'{', re.MULTILINE)
right_brace = re.compile (r'}', re.MULTILINE)
forward_decl = re.compile (r'(class|struct) [A-Za-z:<>_0-9]+\s*;', re.MULTILINE)
friend_decl = re.compile (r'friend [^;]+;', re.MULTILINE)
gnuc_printf = re.compile (' *G_GNUC_[^;]+', re.MULTILINE)
# You could think of cleanup() as a primitive tokenizer;
# it removes all the crufty stuff, and puts "tokens" on lines
# by themselves.
def cleanup (str):
    str = strip_comments (str)
    
    # Preprocessor directives
    str = preprocessor_directive.sub ('', str)

    # remove G_GNUC_PRINTF
    str = gnuc_printf.sub ('', str)

    # put curly braces and semicolons on lines by themselves
    str = left_brace.sub ('\n{\n', str)
    str = right_brace.sub ('\n}\n', str)
    str = semicolon.sub ('\n;\n', str)

    # remove friends (must be before forward decls, since they contain
    # 'class Blah')
    str = friend_decl.sub ('', str)

    # remove forward declarations
    str = forward_decl.sub ('', str)

    # multiple whitespace
    str = spaces.sub (' ', str)

    # clean up line ends
    str = trailing_space.sub ('\n', str)

    # remove all indentation
    str = leading_space.sub ('', str)

    #associate *, &, and [] with type instead of variable
    #pat=re.compile(r'\s+([*|&]+)\s*(\w+)')
    #    pat=re.compile(r' \s+ ([*|&]+) \s* (\w+)',re.VERBOSE)
    #    buf=pat.sub(r'\1 \2', buf)
    #    pat=re.compile(r'\s+ (\w+) \[ \s* \]',re.VERBOSE)
    #    buf=pat.sub(r'[] \1', buf)

    # verbosity ("after cleanup:\n" + str)

    return str

doc_counter = 0
docs = {}
doc_comment = re.compile (r"""/[*][$](.|\n)*?[$][*]/""", re.MULTILINE)
def doc_comment_matched (match):
    global doc_counter, docs
    doc_counter = doc_counter + 1
    docs[doc_counter] = docobjects.parse_doc_comment (match.groups()[-1])
    return 'doc %d;\n' % doc_counter

def process_file (filename, target_dir, doc_template_dir, include_base):
    contents = open(filename).read ()

    # filter all doc comments, we aren't doing that anymore
    #    contents = doc_comment.sub (doc_comment_matched, contents)

    contents = cleanup (contents)
    
    stack = BlockStack ()
    lines = split (contents, '\n')

    for line in lines:
        if line == '{':
            stack.open_block ()
        elif line == '}':
            stack.close_block ()
        elif line != '':
            stack.add_line (line)

    root_decl = Header ()
    root_decl.set_name (os.path.join (include_base, os.path.basename (filename)))
    decl_context = DeclProcessContext ()
    decl_context.push_decl (root_decl)
    process_block ("", stack.root_block, decl_context)
    decl_context.pop_decl ()

    postprocess_decls (root_decl)

    #    docoutput.output_header_docs (root_decl, target_dir)

    doctemplate.merge_template (root_decl, doc_template_dir, xref_id_dict)

    dump_file = os.path.basename (filename)[:-2] + '.pyobjects'
    dump_file = os.path.join (target_dir, dump_file)
    dump_decls (dump_file)

if os.environ.has_key ('INTI_NO_DOCS'):
    print "  not generating documentation because INTI_NO_DOCS is set"
    sys.exit (0)

header_dir = None
include_base = None
script_dir = None
docs_dir = None
file_basename = None
doc_template_dir = None

opts, args = getopt.getopt(sys.argv[1:], 'v', ['header-dir=','include-base=','script-dir=','docs-dir=', 'template=', 'doc-template-dir='])
for opt, val in opts:
    if opt == '-v':
        verbose = 1
    elif opt == '--template':
        template = val
        if template[-4:] != '.gen':
            die ("template must end in .gen")
        file_basename = template[:-4]
    elif opt == '--include-base':
        include_base = val
    elif opt == '--script-dir':
        script_dir = val
    elif opt == '--docs-dir':
        docs_dir = val
    elif opt == '--header-dir':
        header_dir = val
    elif opt == '--doc-template-dir':
        doc_template_dir = val

if not header_dir:
    message.die ("must specify header dir")

if not file_basename:
    message.die ("must specify template filename")

if not include_base:
    message.die ("must specify include base")

if not script_dir:
    message.die ("must specify script dir")

if not docs_dir:
    message.die ("must specify docs dir")

if not doc_template_dir:
    message.die ("must specify doc template dir")

try:
    os.mkdir (docs_dir)
except OSError, e:
    if e.errno != errno.EEXIST:
        message.die (e)

try:
    os.mkdir (doc_template_dir)
except OSError, e:
    if e.errno != errno.EEXIST:
        message.die (e)

process_file (os.path.join (header_dir, file_basename + '.h'),
              docs_dir, doc_template_dir, include_base)


