// -*- C++ -*-
// Copyright (C) 2000 Red Hat, Inc.
#ifndef _INTI_BASE_PTR_H_
#define _INTI_BASE_PTR_H_

/*
  Inti is a C++ development platform based on glib, GTK+, and the
  Standard Template Library. This header file is machine-generated
  by a script; do not edit it directly.

  This file is distributed under the GNU Library General Public License.
*/
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*$ doc header
  $short Smart pointers
  $*/

namespace Inti
{
  
  namespace Private
  {
    template <class T>
    class ptr_base
    {
    protected:
      ptr_base (T *obj = 0)
        : obj_ (0)
      {
        grab (obj);
      }
      
      ~ptr_base ()
      {
        grab (0);
      }
      
      void _set(T *obj)
      {
        grab (obj);
      }
      
      T* _get() const { return obj_; }

    private:
      T *obj_;

      // Keeping this inline apparently results in less code being
      // generated
      void grab(T *obj)
      {
        // Ref new object first, to handle self assignment properly.
        if (obj != 0)
          {
            obj->ref();
            obj->sink();
          }
      
        if (obj_ != 0)
          obj_->unref();

        obj_ = obj;
      }
    };
    
    template <class T>
    class strict_ptr_base
    {
    protected:
      strict_ptr_base (T *obj = 0)
        : obj_ (0)
      {
        grab (obj);
      }
      
      ~strict_ptr_base ()
      {
        grab (0);
      }
      
      void _set(T *obj)
      {
        grab (obj);
      }
      
      T* _get() const { return obj_; }

    private:
      T *obj_;

      void grab(T *obj)
      {
        // Ref new object first, to handle self assignment properly.
        if (obj != 0)
          obj->ref();
        
        if (obj_ != 0)
          obj_->unref();
        
        obj_ = obj;
      }
    };
    
  }; // namespace Private
  
  // Pointer which holds a reference to an object
  // with ref/unref/is_floating/sink methods
  template <class T>
  class ptr : private Private::ptr_base<T>
  {
    /*$ doc class ptr
      $short Smart pointer for objects using the "floating reference count" memory model
      $memory c++
      $copy value

      $long

      <para>

      The #Inti::ptr smart pointer is intended for use with objects
      that implement the "floating reference count" memory management
      model ([memory:floating]).  This means the object implements
      ref(), unref(), and sink() methods.

      </para>

      <para>

      A #Inti::ptr simply holds a reference to the object it points
      to. It will also sink the object, if the object is floating.
      This means that while a #Inti::ptr exists which points to
      a given object, the object will not be destroyed. It also means
      that the object may be destroyed when the #Inti::ptr is destroyed,
      if the #ptr held the last reference to the object.

      </para>
      
      $*/
    
    typedef Private::ptr_base<T> base;
  public:
    typedef T element_type;

    ptr() : base(0) {}      
    
    explicit ptr(T* obj)  : base(obj) {}
    
    ptr(ptr& src)
    {
      base::_set(src.base::_get());
    }

    // See Stroustrup 13.6.3.1 for how this is supposed to work.
    template <class T1> ptr(const ptr<T1>& src)
    {
      reset(src.get());
    }
    
    ptr& operator=(const ptr& src)
    {
      base::_set(src.base::_get());
      return *this;
    }

    ptr& operator=(T* obj)
    {
      base::_set(obj);
      return *this;
    }

    // See Stroustrup 13.6.3.1 for how this is supposed to work.
    template <class T1>
    ptr& operator=(const ptr<T1>& src)
    {
      reset(src.get());
      return *this;
    }
      
    ~ptr()
    {
      base::_set(0);
    }

    T& operator*() const
    {
      return *get();
    }

    T* operator->() const
    {
      return get();
    }

    // This is normally considered pretty evil (see p 171 of "More
    // Effective C++"), but the two reasons it's metioned as evil there
    // don't apply:
    //   - nothing will break if people mix smart and regular pointers;
    //     they just have to follow the refcounting rules as normal
    //     for each of them.
    //   - "delete smart_pointer" won't happen, because delete is
    //     forbidden on objects with refcounting in Inti
    
    operator T*() const { return get(); }
    
    /*$ doc method
      $short return a dumb pointer (regular C pointer) to our object
      $*/
    T* get() const
    {
      return base::_get();
    }

    /*$ doc method
      $short return the object we point to, then set ourselves to 0, without unreferencing the object (i.e. release our reference count to someone else)
      $long
      <para>
      release() is a convenient way to obtain the smart pointer's reference
      to the object for yourself. For example:
      <programlisting>
       ptr&lt;Foo&gt; f (new Foo);
       Foo * my_foo = f.release (); // now f == 0, and we own a reference
       my_foo-&gt;unref ();            // we must call unref(), since we own a ref
      </programlisting>
      release() is equivalent to this:
      <programlisting>
      ptr&lt;Foo&gt; f (new Foo);
      Foo * my_foo = f.get ();
      my_foo-&gt;ref ();               // grab a reference
      f = 0;                        // set f to zero, causing it to drop its reference
      </programlisting>
      </para>
      
      $*/
    T* release()
    {
      T *tmp = base::_get();
      if (tmp)
        tmp->ref();
      base::_set(0);
      return tmp;
    }

    /*$ doc method
      $short assign a new value to the smart pointer
      $long
      <para>
      Because reset()'s only argument defaults to 0, reset() is most
      commonly used to set the pointer to 0:
      <programlisting>
      ptr&lt;Foo&gt; f (new Foo);
      f.reset ();          // set f to 0
      f.reset (new Foo);   // set f to new value
      </programlisting>
      There is no real reason to use reset() instead of simple assignment:
      <programlisting>
      ptr&lt;Foo&gt; f (new Foo);
      f = 0;          // set f to 0
      f = new Foo;    // set f to new value
      </programlisting>
      </para>
      $*/
    void reset(T* obj = 0)
    {
      base::_set(obj);
    }
  };

  
  // Pointer which holds a reference to an object
  // with ref/unref/is_floating/sink methods
  template <class T>
  class strict_ptr : private Private::strict_ptr_base<T>
  {
    /*$ doc class strict_ptr
      $short Smart pointer for objects using a strict reference count memory model
      $memory c++
      $copy value

      $long

      <para>

      The #Inti::strict_ptr smart pointer is intended for use with
      objects that implement a pure reference count memory model
      ([memory:strict_refcount]).  This means the object implements
      ref() and unref(). You can use a #strict_ptr with an object that
      has floating reference count ([memory:floating]) as well, but
      be aware that the #strict_ptr will never sink() the object, that is,
      if there's a floating reference count it will not be removed.
      Conceptually, using #strict_ptr with a floating memory model object
      means that you are not willing to become the owner of the object,
      i.e. you are not willing to call the sink() method on it.
      </para>

      <para>

      A #Inti::strict_ptr simply holds a reference to the object it
      points to. #strict_ptr always adds a new reference to an object
      (unlike #ptr, which can "adopt" the object's floating reference
      by calling ref() then sink()). Unlike #ptr, #strict_ptr does not
      "take ownership" of an object and will not cause the object to
      be deleted.
      (All valid objects have a reference count of at least 1, and
      #strict_ptr always adds a reference; so, all else being equal,
      when the #strict_ptr drops its reference, the object's reference
      count remains at least 1.)

      </para>

      <para>

      Because #strict_ptr won't sink() an object, you always
      <emphasis>retain</emphasis> current references that you hold to
      an object; #strict_ptr does not adopt objects. For example:

      <programlisting>
      strict_ptr&lt;Foo&gt; p (new Foo);
      // the Foo instance now has a reference count of TWO,
      // because it was born with 1 reference, and strict_ptr
      // added a reference.

      // Thus it's necessary to unref() the Foo instance when we're
      // done with it
      
      p-&gt;unref (); // release the reference we owned

      // strict_ptr will drop its reference when the strict_ptr
      // goes out of scope, bringing the refcount to 0 and
      // deleting the Foo instance
      </programlisting>

      Compare this to #ptr:
      <programlisting>
      ptr&lt;Foo&gt; p (new Foo);
      // the Foo instance now has a reference count of ONE,
      // because it was born with a floating reference count;
      // the floating count was "adopted" by the ptr;
      // when the ptr goes out of scope, the Foo instance will be
      // deleted.
      </programlisting>
      </para>

      <para>
      For objects with no "floating" state, clearly you must
      use #strict_ptr rather than #ptr. However, you may want to
      use #strict_ptr with floatable objects as well, if your
      use of the object is transitory and you don't want to risk
      deleting the object unexpectedly.
      </para>
      
      $*/
    
    typedef Private::strict_ptr_base<T> base;
  public:
    typedef T element_type;

    strict_ptr() : base(0) {}      
    
    explicit strict_ptr(T* obj)  : base(obj) {}
    
    strict_ptr(strict_ptr& src)
    {
      base::_set(src.base::_get());
    }

    // See Stroustrup 13.6.3.1 for how this is supposed to work.
    template <class T1> strict_ptr(const strict_ptr<T1>& src)
    {
      reset(src.get());
    }
    
    strict_ptr& operator=(const strict_ptr& src)
    {
      base::_set(src.base::_get());
      return *this;
    }

    strict_ptr& operator=(T* obj)
    {
      base::_set(obj);
      return *this;
    }

    // See Stroustrup 13.6.3.1 for how this is supposed to work.
    template <class T1>
    strict_ptr& operator=(const strict_ptr<T1>& src)
    {
      reset(src.get());
      return *this;
    }
      
    ~strict_ptr()
    {
      base::_set(0);
    }

    T& operator*() const
    {
      return *get();
    }

    T* operator->() const
    {
      return get();
    }

    // This is normally considered pretty evil (see p 171 of "More
    // Effective C++"), but the two reasons it's metioned as evil there
    // don't apply:
    //   - nothing will break if people mix smart and regular pointers;
    //     they just have to follow the refcounting rules as normal
    //     for each of them.
    //   - "delete smart_pointer" won't happen, because delete is
    //     forbidden on objects with refcounting in Inti
    
    operator T*() const { return get(); }
    
    /*$ doc method
      $short return a dumb pointer (regular C pointer) to our object
      $*/
    T* get() const
    {
      return base::_get();
    }

    /*$ doc method 
      $short return the object we point to, then set ourselves to 0, without unreferencing the object (i.e. release our reference count to someone else)
      $long
      <para>
      release() is a convenient way to obtain the smart pointer's reference
      to the object for yourself. For example:
      <programlisting>
       strict_ptr&lt;Foo&gt; f (new Foo);
       Foo * my_foo = f.release (); // now f == 0, and we own a reference
       my_foo-&gt;unref ();            // we must call unref(), since we own a ref
                                    // from the initial creation of the object
       my_foo-&gt;unref ();            // we must call unref() again,
                                    // since we own the ref the strict_ptr
                                    // added
      </programlisting>
      release() is equivalent to this:
      <programlisting>
      strict_ptr&lt;Foo&gt; f (new Foo);
      Foo * my_foo = f.get ();
      my_foo-&gt;ref ();               // grab a reference
      f = 0;                        // set f to zero, causing it to drop its reference
      </programlisting>
      </para>
      
      $*/
    T* release()
    {
      T *tmp = base::_get();
      if (tmp)
        tmp->ref();
      base::_set(0);
      return tmp;
    }

    /*$ doc method
      $short assign a new value to the smart pointer
      $long
      <para>
      Because reset()'s only argument defaults to 0, reset() is most
      commonly used to set the pointer to 0:
      <programlisting>
      strict_ptr&lt;Foo&gt; f (new Foo);
      f-&gt;unref ();         // remove starting ref of Foo, dropping
                           // refcount of the Foo instance to 1
      f.reset ();          // set f to 0, dropping strict_ptr ref of Foo,
                           // and deleting the Foo instance
      f.reset (new Foo);   // set f to new value
      </programlisting>
      There is no real reason to use reset() instead of simple assignment:
      <programlisting>
      strict_ptr&lt;Foo&gt; f (new Foo);
      f-&gt;unref ();    // drop the initial ref of Foo that we own
      f = 0;          // set f to 0, dropping ref owned by strict_ptr
                      // and deleting the Foo instance
      f = new Foo;    // set f to new value
      </programlisting>
      </para>
      $*/
    void reset(T* obj = 0)
    {
      base::_set(obj);
    }
  };  
  
}; // namespace Inti

#endif // _INTI_BASE_PTR_H_
