#!/usr/bin/env python
# -*- Mode: Python; py-indent-offset: 4 -*-

import message
import string

class VarList:
    """Nicely format a C variable list"""
    def __init__(self):
        self.vars = [] ## list of type-name tuples
    def add(self, ctype, name):
        self.vars.append ((ctype, name))
    def __str__(self):
        ret = []
        for (ctype, name) in self.vars:
            ret.append (ctype + ' ' + name)

        return string.join(ret, ', ')

class ArgType:
    def write_in_param(self, varlist, param_name, canbenull):
        """Write an 'in' parameter for a function declaration to a VarList"""
        raise RuntimeError, "this is an abstract class"
    def write_out_param(self, varlist, param_name, canbenull):
        """Write an 'out' parameter for a function declaration to a VarList"""
        raise RuntimeError, "this is an abstract class"
    def write_inout_param(self, varlist, param_name, canbenull):
        """Write an 'inout' parameter for a function declaration to a VarList"""
        raise RuntimeError, "this is an abstract class"
    def get_arg(self, param_name, canbenull):
        """Obtain a value to pass to the wrapped GTK function (might return param_name.c_str() for example)"""
        return param_name ## by default, just return the parameter name
    def get_return(self, canbenull):
        """get a string for the return type for a function declaration"""
        raise RuntimeError, "this is an abstract class"
    def get_pre_call_setup (self, canbenull):
        """return code to go before the wrapped function"""
        pass
    def get_post_call_shutdown (self, canbenull):
        """return code to go after the wrapped function"""
        pass
    def get_function_return_marshal (self, canbenull):
        """return code to go on the same line as the function call (typically just 'return', but maybe 'foo = ' is also useful)"""
        return 'return'

class NoneArg (ArgType):
    def get_return (self, canbenull):
        return 'void'

    def get_function_return_marshal (self, canbenull):
        return ''

class BasicArg (ArgType):
    def __init__ (self, cpp_in, cpp_out, cpp_inout, cpp_return):
        self.cpp_in = cpp_in
        self.cpp_out = cpp_out
        self.cpp_inout = cpp_inout
        self.cpp_return = cpp_return
        
    def write_in_param(self, varlist, param_name, canbenull):
        if canbenull:
            message.warn ("don't know how to allow " + self.cpp_in + " to be NULL")
        varlist.add (self.cpp_in, param_name)

    def write_out_param(self, varlist, param_name, canbenull):
        if canbenull:
            message.warn ("don't know how to allow " + self.cpp_out + " to be NULL")
        varlist.add (self.cpp_out, param_name)

    def write_inout_param(self, varlist, param_name, canbenull):
        if canbenull:
            message.warn ("don't know how to allow " + self.cpp_inout + " to be NULL")
        varlist.add (self.cpp_inout, param_name)

    def get_return(self, canbenull):
        if canbenull:
            message.warn ("don't know how to allow " + self.cpp_return + " to be NULL")
        return self.cpp_return

class EnumArg (BasicArg):
    def __init__(self, cpp_in, cpp_out, cpp_inout, cpp_return, c_name):
        BasicArg.__init__(self, cpp_in, cpp_out, cpp_inout, cpp_return)
        self.c_name = c_name

    def get_arg (self, param_name, canbenull):
        return '(' + self.c_name + ') ' + param_name 

class StructArg (BasicArg):
    def __init__(self, cpp_in, cpp_out, cpp_inout, cpp_return, get_func):
        BasicArg.__init__(self, cpp_in, cpp_out, cpp_inout, cpp_return)
        self.get_func = get_func

    def get_arg (self, param_name, canbenull):
        return param_name + '.' + self.get_func + ' ()'

class StringArg (BasicArg):
    def __init__(self):
        BasicArg.__init__(self, 'const ustring&', 'ustring*', 'ustring*', 'ustring')

    def get_arg (self, param_name, canbenull):
        return param_name + '.c_str ()'

class ObjectArg (BasicArg):
    def __init__(self, object_cpp_name, object_get_method):
        self.get_method = object_get_method
        BasicArg.__init__(self,
                          object_cpp_name + '*',
                          None,
                          None,
                          object_cpp_name + '*')
                          
    def get_arg (self, param_name, canbenull):
        return param_name + '->' + self.get_method + ' ()'

class ArgMatcher:
    def __init__(self):
        self.argtypes = {}

    def register(self, ptype, handler):
        self.argtypes[ptype] = handler

    def register_enum (self, e):
        quals = ''
        if e.in_module != 'G':
            quals = e.in_module + "::"
        handler = None
        if e.deftype == 'enum':
            handler = EnumArg(quals + e.name,
                              quals + e.name + ' *',
                              quals + e.name + ' *',
                              quals + e.name,
                              e.c_name)
            self.argtypes[e.c_name] = handler
        else:
            ## for now, we drop flags on the floor
            pass


    def get(self, ptype):
        return self.argtypes.get(ptype)

matcher = ArgMatcher ()

matcher.register ('none', NoneArg())

## char* and gchar* are ambiguous, so we don't handle them.
arg = StringArg ()
matcher.register('const-char*', arg)
matcher.register('const-gchar*', arg)
matcher.register('string', arg)
matcher.register('static_string', arg)

arg = BasicArg('const unsigned char *', 'unsigned char *', 'unsigned char *', 'unsigned char *')
matcher.register('unsigned-char*', arg)
matcher.register('guchar*', arg)

arg = BasicArg('char', 'char *', 'char *', 'char')
matcher.register('char', arg)
matcher.register('gchar', arg)
matcher.register('guchar', arg)

arg = BasicArg('int', 'int *', 'int *', 'int')
matcher.register('int', arg)
matcher.register('gint', arg)
arg = BasicArg('unsigned int', 'unsigned int *', 'unsigned int *', 'unsigned int')
matcher.register('guint', arg)
arg = BasicArg('short', 'short *', 'short *', 'short')
matcher.register('short', arg)
matcher.register('gshort', arg)
arg = BasicArg('unsigned short', 'unsigned short *', 'unsigned short *', 'unsigned short')
matcher.register('gushort', arg)
arg = BasicArg('long', 'long *', 'long *', 'long')
matcher.register('long', arg)
matcher.register('glong', arg)
arg = BasicArg('unsigned long', 'unsigned long *',  'unsigned long *', 'unsigned long')
matcher.register('gulong', arg)
## throw exception for in and inout for bool, they require a proxy value
arg = BasicArg('bool', None, None, 'bool')
matcher.register('gboolean', arg)

arg = BasicArg ('guint8', 'guint8 *', 'guint8 *', 'guint8')
matcher.register('guint8', arg)
arg = BasicArg ('gint8', 'gint8 *', 'gint8 *', 'gint8')
matcher.register('gint8', arg)

arg = BasicArg ('guint16', 'guint16 *', 'guint16 *', 'guint16')
matcher.register('guint16', arg)
arg = BasicArg ('gint16', 'gint16 *', 'gint16 *', 'gint16')
matcher.register('gint16', arg)

arg = BasicArg ('guint32', 'guint32 *', 'guint32 *', 'guint32')
matcher.register('guint32', arg)
arg = BasicArg ('gint32', 'gint32 *', 'gint32 *', 'gint32')
matcher.register('gint32', arg)

arg = BasicArg ('double', 'double *', 'double *', 'double')
matcher.register('double', arg)
matcher.register('gdouble', arg)
arg = BasicArg ('float', 'float *', 'float *', 'float')
matcher.register('float', arg)
matcher.register('gfloat', arg)

arg = BasicArg ('gunichar', 'gunichar *', 'gunichar *', 'gunichar')
matcher.register('gunichar', arg)

arg = ObjectArg ('Inti::Gtk::Widget', 'gtk_widget')
matcher.register ('GtkWidget*', arg)

arg = ObjectArg ('Inti::Gtk::Adjustment', 'gtk_adjustment')
matcher.register ('GtkAdjustment*', arg)

arg = BasicArg ('Inti::Gdk::Atom', 'Inti::Gdk::Atom *', 'Inti::Gdk::Atom *', 'Inti::Gdk::Atom')
matcher.register ('GdkAtom', arg)


arg = StructArg ('const Inti::Gdk::Rectangle&', None, None, None, 'gdk_rectangle')
matcher.register ('GdkRectangle*', arg)

arg = StructArg ('const Inti::Gtk::Allocation&', None, None, None, 'gtk_allocation')
matcher.register ('GtkAllocation*', arg)

arg = StructArg ('const Inti::Gtk::Requisition&', None, None, None, 'gtk_requisition')
matcher.register ('GtkRequisition*', arg)

## This is bogus, we should instead use the in/inout flags in the defs file,
## if they existed
arg = StructArg ('const Inti::Gtk::TextIter&', None, None, None, 'gtk_text_iter')
matcher.register ('GtkTextIter*', arg)
arg = StructArg ('Inti::Gtk::TextIter*', None, None, None, 'gtk_text_iter')
matcher.register ('const-GtkTextIter*', arg)

# enums, flags, objects

#  matcher.register_boxed('GtkAccelGroup', 'PyGtkAccelGroup_Type',
#                         'PyGtkAccelGroup_Get', 'PyGtkAccelGroup_New')
#  matcher.register_boxed('GtkStyle', 'PyGtkStyle_Type',
#                         'PyGtkStyle_Get', 'PyGtkStyle_New')
#  matcher.register_boxed('GdkFont', 'PyGdkFont_Type',
#                         'PyGdkFont_Get', 'PyGdkFont_New')
#  matcher.register_boxed('GdkColor', 'PyGdkColor_Type',
#                         'PyGdkColor_Get', 'PyGdkColor_New')
#  matcher.register_boxed('GdkEvent', 'PyGdkEvent_Type',
#                         'PyGdkEvent_Get', 'PyGdkEvent_New')
#  matcher.register_boxed('GdkWindow', 'PyGdkWindow_Type',
#                         'PyGdkWindow_Get', 'PyGdkWindow_New')
#  matcher.register('GdkPixmap*', matcher.get('GdkWindow*'))
#  matcher.register('GdkBitmap*', matcher.get('GdkWindow*'))
#  matcher.register('GdkDrawable*', matcher.get('GdkWindow*'))
#  matcher.register_boxed('GdkGC', 'PyGdkGC_Type',
#                         'PyGdkGC_Get', 'PyGdkGC_New')
#  matcher.register_boxed('GdkColormap', 'PyGdkColormap_Type',
#                         'PyGdkColormap_Get', 'PyGdkColormap_New')
#  matcher.register_boxed('GdkDragContext', 'PyGdkDragContext_Type',
#                         'PyGdkDragContext_Get', 'PyGdkDragContext_New')
#  matcher.register_boxed('GtkSelectionData', 'PyGtkSelectionData_Type',
#                         'PyGtkSelectionData_Get', 'PyGtkSelectionData_New')
#  matcher.register_boxed('GdkCursor', 'PyGdkCursor_Type',
#                         'PyGdkCursor_Get', 'PyGdkCursor_New')
#  matcher.register_boxed('GtkCTreeNode', 'PyGtkCTreeNode_Type',
#                         'PyGtkCTreeNode_Get', 'PyGtkCTreeNode_New')
#  matcher.register('GdkAtom', AtomArg())
