/* Tests out the IDE controller */

#include "../lib/oea_pal.h"

typedef struct _controller {
  volatile char *command;
  volatile char *control;
  volatile char *dma;
  volatile unsigned *interrupt;
} controller;

enum {
  data_reg,
  error_reg,
  sector_count_reg,
  block_0_7_reg,
  block_8_15_reg,
  block_16_23_reg,
  block_24_27_reg,
  status_reg,
};

enum {
  busy = 0x80,
  ready_for_command = 0x40,
  ready_for_data = 0x08,
};

static int
wait_for_status(controller *ide,
		unsigned char mask,
		unsigned char val)
{
  int i;
  for (i = 0; i < 1000; i++) {
    if ((ide->control[0] & mask) == val) {
      return i;
    }
  }
  write_line();
  write_string("Timeout waiting for ((status = 0x");
  write_hex(ide->control[0]);
  write_string(") & 0x");
  write_hex(mask);
  write_string(" ) == 0x");
  write_hex(val);
  write_string(" after ");
  write_int(i);
  write_string(" iterations");
  write_line();
  exit(1);
  return 0;
}

static void
clear_interrupt(controller *ide)
{
  if (ide->interrupt[0] == 0) {
    write_string("interrupt at 0x");
    write_hex(ide->interrupt);
    write_string(" not set");
    write_line();
    exit (1);
  }
  write_string("interrupt status = 0x");
  write_hex(ide->command[status_reg]);
  write_line();
  if (ide->interrupt[0] != 0) {
    write_string("interrupt not cleared");
    write_line();
    exit (1);
  }
}


int
main(int argc, char **argv)
{
  controller ides[2];
  controller *ide = ides;
  int block;
  int byte;
  int delay;
  int argi;
  
  /* check the arguments */
  if (argc != 9) {
    write_string("Usage: 2*(interrupt command control dma)");
    write_line();
    exit (1);
  }

  /* get the list of addresses */
  argi = 1;
  ides[0].interrupt = (void*)strtoul(argv[argi++], 0, 0);
  ides[0].command = (void*)strtoul(argv[argi++], 0, 0);
  ides[0].control = (void*)strtoul(argv[argi++], 0, 0);
  ides[0].dma = (void*)strtoul(argv[argi++], 0, 0);
  ides[1].interrupt = (void*)strtoul(argv[argi++], 0, 0);
  ides[1].command = (void*)strtoul(argv[argi++], 0, 0);
  ides[1].control = (void*)strtoul(argv[argi++], 0, 0);
  ides[1].dma = (void*)strtoul(argv[argi++], 0, 0);

  /* turn off interrupts */
  ide->control[0] = 0x02;

  /* ready? */
  delay = wait_for_status(ide, busy | ready_for_command, ready_for_command);
  write_string("controller ready for write in ");
  write_int(delay);
  write_line();

  /* do a write */
  ide->command[block_24_27_reg] = 0x40;/*LBA*/
  ide->command[sector_count_reg] = 2;
  ide->command[status_reg] = 0x30;
  for (block = 0; block < 2; block++) {
    write_string("Writing ");
    for (byte = 0; byte < 512; byte++) {
      delay = wait_for_status(ide, busy | ready_for_data, ready_for_data); /* ready for data */
      ide->command[data_reg] = ((byte & 0xff) ^ block);
      write_string(".");
    }
    write_line();
  }
  write_string("data written");
  write_line();

  /* finished?*/
  delay = wait_for_status(ide, busy | ready_for_command, ready_for_command);
  write_string("write finished in ");
  write_int(delay);
  write_line();


  /* turn on interrupts */
  ide->control[0] = 0x00;

  /* ready? */
  delay = wait_for_status(ide, busy | ready_for_command, ready_for_command);
  write_string("controller ready for read in ");
  write_int(delay);
  write_line();

  /* do a read */
  ide->command[block_24_27_reg] = 0x40;/*LBA*/
  ide->command[sector_count_reg] = 2;
  ide->command[status_reg] = 0x20; /*read*/
  delay = wait_for_status(ide, busy, 0);
  clear_interrupt(ide);
  write_string("data in fifo in ");
  write_int(delay);
  write_line();

  for (block = 0; block < 2; block++) {
    for (byte = 0; byte < 512; byte++) {
      char data;
      delay = wait_for_status(ide, busy | ready_for_data, ready_for_data);
      data = ide->command[data_reg];
      if (data != ((byte & 0xff) ^ block)) {
	write_string("data missmatch at ");
	write_int(byte);
	write_string(", ");
	write_int(data);
	write_line();
	exit(1);
      }
    }
  }
  write_string("data read");
  write_line();

  /* finished?*/
  delay = wait_for_status(ide, busy | ready_for_command, ready_for_command);
  write_string("transfer finished in ");
  write_int(delay);
  write_line();
}
