// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// BCond<Field<Geom, T, EngineTag>, ZeroFaceBC> template definitions.
//-----------------------------------------------------------------------------

// Include files.

#include "Geometry/CenteringTags.h"
#include "Meshes/MeshTraits.h"
#include "Utilities/PAssert.h"

//-----------------------------------------------------------------------------
// BCond<Field<Geom, T, EngineTag>, ZeroFaceBC>()
// 
// Constructor.
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
BCond<Field<Geom, T, EngineTag>, ZeroFaceBC>::
BCond(const Field<Geom, T, EngineTag> &f, const ZeroFaceBC &bc)
  : Base_t(f, f.totalDomain()), bc_m(bc)
{
  // This BC only makes sense for geometries on logically rectilinear meshes.

  CTAssert(MeshTraits<typename Geom::Mesh_t>::isLogicallyRectilinear);

  int adjust;
  typedef typename Geom::Centering_t Centering_t;
      
  // Get the direction.
  
  int d = bc.face() / 2;
  
  // Select the high or low face.
  
  if (bc.face() & 1) 
    {
      // High face.
      
      // Get the number of guard layers in the upper direction.
      
      int nGuards = subject().geometry().guardLayers().upper(d);
      
      // The other directions span the subject's total domain. Therefore, we
      // just chop out the guard layers, taking care to handle the case
      // where we are enforcing a zero boundary (appropriate only for
      // vert-centering).
      
      if (bc.enforceZeroBoundary() && 
        subject().geometry().centering().centeringTag(d) == vert)
        adjust = 0;
      else
        adjust = 1;
         
      destDomain()[d] = Interval<1>(destDomain()[d].max() - nGuards + adjust, 
			            destDomain()[d].max());
    }
  else 
    {
      // Low face.
        
      // Get the number of guard layers in the lower direction.
      
      int nGuards = subject().geometry().guardLayers().lower(d);
	
      // The other directions span the subject's total domain. Therefore, we
      // just chop out the guard layers, taking care to handle the case
      // where we are enforcing a zero boundary (appropriate only for
      // vert-centering).
      
      if (bc.enforceZeroBoundary() && 
        subject().geometry().centering().centeringTag(d) == vert)
        adjust = 0;
      else
        adjust = 1;

      destDomain()[d] = Interval<1>(destDomain()[d].min(), 
			            destDomain()[d].min() + nGuards - adjust);
    }
}


//-----------------------------------------------------------------------------
// void applyBoundaryCondition()
// 
// The apply method for this boundary condition. (Kind of self-describing.)
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
void BCond<Field<Geom, T, EngineTag>, ZeroFaceBC>::
applyBoundaryCondition()
{
  subject()(destDomain()) = T(0.0);
}


//-----------------------------------------------------------------------------
// This_t *retarget()
// 
// This boundary condition's retarget function.
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
FieldBCondBase< Field<Geom, T, EngineTag> > *
BCond<Field<Geom, T, EngineTag>, ZeroFaceBC>::
retarget(const Field<Geom, T, EngineTag> &f) const
{
  return new This_t(f, bc_m);
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ZeroFaceBC.cpp,v $   $Author: swhaney $
// $Revision: 1.12 $   $Date: 2000/03/07 13:16:13 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
