// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes:
//   ZeroFaceBC
//   BCond<Field<Geom, T, EngineTag>, ZeroFaceBC>
//   AllZeroFace
//-----------------------------------------------------------------------------

#ifndef POOMA_BCONDS_ZEROFACEBC_H
#define POOMA_BCONDS_ZEROFACEBC_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// ZeroFaceBC: BCondCategory class setting all guard layers beyond a
//             specified (logically) rectilinear mesh face to zero.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "BConds/FieldBCondBase.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Full Description:
// 
// ZeroFaceBC is a specialization of BCondCategory.
// 
// It represents a zero Dirichlet boundary condition on a logically rectilinear
// domain.  A constructor switch allows the BC to enforce that the
// mesh-boundary value is zero; this affects only vertex-centered Field
// values/components because the boundary is defined to be the last vertex.
//-----------------------------------------------------------------------------

class ZeroFaceBC : public BCondCategory<ZeroFaceBC>
{
public:

  //---------------------------------------------------------------------------
  // Constructors. 
  
  ZeroFaceBC(int face, bool enforceZeroBoundary = false) : 
    face_m(face), enforceZeroBoundary_m(enforceZeroBoundary) { }
  ZeroFaceBC(const ZeroFaceBC &model) : 
    face_m(model.face()), enforceZeroBoundary_m(model.enforceZeroBoundary())
  { }

  //---------------------------------------------------------------------------
  // Assignment operator. Does deep assignment.
  
  ZeroFaceBC &operator=(const ZeroFaceBC &rhs)
  {
    face_m = rhs.face();

    enforceZeroBoundary_m = rhs.enforceZeroBoundary();

    return *this;
  }

  //---------------------------------------------------------------------------
  // Accessors for data members.
  
  int face() const { return face_m; }
  bool enforceZeroBoundary() const { return enforceZeroBoundary_m; }

private:

  int face_m;
  bool enforceZeroBoundary_m;
};


// ----------------------------------------------------------------------------
// An actual boundary condition class for ZeroFaceBC (partial specialization of
// the BCond class)
// ----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
class BCond<Field<Geom, T, EngineTag>, ZeroFaceBC> :
  public FieldBCondBase<Field<Geom, T, EngineTag> >
{
public:

  typedef BCond<Field<Geom, T, EngineTag>, ZeroFaceBC> This_t;
  typedef FieldBCondBase<Field<Geom, T, EngineTag> > Base_t;
  
  //---------------------------------------------------------------------------
  // Constructors.

  BCond(const Field<Geom, T, EngineTag> &f, const ZeroFaceBC &bc);

  //---------------------------------------------------------------------------
  // Destructor.

  ~BCond() { }

  //---------------------------------------------------------------------------
  // Methods.

  // Apply the boundary conditions.
  
  void applyBoundaryCondition();
  
  // Clone and retarget this boundary condition.
  
  Base_t *retarget(const Field<Geom, T, EngineTag> &f) const;

private:

  // Store a copy of the BCondCategory object used to construct this.

  ZeroFaceBC bc_m;
};

//-----------------------------------------------------------------------------
// Full Description:
// 
// AllZeroFaceBC is a functor class that sets all of a Field's BCs to be 
// ZeroFaceBC.
//-----------------------------------------------------------------------------

class AllZeroFaceBC
{
public:

  AllZeroFaceBC(bool enforceZeroBoundary = false)
  : ezb_m(enforceZeroBoundary) { }
  
  template<class Geom, class T, class EngineTag>
  void operator()(Field<Geom, T, EngineTag> &f) const
    {
      for (int i = 0; i < 2 * Geom::dimensions; i++)
        f.addBoundaryCondition(ZeroFaceBC(i, ezb_m));
    }

private:

  bool ezb_m;
};

#include "BConds/ZeroFaceBC.cpp"

#endif // POOMA_BCONDS_ZEROFACEBC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ZeroFaceBC.h,v $   $Author: swhaney $
// $Revision: 1.12 $   $Date: 2000/03/07 13:16:13 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
