// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// WhereProxy<F,B>
//-----------------------------------------------------------------------------

#ifndef POOMA_EVALUATOR_WHEREPROXY_H
#define POOMA_EVALUATOR_WHEREPROXY_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// WhereProxy is used to implement 2 argument where().
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Evaluator/OpMask.h"
#include "Pooma/PETE/ExpressionTraits.h"

//-----------------------------------------------------------------------------
// We need the tools to convert a WhereProxy into an Array or Field or
// whatever, so users need to specialize
//-----------------------------------------------------------------------------

template<class ETrait, class Tree>
struct ConvertWhereProxy
{ };

//-----------------------------------------------------------------------------
//
// WhereProxy<Expr>
//
// The only legal use of where(a,b) is in an expression like:
//
// A = where(f,B);
//
// Rather than have where(f,B) return an array that could be combined in
// an expression, we return a WhereProxy that is recognized by assingment
// operators.
//
// The WhereProxy is also necessary because the elements returned by
// where are MaskAssign<T> objects, so a special assignment operator,
// OpMask<Op> must be used.
//
//-----------------------------------------------------------------------------

template<class F, class B>
struct WhereProxy
{
  WhereProxy(const F& f, const B& b) : f_m(f), b_m(b) { }

  typedef BinaryNode<WhereMask,
    typename CreateLeaf<F>::Leaf_t,
    typename CreateLeaf<B>::Leaf_t> Tree_t;

  typedef typename ExpressionTraits<Tree_t>::Type_t           ETrait_t;
  typedef typename ConvertWhereProxy<ETrait_t,Tree_t>::Make_t MakeFromTree_t;
  typedef typename MakeFromTree_t::Expression_t               WhereMask_t;

  inline WhereMask_t
  whereMask() const
  {
    return MakeFromTree_t::make(Tree_t(CreateLeaf<F>::make(f_m),
				       CreateLeaf<B>::make(b_m)));
  }

  template<class Op>
  inline OpMask<Op>
  opMask(const Op &op) const
  {
    return OpMask<Op>(op);
  }

  inline const F &flag() { return f_m; }
  inline const B &value() { return b_m; }
  const F &f_m;
  const B &b_m;
};

//-----------------------------------------------------------------------------
//
//-----------------------------------------------------------------------------

template<class F, class B>
inline WhereProxy<F,B>
where(const F &f, const B &b)
{
  return WhereProxy<F,B>(f,b);
}

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_EVALUATOR_WHEREPROXY_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: WhereProxy.h,v $   $Author: swhaney $
// $Revision: 1.3 $   $Date: 2000/07/20 15:39:26 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
