// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes:
//   Average
//   WeightedAverage
// Global Function Templates:
//   average
//-----------------------------------------------------------------------------

#ifndef POOMA_FIELD_DIFFOPS_AVERAGE_H
#define POOMA_FIELD_DIFFOPS_AVERAGE_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// Classes:
//
// Average :  Averaging operator (functor) on discrete Fields. Used for 
//            (optionally) weighted averages of an input Field to produce an
//            output Field.
//
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// 
// Global Function Templates:
// 
// average()  :	 Wrapper function around FieldStencil<Average>::operator() . 
//               The Average functors actually used are partial specializations
//               of the generic Average that come from Average.URM.h or
//               Average.LRM.h, for example.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Field/Field.h"
#include "Field/FieldStencil.h"
#include "Field/WeightedFieldStencil.h"
#include "Meshes/MeshTraits.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
// 
// Classes:
//
// Average:
// 
// Average is a functor class serving as the "Functor" template parameter for
// FieldStencil<Functor,Expression>, which implements a 
// discrete averaging operator.
// Partial specializations implement various combinations of input and output
// centerings, for specific coordinate systems, are defined in other headers
// like Average.LRM.h.
// 
// WeightedAverage:
// 
// Like Average, except for use as the "Functor" parameter in
// WeightedFieldStencil<Functor,Expression>, which implements a discrete 
// averaging operator weighted by a second input Field.
// 
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// 
// Global Function Templates:
// 
// average() : Averages from a neighborhood of centering points in the input
//             Field, optionally weighted by a weighting Field on the same
//             Geometry, to an output centering point in the output
//             Field. Takes as input a ConstField on a discrete geometry with
//             one centering and returns a ConstField a geometry that's the
//             same except (possibly) for the centering. All the work happens
//             in the embedded Average functor partial specialization, in its
//             operator()() methods.
//
//             The prototype with two Field arguments uses the second one as
//             the weights applied in the averaging. It uses a WeightedAverage
//             functor.
//-----------------------------------------------------------------------------


// ----------------------------------------------------------------------------
// General Average template
// ----------------------------------------------------------------------------

template<class OutputCentering, class Geometry, class T, 
  bool LR = MeshTraits<typename Geometry::Mesh_t>::isLogicallyRectilinear>
class Average;


// ----------------------------------------------------------------------------
// General weighted Average template
// ----------------------------------------------------------------------------

template<class OutputCentering, class Geometry, class T, class TW,
  bool LR = MeshTraits<typename Geometry::Mesh_t>::isLogicallyRectilinear>
class WeightedAverage;


// ----------------------------------------------------------------------------
// 
// Global Function Templates:
//
// ----------------------------------------------------------------------------

// Unweighted average:

template<class OutputCentering, class Geometry, class T, class EngineTag>
typename 
  View1<FieldStencil<Average<OutputCentering, Geometry, T, 
    MeshTraits<typename Geometry::Mesh_t>::isLogicallyRectilinear> >, 
    ConstField<Geometry, T, EngineTag> >::Type_t 
average(const ConstField<Geometry, T, EngineTag> &f)
{
  typedef FieldStencil<Average<OutputCentering, Geometry, T, 
    MeshTraits<typename Geometry::Mesh_t>::isLogicallyRectilinear> >
    Functor_t;
  typedef ConstField<Geometry, T, EngineTag> Expression_t;
  typedef View1<Functor_t, Expression_t> Ret_t;
  return Ret_t::make(Functor_t(), f);
}

// Weighted average:

template<class OutputCentering, class Geometry, class T, class EngineTag,
  class TW, class EngineTagW>
typename 
  View2<WeightedFieldStencil<WeightedAverage<OutputCentering, Geometry, T, 
    TW, MeshTraits<typename Geometry::Mesh_t>::isLogicallyRectilinear> >, 
    ConstField<Geometry, T, EngineTag>,
    ConstField<Geometry, TW, EngineTagW> >::Type_t 
average(const ConstField<Geometry, T, EngineTag> &f, 
	const ConstField<Geometry, TW, EngineTagW> &weight)
{
  typedef WeightedFieldStencil<WeightedAverage<OutputCentering, Geometry, T, 
    TW, MeshTraits<typename Geometry::Mesh_t>::isLogicallyRectilinear> >
    Functor_t;
  typedef ConstField<Geometry, T, EngineTag> Expression1_t;
  typedef ConstField<Geometry, TW, EngineTagW> Expression2_t;
  typedef View2<Functor_t, Expression1_t, Expression2_t> Ret_t;
  return Ret_t::make(Functor_t(), f, weight);
}

#endif     // POOMA_FIELD_DIFFOPS_AVERAGE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Average.h,v $   $Author: swhaney $
// $Revision: 1.10 $   $Date: 2000/03/07 13:17:10 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
