#include "Pooma/Pooma.h"
#include "Array/Array.h"
#include "Geometry/DiscreteGeometry.RM.h"
#include "CoordinateSystems/Cylindrical.h"
#include "Utilities/Tester.h"

#include <iostream>

#define PI_M  3.141592653589793238462643383279502

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  
  const int D = 2; // Dimensionality

  // Create the mesh
  Interval<1> I(4);
  Interval<2> vertDomain;
  for (int d = 0; d < D; d++) { vertDomain[d] = I; }
  typedef RectilinearMesh<2> Mesh_t;
  Mesh_t::PointType_t origin(0.0);
  Vector< 2, Array<1,Mesh_t::AxisType_t> > spacings;
  for (int d = 0; d < D; d++) {
    spacings(d).initialize(vertDomain[d].size()-1);
    for (int i = 0; i < vertDomain[d].size()-1; i++) {
      spacings(d)(i) = origin(d) + (d + 1)*(i + 1);
    }
  }
  Mesh_t mesh(vertDomain, origin, spacings);

  // Output the cell volumes:
  std::cout << "======= mesh.cellVolumes() =======" << std::endl
	    << mesh.cellVolumes() << std::endl;

  // Make a Cell-centered Geometry, and output the position values:
  DiscreteGeometry<Cell, Mesh_t> geomCell(mesh);
  std::cout << "======= geomCell.x() =======" << std::endl
	    << geomCell.x() << std::endl;
  
  // Make a Vert-centered Geometry, and output the position values:
  DiscreteGeometry<Vert, Mesh_t> geomVert(mesh);
  std::cout << "======= geomVert.x() =======" << std::endl
	    << geomVert.x() << std::endl;
  
  // 2D vector-face centering:
  typedef RectilinearCentering<2,VectorFaceRCTag<2> > 
    Centering_t;
  DiscreteGeometry<Centering_t, Mesh_t> geomVF(mesh);
  
  int NComponents = 2;
  for (int c = 0; c < NComponents; c++) {
    std::cout << "======= geomVF.x().comp(" << c << ") =======" << std::endl;
    std::cout << geomVF.x().comp(c) << std::endl;
  }

  // For centering something like 2D Tensors on y-edges:
  typedef RectilinearCentering<2,EdgeRCTag<1> > 
    Centering2_t;
  DiscreteGeometry<Centering2_t, Mesh_t> geomE(mesh);
  
  std::cout << "======= geomE<1>.x() =======" << std::endl
	    << geomE.x() << std::endl;


  // ///////////////////////////////////////////////////////////////////////////
  // Curvilinear coordinates
  // ///////////////////////////////////////////////////////////////////////////

  // Create a Polar mesh
  Interval<2> vertPolarDomain;
  for (int d = 0; d < 2; d++) { vertPolarDomain[d] = I; }
  typedef RectilinearMesh<2,Cylindrical::Polar_t> PolarMesh_t;
  PolarMesh_t::PointType_t originP(0.0);
  Vector< 2, Array<1,PolarMesh_t::AxisType_t> > spacingsP;
  spacingsP(0).initialize(vertDomain[0].size()-1);
  spacingsP(1).initialize(vertDomain[1].size()-1);
  for (int i = 0; i < vertDomain[0].size()-1; i++)
    // Varying radial spacing:
    spacingsP(0)(i) = originP(0) + (i + 1);
  for (int i = 0; i < vertDomain[1].size()-1; i++)
    // Constant angular spacing from 0 to pi:
    spacingsP(1)(i) = PI_M/vertDomain[1].size()-1;
  PolarMesh_t polarMesh(vertPolarDomain, originP, spacingsP);

  // Output the cell volumes:
  std::cout << "======= polarMesh.cellVolumes() =======" << std::endl
	    << polarMesh.cellVolumes() << std::endl;

  // Make a Vert-centered Geometry, and output the position values:
  DiscreteGeometry<Vert, PolarMesh_t> geomPolarVert(polarMesh);
  std::cout << "======= Polar geomPolarVert.x() =======" << std::endl
	    << geomPolarVert.x() << std::endl;
  
  
  Pooma::finalize();

  return 0;
}
