// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// UniformRectilinearMeshData<Dim, CoordinateSystems, T> template definitions.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include files
//-----------------------------------------------------------------------------

#include "Meshes/UniformRectilinearMesh.h"


///////////////////////////////////////////////////////////////////////////////
//
// UniformRectilinearMeshData<Dim, CoordinateSystem, T> members.
//
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Performs initialization of the compute-base arrays.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
void 
UniformRectilinearMeshData<Dim, CoordinateSystem, T>::
initializeFunctionArrays()
{
  cellVolumes_m.engine().setFunctor(VolumeFunctor(*this));
  vertexPositions_m.engine().setFunctor(PositionFunctor(*this));
  vertexDeltas_m.engine().setConstant(meshSpacing());
  
  Vector<2*Dim, PointType_t> normals;
  normals = 0.0;
  for (int d = 0; d < Dim; d++)
    {
      normals(2 * d)(d) = -1.0;
      normals(2 * d + 1)(d) = 1.0;
      invspacings_m(d) = (spacings_m(d) != 0.0 ? 1.0 / spacings_m(d) : 0.0);
    }
  cellSurfaceNormals_m.engine().setConstant(normals);
}

//-----------------------------------------------------------------------------
// General constructor.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
UniformRectilinearMeshData<Dim, CoordinateSystem, T>::
UniformRectilinearMeshData(const Domain_t &pd, const PointType_t &origin,
  const PointType_t &spacings)
: initialized_m(true),
  physicalDomain_m(pd),
  guardLayers_m(guardLayerSizes(), guardLayerSizes()),
  totalDomain_m(guardLayers().addGuardLayersToDomain(physicalDomain())),
  totalCellDomain_m(
    guardLayers().addGuardLayersToDomain(physicalCellDomain())),
  cellVolumes_m(totalCellDomain()),
  vertexDeltas_m(totalCellDomain()),
  vertexPositions_m(totalDomain()),
  cellSurfaceNormals_m(totalCellDomain()),
  origin_m(origin),
  spacings_m(spacings)
{
  initializeFunctionArrays();
}

//-----------------------------------------------------------------------------
// Copy constructor.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
UniformRectilinearMeshData<Dim, CoordinateSystem, T>::
UniformRectilinearMeshData(const This_t &model)
: initialized_m(model.initialized()),
  physicalDomain_m(model.physicalDomain()),
  guardLayers_m(model.guardLayers()),
  totalDomain_m(model.totalDomain()),
  totalCellDomain_m(model.totalCellDomain()),
  cellVolumes_m(model.cellVolumes()),
  vertexDeltas_m(model.vertexDeltas()),
  vertexPositions_m(model.vertexPositions()),
  cellSurfaceNormals_m(model.cellSurfaceNormals()),
  origin_m(model.origin()),
  spacings_m(model.spacings()),
  invspacings_m(model.invspacings())
{ }

//-----------------------------------------------------------------------------
// Copy assignment operator.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
typename UniformRectilinearMeshData<Dim,CoordinateSystem,T>::This_t &
UniformRectilinearMeshData<Dim, CoordinateSystem, T>::
operator=(const This_t &rhs)
{
  if (&rhs != this)
    {
      initialized_m = rhs.initialized();
      physicalDomain_m = rhs.physicalDomain();
      guardLayers_m = rhs.guardLayers();
      totalDomain_m = rhs.totalDomain();
      totalCellDomain_m = rhs.totalCellDomain();
      cellVolumes_m = rhs.cellVolumes();
      vertexDeltas_m = rhs.vertexDeltas();
      vertexPositions_m = rhs.vertexPositions();
      cellSurfaceNormals_m = rhs.cellSurfaceNormals();
      origin_m = rhs.origin();
      spacings_m = rhs.spacings();
      invspacings_m = rhs.invspacings();
    }

  return *this;
}

//-----------------------------------------------------------------------------
// Initialize function. Can be used when default constructor was invoked to
// build the object.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
void 
UniformRectilinearMeshData<Dim, CoordinateSystem, T>::
initialize(const Domain_t &pd, const PointType_t &origin,
  const PointType_t &spacings)
{
  // Set up the data members.
    
  physicalDomain_m = pd;
  guardLayers_m.initialize(guardLayerSizes(), guardLayerSizes());
  totalDomain_m = guardLayers().addGuardLayersToDomain(physicalDomain());
  totalCellDomain_m = 
    guardLayers().addGuardLayersToDomain(physicalCellDomain());
  cellVolumes_m.setDomain(totalCellDomain());
  vertexDeltas_m.setDomain(totalCellDomain());
  vertexPositions_m.setDomain(totalDomain());
  cellSurfaceNormals_m.setDomain(totalCellDomain());
  origin_m = origin;
  spacings_m = spacings;

  initializeFunctionArrays();

  // Note that we are now initialized.

  initialized_m = true;
}      

///////////////////////////////////////////////////////////////////////////////
//
// UniformRectilinearMesh<Dim, CoordinateSystem, T> members.
//
///////////////////////////////////////////////////////////////////////////////

// Print a UniformRectilinearMesh on an output stream

template <int Dim, class CoordinateSystem, class T>
template <class Ostream>
void 
UniformRectilinearMesh<Dim, CoordinateSystem, T>::
print(Ostream &ostr) const 
{
  ostr << "UniformRectilinearMesh" << "\n----------------------\n";
  ostr << "Dimension: " << Dim << '\n';
  ostr << "CoordinateSystem: " << coordinateSystem() << '\n';
  ostr << "Physical domain: " << physicalDomain() << '\n';
  ostr << "Total domain: " << totalDomain() << '\n';
  ostr << "GuardLayers: " << guardLayers() << '\n';
  ostr << "Mesh spacing: " << meshSpacing() << '\n';
  ostr << "Origin: " << origin() << '\n';
  ostr << "Vertex positions: " << '\n' << vertexPositions() << '\n';
  ostr << "Cell volumes: " << '\n' << cellVolumes();
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: UniformRectilinearMesh.cpp,v $   $Author: swhaney $
// $Revision: 1.14 $   $Date: 2000/03/07 13:17:41 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
