/*
 * Author: Heinz Mauelshagen, Germany
 *
 * March, October-November 1997
 * May 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    10/12/1997 - added check for active physical volume
 *    11/07/1997 - added error message for not existing partitions
 *    05/03/1998 - enhanced error checking
 *    05/04/1998 - added multiple device support
 *    05/16/1998 - added lvmtab checking
 *
 */

#include <lvm_user.h>

int do_pv_create ( char *, pv_t *, uint);

char *cmd = NULL;
#ifdef DEBUG
int opt_d = 0;
#endif
int opt_v = 0;


int main ( int argc, char **argv)
{
   int c = 0;
   int c1 = 0;
   int force_needed = 0;
   int opt_f = 0;
   int opt_y = 0;
   int ret = 0;
   int size = 0;
#ifdef DEBUG
   char *options = "dfh?yv";
#else
   char *options = "fh?yv";
#endif
   char *pv_name = NULL;
   pv_t *pv = NULL;
   pv_t pv_new;
   struct partition part;

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif

         case 'f':
            if ( opt_f > 1) {
               fprintf ( stderr, "%s -- f option yet given two times\n\n", cmd);
               return 1;
            }
            opt_f++;
            break;

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Physical volume create\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-f[f]]\n"
                     "\t[-h/-?]\n"
                     "\t[-y]\n"
                     "\t[-v]\n"
                     "\tPhysicalVolumePath [PhysicalVolumePath...]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option yet given\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;

         case 'y':
            if ( opt_y > 1) {
               fprintf ( stderr, "%s -- y option yet given\n\n", cmd);
               return 1;
            }
            opt_y++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return 1;
      }
   }

   if ( optind == argc) {
      fprintf ( stderr, "%s -- please enter a physical volume path\n\n", cmd);
      return 1;
   }

   if ( opt_y > 0 && opt_f == 0) {
      fprintf ( stderr, "%s -- option y can only be given with option f\n",
                         cmd);
      return 1;
   }

   LVM_LOCK ( 0);
   LVM_CHECK_IOP;

   for ( ; optind < argc; optind++) {
      force_needed = 0;
      pv_name = argv[optind];
   
      if ( opt_v > 0) printf ( "%s -- checking physical volume name %s\n",
                               cmd, pv_name);
      if ( pv_check_name ( pv_name) < 0) {
         fprintf ( stderr, "%s -- invalid physical volume path %s\n",
                            cmd, pv_name);
         continue;
      }

      if ( opt_v > 0) printf ( "%s -- getting physical volume size\n", cmd);
      if ( ( size = pv_get_size ( pv_name, &part)) < 0) {
         if ( size == -LVM_EPV_GET_SIZE_PART) {
            fprintf ( stderr, "%s -- partition %s doesn't exist\n",
                              cmd, pv_name);
            continue;
         } else if ( size == -LVM_EPV_GET_SIZE_OPEN) {
            fprintf ( stderr, "%s -- can't open %s to get its size\n",
                              cmd, pv_name);
         } else if ( size == -LVM_EPV_GET_SIZE_NO_EXTENDED) {
            fprintf ( stderr, "%s -- extended partition %s doesn't exist\n",
                              cmd, pv_name);
         } else {
            fprintf ( stderr, "%s -- ERROR %d getting size of %s\n",
                              cmd, size, pv_name);
         }
         continue;
      }

      if ( opt_v > 0) printf ( "%s -- checking partition type\n", cmd);
      if ( part.sys_ind != 0) {
         if ( part.sys_ind == DOS_EXTENDED_PARTITION ||
              part.sys_ind == LINUX_EXTENDED_PARTITION) {
            fprintf ( stderr, "%s -- can't use extended partition %s\n\n",
                      cmd, pv_name);
            continue;
         } else if ( part.sys_ind != LVM_PARTITION) {
            fprintf ( stderr, "%s -- invalid partition type 0x%x for %s"
                              " (must be 0x%x)\n", cmd, part.sys_ind,
                              pv_name, LVM_PARTITION);
            continue;
         }
      }
   
      if ( ( ret = pv_read ( pv_name, &pv, NULL)) < 0 &&
           ret != -LVM_EPV_READ_MD_DEVICE) {
         if ( ret == -LVM_EPV_READ_PV_EXPORTED) {
            pv->vg_name[strlen(pv->vg_name)-strlen(EXPORTED)] = 0;
            fprintf ( stderr, "%s -- physical volume %s belongs to "
                              "exported volume group %s\n",
                               cmd, pv_name, pv->vg_name);
            force_needed++;
         } else if ( ret == -LVM_EPV_READ_ID_INVALID ) {
            fprintf ( stderr, "%s -- %s has an invalid physical "
                              "volume identifier\n", cmd, pv_name);
            force_needed = 0;
         } else {
            fprintf ( stderr, "%s -- ERROR %d: invalid physical volume %s\n",
                               cmd, ret, pv_name);
            force_needed = 0;
         }
      } else {
         if ( opt_v > 0) printf ( "%s -- checking volume group name\n", cmd);
         if ( vg_check_name ( pv->vg_name) == 0 &&
              pv->vg_name[0] != 0) {
            if ( opt_f < 2) {
               printf ( "%s -- can't initialize %s of volume group %s "
                        "without -ff\n", cmd, pv_name, pv->vg_name);
               continue;
            } else {
               if ( opt_y == 0) {
                  c = 0;
                  while ( c != 'y' && c != 'n') {
                     if ( c == '\n' || c == 0)
                        printf ( "%s -- really INITIALIZE physical "
                                 "volume %s of %s [y/n]? ",
                                 cmd, pv_name, pv->vg_name);
                     c = tolower ( getchar ());
                  }
                  c1 = c;
                  while ( c != '\n') c = tolower ( getchar ());
                  if ( c1 == 'y') {
                     force_needed = 0;
                  } else {
                     printf ( "%s -- physical volume %s not initialized\n",
                              cmd, pv_name);
                     continue;
                  }
               } else force_needed = 0;
            }
         }
      }
   
      if ( opt_f == 0 && force_needed > 0) {
         printf ( "%s -- need -f to initialize physical volume %s\n",
                   cmd, pv_name);
         continue;
      }

      if ( pv_check_active ( pv->vg_name, pv_name) == TRUE) {
         fprintf ( stderr, "%s -- can't force create on active physical "
                           "volume %s\n", cmd, pv_name);
         continue;
      }

      if ( opt_f > 0) {
         printf ( "%s -- forcing physical volume creation on %s", cmd, pv_name);
         if ( pv_check_new == FALSE) printf ( " of volume group %s",
                                              pv->vg_name);
         printf ( "\n");
      }
   
      if ( opt_v > 0) printf ( "%s -- checking for new physical volume\n", cmd);
      if ( pv_check_new ( pv) == TRUE)
         printf ( "%s -- reinitializing physical volume\n", cmd);
   
      if ( lvm_tab_vg_check_exist ( pv->vg_name, NULL) == TRUE) {
         printf ( "%s -- removing lvmtab entry\n", cmd);
         if ( ( ret = lvm_tab_vg_remove ( pv->vg_name)) < 0) {
            fprintf ( stderr, "%s -- ERROR %d removing %s from %s\n\n",
                              cmd, ret, pv->vg_name, LVMTAB);
            return 1;
         }
      }

      if ( opt_v > 0) printf ( "%s -- creating new physical volume\n", cmd);
      if ( ( ret = do_pv_create ( pv_name, &pv_new, size)) != 0) {
         fprintf ( stderr, "%s -- ERROR %d creating physical volume %s\n",
                   cmd, ret, pv_name);
         continue;
      }
   }

   lvm_interrupt ();
   LVM_UNLOCK ( 0);

   printf ( "\n");
   return 0;
}


int do_pv_create ( char *pv_name, pv_t *pv, uint size) {
   int ret = 0;

   if ( opt_v > 0) printf ( "%s -- creating physical volume on "
                            "%s with %u sectors\n",
                            cmd, pv_name, size);

   if ( opt_v > 0) printf ( "%s -- setting physical volume up\n", cmd);
   if ( ( ret = pv_setup_for_create ( pv_name, pv, size)) != 0) {
      fprintf ( stderr, "%s -- ERROR %d setting up  physical volume on %s\n",
                cmd, ret, pv_name);
      return 1;
   }

   lvm_dont_interrupt ( 0);

   if ( opt_v > 0) printf ( "%s -- storing physical volume data on disk\n",
                            cmd);
   if ( ( ret = pv_write ( pv_name, pv)) == 0) {
      printf ( "%s -- physical volume on %s successfully created\n",
                cmd, pv_name);
   } else {
      fprintf ( stderr, "%s -- ERROR %d creating physical volume on %s\n",
                cmd, ret, pv_name);
      return -2;
   }
   lvm_interrupt ();

   return 0;
}
