/*
 * Author: Heinz Mauelshagen, Germany (mge@ez-darmstadt.telekom.de)
 *
 * May, November 1997
 * May 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    10/08/1997 - changed some messages
 *    11/30/1997 -           "
 *    05/05/1998 - added multiple device support
 *    05/16/1998 - added lvmtab checking
 *    05/17/1998 - obsoleted physical volume name change (it's now in pv_read())
 *    05/18/1998 - fixed display bug for unused physical volumes
 *    05/22/1998 - changed display of physical volumes in case
 *                 of multiple devives
 *
 */

#include <lvm_user.h>

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv)
{
   int c = 0;
   int opt_v = 0;
   int p = 0;
   int ret = 0;
   int vg_name_len = 0;
   char *cmd = NULL;
   char *dummy1 = NULL;
   char *dummy2 = NULL;
#ifdef DEBUG
   char *options = "dh?v";
#else
   char *options = "h?v";
#endif
   char vg_name_this[NAME_LEN] = { 0, };
   pv_t **pv = NULL;
   pv_t *pv_tmp;

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Physical volume scan\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-h/?]\n"
                     "\t[-v[v]]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'v':
            if ( opt_v > 1) {
               fprintf ( stderr, "%s -- v option yet given two times\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return 1;
      }
   }

   LVM_LOCK ( 1);
   LVM_CHECK_IOP;

   printf ( "%s -- reading all physical volumes (this may take a while...)\n",
            cmd);
   if ( ( ret = pv_read_all_pv ( &pv, TRUE)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d: reading physical volumes\n\n",
                cmd, ret);
      return 1;
   }


   if ( opt_v > 0) printf ( "%s -- walking through all physical "
                            "volumes found\n", cmd);
   if ( pv != NULL) for ( p = 0; pv[p] != NULL; p++) {
      if ( ( ret = pv_read ( pv[p]->pv_name, &pv_tmp, NULL)) < 0 &&
           ret == -LVM_EPV_READ_MD_DEVICE) {
         if ( MAJOR ( pv_create_kdev_t ( pv[p]->pv_name)) != MD_MAJOR) {
            printf ( "%s -- WARNING: physical volume %s belongs "
                     "to a multiple device\n",
                     cmd, pv[p]->pv_name);
         }
         if ( MAJOR ( pv[p]->pv_dev) != MD_MAJOR) continue;
      }

      if ( opt_v > 1) {
         printf ( "%s -- getting data for %s from kernel\n",
                  cmd, pv[p]->pv_name);
         if ( pv_status ( pv[p]->vg_name, pv[p]->pv_name, &pv_tmp) < 0)
            printf ( "%s -- physical volume %s is not active\n",
                     cmd, pv[p]->pv_name);
         pv_show ( pv[p]);
         printf ( "System Id             %s\n\n", pv[p]->system_id);
         printf ( "\n");
      } else {
         printf ( "%s -- ", cmd);
         if ( pv_status ( pv[p]->vg_name, pv[p]->pv_name, &pv_tmp) < 0)
            printf ( "inactive ");
         else
            printf ( "ACTIVE   ");
         vg_name_len = strlen ( pv[p]->vg_name) - sizeof ( EXPORTED) + 1;
         if ( pv_check_new ( pv[p]) == TRUE) {
            printf ( "PV %s  is in no VG  [%s]\n",
                     pv[p]->pv_name,
                     ( dummy1 = show_size ( pv[p]->pv_size / 2, SHORT)));
            free ( dummy1); dummy1 = NULL;
         } else if ( strcmp ( &pv[p]->vg_name[vg_name_len],
                              EXPORTED) == 0) {
            strncpy ( vg_name_this, pv[p]->vg_name, vg_name_len);
            printf ( "PV %s  is in EXPORTED VG %s [%s / %s free]\n",
                     pv[p]->pv_name, vg_name_this,
                     ( dummy1 = show_size ( pv[p]->pe_total *
                                            pv[p]->pe_size / 2, SHORT)),
                     ( dummy2 = show_size ( ( pv[p]->pe_total - pv[p]->pe_allocated) * pv[p]->pe_size / 2, SHORT)));
            free ( dummy1); dummy1 = NULL;
            free ( dummy2); dummy2 = NULL;
         } else if ( lvm_tab_vg_check_exist ( pv[p]->vg_name, NULL) != TRUE) {
            printf ( "PV %s  is associated to an unknown VG "
                     "(run vgscan)\n", pv[p]->pv_name);
         } else {
            printf ( "PV %-10s of VG %s [%s / %s free]\n",
                     pv[p]->pv_name, pv[p]->vg_name,
                     ( dummy1 = show_size ( pv[p]->pe_total *
                                            pv[p]->pe_size / 2, SHORT)),
                     ( dummy2 = show_size ( ( pv[p]->pe_total - pv[p]->pe_allocated) *pv[p]->pe_size / 2, SHORT)));
            free ( dummy1); dummy1 = NULL;
            free ( dummy2); dummy2 = NULL;
         }
      }
   }

   LVM_UNLOCK ( 1);

   if ( p == 0) printf ( "%s -- no", cmd);
   else         printf ( "%s -- %d", cmd, p);
   printf ( " valid physical volumes found\n\n");

   if ( p == 0) return 1;
   else         return 0;
}
