/*
 * Author: Heinz Mauelshagen, Germany
 *
 * May 1997
 * May 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    05/02/1998 - don't make LV name change in LV structs to avoid
 *                 usage of lv_create_name() because of free logical volume
 *                 name support
 *    05/03/1998 - enhanced error checking with lvm_tab_vg_remove()
 *    05/16/1998 - added lvmtab checking
 *
 */

#include <lvm_user.h>

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv) {
   int c = 0;
   int opt_v = 0;
   int p = 0;
   int ret = 0;
   char *vg_name = NULL;
   char *cmd = NULL;
#ifdef DEBUG
   char *options = "dh?v";
#else
   char *options = "h?v";
#endif
   char vg_name_exp[NAME_LEN] = { 0, };
   vg_t *vg = NULL;

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Volume group export\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-h/?]\n"
                     "\t[-v]\n"
                     "\tVolumeGroupName [VolumeGroupName...]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option yet given\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n\n",
                      cmd, c);
            return 1;
      }
   }
  
   if ( optind == argc) {
      fprintf ( stderr, "%s -- please give a volume group name\n\n", cmd);
      return 1;
   }

   LVM_LOCK ( 0);
   LVM_CHECK_IOP;

   for ( ; optind < argc; optind++) {
      vg_name = argv[optind];

      if ( opt_v > 0) printf ( "%s -- checking volume group name\n", cmd);
      if ( vg_check_name ( vg_name) < 0) {
         fprintf ( stderr, "%s -- invalid volume group name %s\n",
                   cmd, vg_name);
         continue;
      }
   
      if ( opt_v > 0) printf ( "%s -- checking volume group existence\n", cmd);
      if ( ( ret = vg_check_exist ( vg_name)) != TRUE) {
         fprintf ( stderr, "%s -- volume group %s doesn't exist\n",
                   cmd, vg_name);
         continue;
      }

      sprintf ( vg_name_exp, "%s%s%c", vg_name, EXPORTED, 0);
      if ( ( ret = vg_check_exist ( vg_name_exp)) != TRUE) {
         if ( ret == -LVM_EVG_READ_VG_EXPORTED) {
            fprintf ( stderr, "%s -- a volume group named %s is "
                              "allready exported\n",
                              cmd, vg_name);
            continue;
         }
      }

      if ( opt_v > 0) printf ( "%s -- checking volume group activity\n", cmd);
      if ( vg_check_active ( vg_name) == TRUE) {
         fprintf ( stderr, "%s -- can't export active volume group %s\n",
                   cmd, vg_name);
         continue;
      }

      if ( opt_v > 0) printf ( "%s -- reading volume group data for %s "
                               "from disk(s)\n", cmd, vg_name);
      if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name, &vg)) != 0) {
         fprintf ( stderr, "%s -- ERROR %d reading data of %s\n\n",
                   cmd, ret, vg_name);
         continue;
      }

      if ( opt_v > 0) printf ( "%s -- checking for physical volumes in "
                               "volume group %s\n", cmd, vg_name);
      if ( vg->pv_cur == 0) {
         fprintf ( stderr, "%s -- no physical volumes in volume group %s\n",
                   cmd, vg_name);
         continue;
      }
   
      if ( opt_v > 0) printf ( "%s -- checking volume group consistency "
                               "of %s\n", cmd, vg_name);
      if ( ( ret = vg_check_consistency_with_pv_and_lv ( vg)) != 0) {
         fprintf ( stderr, "%s -- ERROR %d: %s is inconsistent\n\n",
                   cmd, ret, vg_name);
         continue;
      }

      /* set my export marker */
      if ( opt_v > 0) printf ( "%s -- setting up volume group %s for export\n",
                               cmd, vg_name);
      system_id_set_exported ( vg->pv[0]->system_id);
      strcat ( vg->pv[0]->vg_name, EXPORTED);

      /* correct it in the rest of the PVs */
      for ( p = 1; vg->pv[p] != NULL; p++) {
         strcpy ( vg->pv[p]->system_id, vg->pv[0]->system_id);
         strcpy ( vg->pv[p]->vg_name, vg->pv[0]->vg_name);
      }

      vg->vg_status |= VG_EXPORTED;
      strcpy ( vg->vg_name, vg->pv[0]->vg_name);

      if ( opt_v > 0) printf ( "%s -- exporting %s with %lu physical "
                               "and %lu logical volumes\n",
                               cmd, vg_name, vg->pv_cur, vg->lv_cur);
   
      lvm_dont_interrupt ( 0);

      if ( ( ret = vg_write_with_pv_and_lv ( vg)) < 0) {
         fprintf ( stderr, "%s -- ERROR %d storing volume group "
                           "data on disk(s)\n", cmd, ret);
      }

      if ( opt_v > 0) printf ( "%s -- removing special files for %s\n",
                               cmd, vg_name);
      vg_remove_dir_and_group_and_nodes ( vg_name);
   
      if ( ( ret = lvm_tab_vg_remove ( vg_name)) < 0) {
         fprintf ( stderr, "%s -- ERROR %d removing %s from %s\n",
                           cmd, ret, vg_name, LVMTAB);
         continue;
      }

      lvm_interrupt ();

      printf ( "%s -- %s", cmd, vg_name);
      if ( ret < 0) printf ( " NOT");
      printf ( " sucessfully exported\n");
      if ( ret < 0) continue;

   }

   LVM_UNLOCK ( 0);

   printf ( "\n");
   return 0;
}
