/*
 * Author: Heinz Mauelshagen, Germany
 *
 * April 1997
 * May 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    11/09/1997 - added lvmtab handling
 *    05/09/1998 - check for volume group beeing extendable
 *    05/16/1998 - added lvmtab checking
 *
 */

/*
 * TODO
 *
 *    - avoid pv_read_all_pv
 *
 */

#include <lvm_user.h>

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv) {
   int c = 0;
   int new_pv = 0;
   int np = 0;
   int np_sav = 0;
   int opt_A = 1;
   int opt_v = 0;
   int p = 0;
   int p1 = 0;
   int ret = 0;
   uint size = 0;
   ulong rest = 0;
   char *cmd = NULL;
   char *dummy = NULL;
   char *vg_name = NULL;
   vg_t *vg = NULL;
   pv_t **pv = NULL;
#ifdef DEBUG
   char *options = "A:dh?v";
#else
   char *options = "A:h?v";
#endif

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
         case 'A':
            if ( opt_A > 1) {
               fprintf ( stderr, "%s -- A option yet given\n\n", cmd);
               return 1;
            }
            if ( strcmp ( optarg, "y") == 0);
            else if ( strcmp ( optarg, "n") == 0) opt_A = 0;
            else {
               fprintf ( stderr, "%s -- invalid option argument %s\n\n",
                                 cmd, optarg);
               return 1;
            }
            break;

#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif
 
         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Volume group extend\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
                     "\t[-A y/n]\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-h/-?]\n"
                     "\t[-v]\n"
                     "\tVolumeGroupName\n"
                     "\tPhysicalDevicePath [PhysicalDevicePath...]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option yet given\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;
 
         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return 1;
      }
   }

   if ( optind == argc) {
      fprintf ( stderr, "%s -- please enter a volume group name"
                        " and a physical volume path\n\n", cmd);
      return 1;
   }
   vg_name = argv[optind];

   if ( pv_create_kdev_t ( vg_name) != 0) {
      fprintf ( stderr, "%s -- please enter a volume group name first\n\n",
                        cmd);
      return 1;
   }

   optind++;

   if ( optind == argc) {
      fprintf ( stderr, "%s -- please enter a physical volume path\n\n", cmd);
      return 1;
   }

   /* valid VG name? */
   if ( opt_v > 0) printf ( "%s -- checking volume group name %s\n",
                            cmd, vg_name);
   if ( vg_check_name ( vg_name) < 0) {
	  fprintf ( stderr, "%s -- invalid volume group name %s\n\n",
                cmd, vg_name);
	  return 1;
   } 

   if ( lvm_tab_vg_check_exist ( vg_name, &vg) != TRUE) {
      fprintf ( stderr, "%s -- volume group %s doesn't exist\n\n",
                        cmd, vg_name);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- checking for inactivity of volume group\n",
                            cmd);
   if ( vg_check_active ( vg_name) != TRUE) {
      fprintf ( stderr, "%s -- volume group %s must be active for "
                        "extension\n\n",
                        cmd, vg_name);
      return 1;
   }


   LVM_LOCK ( 0);
   LVM_CHECK_IOP;

   /* does VG exist? */
   if ( opt_v > 0) printf ( "%s -- checking volume group %s existence\n",
                            cmd, vg_name);
   if ( lvm_tab_vg_check_exist ( vg_name, NULL) != TRUE) {
	  fprintf ( stderr, "%s -- can't extend: %s doesn't exist\n\n",
                cmd, vg_name);
	  return 1;
   } 

   /* read complete VG */
   if ( opt_v > 0) printf ( "%s -- reading volume group %s data from disk(s)\n",
                            cmd, vg_name);
   if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name, &vg)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d: can't extend;"
                        " couldn't get volume group data of %s\n\n",
                        cmd, ret, vg_name);
      return 1;
   }

   if ( ! ( vg->vg_status & VG_EXTENDABLE)) {
      fprintf ( stderr, "%s -- volume group %s can't be extended "
                        "(see vgchange(8))\n\n",
                        cmd, vg_name);
      return 1;
   }

   printf ( "%s -- INFO: maximum logical volume size is %s\n",
            cmd, ( dummy = show_size ( LVM_LV_SIZE_MAX ( vg) / 2, LONG)));
   free ( dummy); dummy = NULL;


   if ( vg->pv_cur >= vg->pv_max) {
      fprintf ( stderr, "%s -- maximum physical volume count exceeded\n\n",
                 cmd);
      return 1;
   }

   /* read all PVs */
   if ( opt_v > 0) printf ( "%s -- reading data for all physical volumes "
                            "from disk(s)\n", cmd);
   if ( ( ret = pv_read_all_pv ( &pv, FALSE)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d: can't extend;"
                        " couldn't read physical volume data\n\n",
                cmd, ret);
      return 1;
   }


   /* check, if PVs are all defined and new
      and extend them in VG structures */
   if ( opt_v > 0) printf ( "%s -- checking physical volumes "
                            "for new\n", cmd);
   new_pv = np = 0;
   while ( vg->pv[np] != NULL && np < vg->pv_max) np++;
   np_sav = np;

   for ( ; optind < argc; optind++) {
      if ( pv_check_name ( argv[optind]) < 0) {
         fprintf ( stderr, "%s -- physical volume path %s is invalid\n\n",
                   cmd, argv[optind]);
         return 1;
      }

      for ( p = 0; pv[p] != NULL; p++) {
         if ( strcmp ( argv[optind], pv[p]->pv_name) == 0) {
            if ( opt_v > 0) printf ( "%s -- checking for maximum physical "
                                     "volume count of %s\n", cmd, vg_name);
            if ( np >= vg->pv_max) {
               fprintf ( stderr, "%s -- maximum physical volume count of "
                                 "volume group %s exceeded\n\n", cmd, vg_name);
               return 1;
            }

            if ( opt_v > 0) printf ( "%s -- getting size of physical "
                                     "volume %s\n", cmd, pv[p]->pv_name);
            if ( ( size = pv_get_size ( pv[p]->pv_name, NULL)) <= 0) {
               fprintf ( stderr, "%s -- ERROR %d getting size of "
                                 "physical volume %s\n\n",
                         cmd, size, pv[p]->pv_name);
               return 1;
 
            }

            if ( opt_v > 0) printf ( "%s -- checking physical volume %s "
                                     "against volume group %s\n",
                                     cmd, argv[optind], vg_name);
            for ( p1 = 0; vg->pv[p1] != NULL; p1++) {
               if ( strcmp ( argv[optind], vg->pv[p1]->pv_name) == 0) {
                  fprintf ( stderr, "%s -- physical volume %s allready belongs "
                            "to %s\n\n", cmd, argv[optind], vg_name);
                  return 1;
               }
            }

            if ( opt_v > 0) printf ( "%s -- checking for new physical "
                                     "volume %s\n", cmd, pv[p]->pv_name);
            if ( pv_check_new ( pv[p]) == FALSE) {
               fprintf ( stderr, "%s -- %s is no new physical volume\n\n",
	                         cmd, argv[optind]);
	       return 1;
            }

            /* setup PV and correct VG */
            pv[p]->vg_on_disk.base = vg->pv[0]->vg_on_disk.base;
            pv[p]->vg_on_disk.size = vg->pv[0]->vg_on_disk.size;
            pv[p]->pv_namelist_on_disk.base =
               vg->pv[0]->pv_namelist_on_disk.base;
            pv[p]->pv_namelist_on_disk.size =
               vg->pv[0]->pv_namelist_on_disk.size;
            pv[p]->lv_on_disk.base = vg->pv[0]->lv_on_disk.base;
            pv[p]->lv_on_disk.size = vg->pv[0]->lv_on_disk.size;
            pv[p]->pe_on_disk.base = vg->pv[0]->pe_on_disk.base;
            rest = pv[p]->pv_size - pv[p]->pe_on_disk.base / SECTOR_SIZE;
            pv[p]->pe_total = rest / vg->pe_size;
            rest -= ( pv[p]->pe_total * vg->pe_size);
            while ( rest * SECTOR_SIZE / sizeof ( disk_pe_t) <
                    pv[p]->pe_total) {
               rest += vg->pe_size;
               pv[p]->pe_total--;
            }
            pv[p]->pe_on_disk.size = ( pv[p]->pv_size -
                                       pv[p]->pe_total * vg->pe_size -
                                       pv[p]->pe_on_disk.base / SECTOR_SIZE)
                                     * SECTOR_SIZE;
            if ( LVM_DISK_SIZE ( pv[p]) % BLOCK_SIZE > 0)
               pv[p]->pe_on_disk.size -= SECTOR_SIZE;
            strcpy ( pv[p]->vg_name, vg_name);
            pv[p]->pv_dev = pv_create_kdev_t ( pv[p]->pv_name);
            pv[p]->pv_status = 0; /* bitfield */
            pv[p]->pv_allocatable = PV_ALLOCATABLE;
            pv[p]->pv_size = size;
            pv[p]->lv_cur = 0;
            pv[p]->pe_size = vg->pe_size;
            pv[p]->pe_total = ( pv[p]->pv_size -
                                ( pv[p]->pe_on_disk.base +
                                  pv[p]->pe_on_disk.size) / SECTOR_SIZE ) /
                                vg->pe_size;
            pv[p]->pe_allocated = 0;
            pv[p]->pe_stale = 0;
            if ( ( pv[p]->pe = malloc ( pv[p]->pe_total *
                                        sizeof ( disk_pe_t))) == NULL) {
               fprintf ( stderr, "%s -- malloc error in %s at line %d\n\n",
                                 cmd, __FILE__, __LINE__);
               return 1;
            }
            memset ( pv[p]->pe, 0, pv[p]->pe_total * sizeof ( disk_pe_t));
            vg->pv[np] = pv[p];
            vg->pv_act++;
            vg->pv_cur++;
            vg->pe_total += vg->pv[np]->pe_total;

            np++;
            new_pv++;
         }
      }
   }

   if ( np > np_sav) {
      p = 0;
      while ( vg->pv[p] != NULL) {
         vg->pv[p]->pv_number = p + 1;
         p++;
      }
   
      if ( opt_v > 0) printf ( "%s -- volume group %s will be extended with "
                               "%d new physical volumes\n",
   			                cmd, vg_name, new_pv);
   
      lvm_dont_interrupt ( 0);
   
      /* extend vg */
      np = np_sav;
      for ( ; vg->pv[np] != NULL; np++) {
         if ( opt_v > 0) printf ( "%s -- extending volume group %s with "
                                  "%s in kernel\n",
                                  cmd, vg_name, vg->pv[np]->pv_name);
         if ( ( ret = vg_extend ( vg_name, vg->pv[np], vg)) < 0) {
            fprintf ( stderr, "%s -- ERROR %d extending %s with %s in kernel\n",
                              cmd, ret, vg_name, vg->pv[np]->pv_name);
            for ( p = np_sav; p < np; p++) vg_reduce ( vg_name, vg->pv[p], vg);
            return 1;
         }
      }
   
      /* store vg on disk(s) */
      if ( opt_v > 0) printf ( "%s -- storing volume group data of "
                               "%s on disk(s)\n", cmd, vg_name);
      if ( ( ret = vg_write_with_pv_and_lv ( vg)) < 0) {
         fprintf ( stderr, "%s -- ERROR %d storing volume group data "
                           "of %s on disk(s)\n\n", cmd, ret, vg_name);
         return 1;
      }
   
      if ( opt_v > 0) printf ( "%s -- changing lvmtab\n", cmd);
      if ( vg_cfgbackup ( vg_name, LVMTAB_DIR, cmd, opt_v, vg) == 0 &&
           opt_A > 0) {
         printf ( "%s -- doing automatic backup of %s\n", cmd, vg_name);
         vg_cfgbackup ( vg_name, VG_BACKUP_DIR, cmd, opt_v, vg);
      }

      printf ( "%s -- volume group %s successfully extended\n\n", cmd, vg_name);
   } else {
      printf ( "%s -- no valid physical volumes to extend %s\n\n",
               cmd, vg_name);
   }

   lvm_interrupt ();
   LVM_UNLOCK ( 0);

   return 0;
}
