/*
 * tools/lib/pv_read.c
 *
 * Copyright (C) 1997 - 1999  Heinz Mauelshagen, Germany
 *
 * March-May,October-November 1997
 * May,August,November 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public License
 * along with GNU CC; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    08/08/1997 - pv_read(): added call to pv_flush to ensure uncached data
 *    10/08/1997 - pv_read_all_pv(): added check for correct partition type
 *    10/11/1997 - pv_read_all_pv(): stop processing > 4 partitions if first
 *                                   4 are not configured
 *    10/12/1997 - pv_read_all_pe_of_vg(): used new function pv_check_number
 *                                       and sorted PE pointers by pv_number
 *    10/15/1997 - pv_read: set *pv if ret == 0
 *    11/18/1997 - enhanced disk loop
 *    05/04/1998 - used lvm_check_dev() in pv_read()
 *    05/05/1998 - avoided members of a MD in pv_read_all_pv()
 *               - extended pv_read_all_pv() to search /dev/dsk too
 *                 for future use
 *    05/16/1998 - canceled wrong physical volume device names by
 *                 inserting the correct ones at load time
 *               - changed algorithm in pv_read_all_pv() to avoid
 *                 multiple MD entries according
 *    05/22/1998 - enhanced disk loop once more (see lvm_dir_cache())
 *    08/19/1998 - seperated disk and core structures in pv_read()
 *                 by using pv_copy_from_disk()
 *    09/06/1998 - implemented data layer in pv_read_pe() by using
 *                 new function pe_copy_from_disk()
 *    11/15/1998 - avoided pv_read'ing all partition special files, if
 *                 disk doesn't exist
 *    01/28/1999 - avoided calling pv_flush each time pv_read was called
 *                 by pv_read_already_red()
 *
 */

#include <liblvm.h>


/* internal function */
int pv_read_already_red ( char*);


int pv_read ( char *pv_name, pv_t **pv, int *open_errno) {
   int pv_handle = -1;
   int ret = 0;
   static pv_disk_t pv_this;
   struct stat stat_b;

#ifdef DEBUG
   debug ( "pv_read -- CALLED with %s\n", pv_name);
#endif

   if ( pv_name == NULL ||
        pv == NULL ||
        pv_check_name ( pv_name) < 0) return -LVM_EPARAM;

   if ( pv_read_already_red ( pv_name) == FALSE && pv_flush ( pv_name) < 0)
      return -LVM_EPV_READ_PV_FLUSH;

   if ( ( pv_handle = open ( pv_name, O_RDONLY)) != -1) {
      if ( fstat ( pv_handle, &stat_b) == 0) {
#ifdef DEBUG
         debug ( "pv_read -- going to read %s\n", pv_name);
#endif
         if ( read ( pv_handle, &pv_this,
                     sizeof ( pv_this)) != sizeof (pv_this))
            ret = -LVM_EPV_READ_READ;
         else if ( stat_b.st_rdev == 0)
            ret = -LVM_EPV_READ_RDEV;
         else if ( lvm_check_dev ( &stat_b, TRUE) == FALSE)
            ret = -LVM_EPV_READ_MAJOR;
      } else ret = -LVM_EPV_READ_STAT;
   } else {
      ret = -LVM_EPV_READ_OPEN;
      if ( open_errno != NULL) *open_errno = errno;
   }

   *pv = NULL;
   if ( ret == 0) {
      *pv = pv_copy_from_disk ( &pv_this);
      /* correct for imported/moved volumes */
      memset ( (*pv)->pv_name, 0, sizeof ( (*pv)->pv_name));
      strncpy ( (*pv)->pv_name,
                pv_create_name_from_kdev_t ( stat_b.st_rdev),
                sizeof ( (*pv)->pv_name) - 1);
      if ( strncmp ( (*pv)->id, LVM_ID, sizeof ( (*pv)->id)) != 0)
         ret = -LVM_EPV_READ_ID_INVALID;
      else if ( (*pv)->version != LVM_STRUCT_VERSION)
         ret = -LVM_EPV_READ_LVM_STRUCT_VERSION;
      else if ( system_id_check_exported ( (*pv)->system_id) == TRUE)
         ret = -LVM_EPV_READ_PV_EXPORTED;
      else if ( (*pv)->pv_dev == MD_MAJOR)
         ret = -LVM_EPV_READ_MD_DEVICE;
      (*pv)->pv_dev = stat_b.st_rdev;
   }

   if ( pv_handle != -1) close ( pv_handle);

#ifdef DEBUG
   debug ( "pv_read -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


int pv_read_all_pv ( pv_t ***pv, int reread) {
   int cache_size = 0;
   int n = 0;
   int np = 0;
   int p = 0;
   int p_sav1 = 0;
   int p_sav2 = 0;
   int pv_read_errno = 0;
   int ret = 0;
   int tst = 0;
   char *dev_name = NULL;
   static int first = 0;
   struct partition partition;
   pv_t **pv_this_sav = NULL;
   static pv_t **pv_this = NULL;
   pv_t *pv_tmp = NULL;
   dir_cache_t *dir_cache = NULL;

#ifdef DEBUG
   debug ( "pv_read_all_pv -- CALLED\n");
#endif

   if ( pv == NULL ||
        ( reread != TRUE && reread != FALSE)) {
      ret = -LVM_EPARAM;
      goto pv_read_all_pv_end;
   }

   *pv = NULL;

   if ( reread == TRUE) {
      if ( pv_this != NULL) {
         for ( p = 0; pv_this[p] != NULL; p++) free ( pv_this[p]);
         free ( pv_this);
         pv_this = NULL;
      }
      first = 0;
   }

   if ( first == 0) {
#ifdef DEBUG
         debug ( "pv_read_all_pv -- calling lvm_dir_cache\n");
#endif
      if ( ( cache_size = lvm_dir_cache ( &dir_cache)) < 1)
         return -LVM_EPV_READ_ALL_PV_LVM_DIR_CACHE;

      np = 0;
      for ( n = 0; n < cache_size; n++) {
         dev_name = dir_cache[n].dev_name;

#ifdef DEBUG
         debug ( "pv_read_all_pv -- calling pv_read with \"%s\"\n",
                  dev_name);
#endif
         if ( ( tst = open ( dev_name, O_RDONLY)) == -1) {
            if ( MAJOR ( dir_cache[n].st_rdev) != MD_MAJOR &&
                 MINOR ( dir_cache[n].st_rdev) % 16 == 0) {
               n += 15;
               continue;
            }
         } else close ( tst);

         pv_read_errno = 0;
         if ( ( ret = pv_read ( dev_name, &pv_tmp, &pv_read_errno)) == 0 ||
              ret == -LVM_EPV_READ_MD_DEVICE ||
              ret == -LVM_EPV_READ_PV_EXPORTED) {
            if ( pv_get_size ( dev_name, &partition) < 0) continue;
            if ( partition.sys_ind != 0 && 
                 partition.sys_ind != LVM_PARTITION) continue;
            if ( pv_check_volume ( dev_name, pv_tmp) == TRUE) {
#ifdef DEBUG
               debug ( "pv_read_all_pv: allocating for %s %s\n",
                         pv_tmp->pv_name, pv_tmp->vg_name);
#endif
               pv_this_sav = pv_this;
               if ( ( pv_this = realloc ( pv_this,
                                          ( np + 2) * sizeof ( pv_t*)))
                    == NULL) {
                  fprintf ( stderr, "realloc error in %s [line %d]\n",
                                    __FILE__, __LINE__);
                  for ( p = 0; pv_this_sav != NULL &&
                               pv_this_sav[p] != NULL; p++)
                     free ( pv_this_sav[p]);
                  ret = -LVM_EPV_READ_ALL_PV_MALLOC;
                  goto pv_read_all_pv_end;
               }
               if ( ( pv_this[np] = malloc ( sizeof ( pv_t))) == NULL) {
                  fprintf ( stderr, "malloc error in %s [line %d]\n",
                                    __FILE__, __LINE__);
                  for ( p = 0; pv_this[p] != NULL; p++)
                     free ( pv_this[p]);
                  free ( pv_this);
                  pv_this = NULL;
                  ret = -LVM_EPV_READ_ALL_PV_MALLOC;
                  goto pv_read_all_pv_end;
               }
               memcpy ( pv_this[np], pv_tmp, sizeof ( pv_t));
               np++;
               pv_this[np] = NULL;
            }
#ifdef DEBUG
            else {
               debug ( "pv_read_all_pv -- device %s NOT used\n", dev_name);
            }
#endif
         } 
#ifdef DEBUG
         else debug ( "pv_read_all_pv -- pv_read returned: %d\n", ret);
#endif
      }
      first = 1;
      ret = 0;
   }

#ifdef DEBUG
   debug ( "pv_read_all_pv -- avoiding multiple entries "
           "in case of MD; np: %d\n", np);
#endif

   /* FIXME: this doesn't work perfect in case of MD inactivity */
   /* check for MD and clear out SCSI, IDE, ... doubles */
   for ( p = 0; pv_this != NULL && pv_this[p] != NULL; p++) {
      if ( MAJOR ( pv_this[p]->pv_dev) == MD_MAJOR) {
         p_sav1 = p;
         for ( p = 0; pv_this[p] != NULL; p++) {
            if ( pv_this[p_sav1] != pv_this[p] && 
                 strcmp ( pv_this[p_sav1]->vg_name, pv_this[p]->vg_name) == 0) {
               if ( pv_this[p_sav1]->pv_dev != pv_this[p]->pv_dev) {
                  free ( pv_this[p]);
                  pv_this[p] = NULL;
                  if ( p < np) np--;
                  p_sav2 = p - 1;
                  p++;
                  for ( ;  pv_this[p] != NULL; p++) {
                     pv_this[p-1] = pv_this[p];
                     pv_this[p] = NULL;
                  }
                  p = p_sav2;
               }
            }
         }
         p = p_sav1;
      }
   }

   *pv = pv_this;

pv_read_all_pv_end:
#ifdef DEBUG
   debug ( "pv_read_all_pv -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


int pv_read_all_pv_of_vg ( char *vg_name, pv_t ***pv, int reread) {
   int p  = 0;
   int np = 0;
   int pv_number = 0;
   int ret = 0;
   static int first = 0;
   static char vg_name_sav[NAME_LEN] = { 0, };
   pv_t **pv_tmp = NULL;
   static pv_t **pv_this = NULL;
   
#ifdef DEBUG
   debug ( "pv_read_all_pv_of_vg -- CALLED with vg_name: \"%s\"\n", vg_name);
#endif

   if ( pv == NULL || vg_name == NULL ||
        ( reread != TRUE && reread != FALSE) ||
        vg_check_name ( vg_name) < 0) return -LVM_EPARAM;

   *pv = NULL;

   if ( strcmp ( vg_name_sav, vg_name) != 0) {
      strcpy ( vg_name_sav, vg_name);
      reread = TRUE;
   }

   if ( reread == TRUE) {
      if ( pv_this != NULL) {
         free ( pv_this);
         pv_this = NULL;
      }
      first = 0;
   }

   if ( first == 0) {
      if ( ( ret = pv_read_all_pv ( &pv_tmp, FALSE)) < 0) return ret;
   
      /* find all PVs of this VG */
   
      /* first pass to find highest pv_number */
      np = pv_number = 0;
      for ( p = 0; pv_tmp != NULL && pv_tmp[p] != NULL; p++) {
         if ( pv_check_consistency ( pv_tmp[p]) == 0 &&
              strcmp ( pv_tmp[p]->vg_name, vg_name) == 0) {
            if ( pv_number < pv_tmp[p]->pv_number)
               pv_number = pv_tmp[p]->pv_number;
            np++;
         }
      }
      if ( np == 0) return -LVM_EPV_READ_ALL_PV_OF_VG_NP;
      if ( pv_number != np) return -LVM_EPV_READ_ALL_PV_OF_VG_PV_NUMBER;

      if ( ( pv_this = malloc ( ( np + 1) * sizeof ( pv_t*))) == NULL) {
         fprintf ( stderr, "malloc error in %s [line %d]\n",
                           __FILE__, __LINE__);
         return -LVM_EPV_READ_ALL_PV_OF_VG_MALLOC;
      }
      memset ( pv_this, 0, ( np + 1) * sizeof ( pv_t*));
   
      /* second pass to fill array */
      np = 0;
      for ( p = 0; pv_tmp[p] != NULL; p++) {
#ifdef DEBUG
         debug ( "pv_read_all_pv_of_vg:  pv_name: %s  vg_name: %s  p: %d\n",
                  pv_tmp[p]->pv_name, pv_tmp[p]->vg_name, p);
#endif
         if ( strcmp ( pv_tmp[p]->vg_name, vg_name) == 0) {
#ifdef DEBUG
            debug ( "pv_read_all_pv_of_vg: %s[%lu] hit %s[%d]\n",
                      pv_tmp[p]->pv_name, pv_tmp[p]->pv_number-1,
                      pv_tmp[p]->vg_name, p);
#endif
            pv_this[pv_tmp[p]->pv_number-1] = pv_tmp[p];
            np++;
         }
      }
      pv_this[np] = NULL;

      /* Check for contiguous PV array */
      for ( p = 0; pv_this[p] != NULL; p++)
         if ( pv_this[p] == NULL && p < np)
            ret = -LVM_EPV_READ_ALL_PV_OF_VG_NP_SORT;

      first = 1;
   }

   if ( ret == 0) *pv = pv_this;

#ifdef DEBUG
   debug ( "pv_read_all_pv_of_vg -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


int pv_read_pe ( pv_t *pv, disk_pe_t **pe) {
   int pv_handle = -1;
   int ret = 0;
   uint size = 0;
   disk_pe_t *pe_this = NULL;

#ifdef DEBUG
   debug ( "pv_read_pe -- CALLED with %s and %lu\n",
            pv->pv_name, pv->pe_total);
#endif

   if ( pv == NULL || pe == NULL ||
        pv_check_name ( pv->pv_name) < 0) return -LVM_EPARAM;

   *pe = NULL;

   size = pv->pe_total * sizeof ( disk_pe_t);
   if ( size + pv->pe_on_disk.base > 
        LVM_DISK_SIZE ( pv)) return -LVM_EPV_READ_PE_SIZE;
   if ( ( pv_handle = open ( pv->pv_name, O_RDONLY)) == -1)
      ret = -LVM_EPV_READ_PE_OPEN;
   else if ( lseek ( pv_handle, pv->pe_on_disk.base, SEEK_SET) !=
             pv->pe_on_disk.base) ret = -LVM_EPV_READ_PE_LSEEK;
   else if ( ( pe_this = malloc ( size)) == NULL) {
      fprintf ( stderr, "malloc error in %s [line %d]\n",
                        __FILE__, __LINE__);
      ret = -LVM_EPV_READ_PE_MALLOC;
   } else {
      memset ( pe_this, 0, size);
      if ( read ( pv_handle, pe_this, size) != size)
         ret = -LVM_EPV_READ_PE_READ;
      else {
         *pe = pe_copy_from_disk ( pe_this, pv->pe_total);
      }
   }
#ifdef DEBUG
   debug ( "pv_read_pe -- ret: %d\n", ret);
#endif

   if ( pv_handle != -1) close ( pv_handle);
   if ( pe_this != NULL) free ( pe_this);

#ifdef DEBUG
   debug ( "pv_read_pe -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


int pv_read_all_pe_of_vg ( char *vg_name, disk_pe_t ***pe, int reread) {
   int p = 0;
   int pv_count = 0;
   int ret = 0;
   static int first = 0;
   static char vg_name_sav[NAME_LEN] = { 0, };
   pv_t **pv_tmp = NULL;
   disk_pe_t **pe_this = NULL;
   static disk_pe_t **pe_this_sort = NULL;

#ifdef DEBUG
   debug ( "pv_read_all_pe_of_vg -- CALLED\n");
#endif

   if ( vg_name == NULL ||
        vg_check_name ( vg_name) < 0 ||
        pe == NULL ||
        ( reread != TRUE && reread != FALSE)) return -LVM_EPARAM;

   *pe = NULL;

   if ( strcmp ( vg_name, vg_name_sav) != 0) {
      strcpy ( vg_name_sav, vg_name);
      reread = TRUE;
   }

   if ( reread == TRUE) {
      if ( pe_this != NULL) {
         for ( p = 0; pe_this[p] != NULL; p++) free ( pe_this[p]);
         free ( pe_this);
         pe_this = NULL;
      }
      first = 0;
   }

   if ( first == 0) {
      if ( ( ret = pv_read_all_pv_of_vg ( vg_name, &pv_tmp, FALSE)) < 0)
         return ret;
      pv_count = 0;
      for ( p = 0; pv_tmp[p] != NULL; p++) pv_count++;
#ifdef DEBUG
      debug ( "pv_read_all_pe_of_vg -- pv_count: %d\n", pv_count);
#endif
      if ( ( pe_this = malloc ( ( pv_count+1) * sizeof ( disk_pe_t*))) == NULL){
         fprintf ( stderr, "malloc error in %s [line %d]\n",
                           __FILE__, __LINE__);
         return -LVM_EPV_READ_ALL_PE_OF_VG_MALLOC;
      }
      if ( ( pe_this_sort =
             malloc ( ( pv_count+1) * sizeof ( disk_pe_t*))) == NULL) {
         fprintf ( stderr, "malloc error in %s [line %d]\n",
                           __FILE__, __LINE__);
         return -LVM_EPV_READ_ALL_PE_OF_VG_MALLOC;
      }
      for ( p = 0; pv_tmp[p] != NULL; p++) {
         if ( ( ret = pv_read_pe ( pv_tmp[p],
                                   &pe_this[p])) < 0) {
            return ret;
         }
#ifdef DEBUG
         debug ( "pv_read_all_pe_of_vg -- %s with %lu PE at address %X\n",
                  pv_tmp[p]->pv_name, pv_tmp[p]->pe_total, ( uint) &pe_this[p]);
#endif
      }
      pe_this[p] = NULL;

      if ( pv_check_number ( pv_tmp, p) < 0)
         return -LVM_EPV_READ_ALL_PE_OF_VG_PV_NUMBER;

      for ( p = 0; pe_this[p] != NULL; p++)
         pe_this_sort[pv_tmp[p]->pv_number-1] = pe_this[p];

#ifdef DEBUG
      if ( opt_d > 0) {
         printf ( "pv_read_all_pe_of_vg -- AFTER LOOP of pv_read_pe\n");
         for ( p = 0; pe_this[p] != NULL; p++)
            printf ( "pv_read_all_pe_of_vg -- %s with %u PE at address %X "
                     "for PV #%d\n",
                     pv_tmp[p]->pv_name,
                     pv_tmp[p]->pe_total,
                     ( uint) pe_this[p],
                     p);
      }
#endif
      first = 1;
   }

   free ( pe_this);

   *pe = pe_this_sort;

#ifdef DEBUG
   debug ( "pv_read_all_pe_of_vg -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


int pv_read_namelist ( char *pv_name, vg_t *vg, char **pv_namelist) {
   int pv_handle = -1;
   int ret = 0;
   static char *this_pv_namelist = NULL;

#ifdef DEBUG
   debug ( "pv_read_namelist -- CALLED\n");
#endif

   if ( this_pv_namelist != NULL) {
      free ( this_pv_namelist);
      this_pv_namelist = NULL;
   }

   if ( pv_name == NULL || pv_check_name ( pv_name) < 0 ||
        vg == NULL || vg_check_name ( vg->vg_name) < 0 ||
        vg->pv[0] == NULL || vg->pv[0]->pv_namelist_on_disk.base == 0 ||
        pv_namelist == NULL) return -LVM_EPARAM;

   if ( ( pv_handle = open ( pv_name, O_RDONLY)) == -1)
      ret = -LVM_EPV_READ_NAMELIST_OPEN;
   else if ( lseek ( pv_handle, vg->pv[0]->pv_namelist_on_disk.base, SEEK_SET) !=
             vg->pv[0]->pv_namelist_on_disk.base)
      ret = -LVM_EPV_READ_NAMELIST_LSEEK;
   else {
      if ( ( this_pv_namelist = malloc ( vg->pv_cur * NAME_LEN)) == NULL) {
         fprintf ( stderr, "malloc error in %s [line %d]\n",
                           __FILE__, __LINE__);
         ret = LVM_EPV_READ_NAMELIST_MALLOC;
      } else {
         memset ( this_pv_namelist, 0, vg->pv_cur * NAME_LEN);
         if ( read ( pv_handle, this_pv_namelist, vg->pv_cur * NAME_LEN) !=
              vg->pv_cur * NAME_LEN) {
            free ( this_pv_namelist);
            this_pv_namelist = NULL;
            ret = -LVM_EPV_READ_NAMELIST_READ;
         }
      }
   }

   if ( pv_handle != -1) close ( pv_handle);
   if ( ret == 0) *pv_namelist = this_pv_namelist;

#ifdef DEBUG
   debug ( "pv_read_namelist -- LEAVING with ret: %d\n");
#endif
   return ret;
}


int pv_read_already_red ( char *pv_name) {
   int i = 0;
   static int pv_count = 0;
   static char* pv_name_cache = NULL;
   char* pv_name_cache_sav = NULL;


   if ( pv_name == 0 || strlen ( pv_name) > NAME_LEN - 1) return FALSE;

   for ( i = 0; i < pv_count; i++)
      if ( strcmp ( pv_name, &pv_name_cache[i*NAME_LEN]) == 0) return TRUE;

   pv_name_cache_sav = pv_name_cache;
   if ( ( pv_name_cache = realloc ( pv_name_cache,
                                    ( pv_count + 1) * NAME_LEN)) == NULL) {
      fprintf ( stderr, "realloc error in %s [line %d]\n",
                        __FILE__, __LINE__);
      if ( pv_name_cache_sav != NULL) free ( pv_name_cache_sav);
      return FALSE;
   }

   memset ( &pv_name_cache[pv_count * NAME_LEN], 0, NAME_LEN);
   strcpy ( &pv_name_cache[pv_count * NAME_LEN], pv_name);
   pv_count++;

   return FALSE;
}
