/*
 * tools/lvdisplay.c
 *
 * Copyright (C) 1997 - 1999  Heinz Mauelshagen, Germany
 *
 * March 1997
 * May-June,August,September 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    04/29/1998 - avoided lv_create_kdev_t() using [lvm_tab_]lv_check_exist()
 *    05/16/1998 - added lvmtab checking
 *    06/07/1998 - added warning message for option -Z
 *    06/27/1998 - changed lvm_tab_* calling convention
 *                 changed to new lv_show_current_pe_text() calling convention
 *    09/07/1998 - used lv_status_byname() instead of obsolete
 *                 lv_status_with_pe()
 *
 */

#include <lvm_user.h>

char *cmd = NULL;

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv)
{
   int c = 0;
   int opt_D = 0;
   int opt_v = 0;
   int ret = 0;
   char *lv_name = NULL;
#ifdef DEBUG
   char *options = "dDh?v";
#else
   char *options = "Dh?v";
#endif
   char *vg_name = NULL;
   lv_t *lv = NULL;

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_d++;
            break;
#endif

         case 'D':
            if ( opt_D > 0) {
               fprintf ( stderr, "%s -- D option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_D++;
            break;

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Logical Volume Display\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-D]\n"
                     "\t[-h/?]\n"
                     "\t[-v[v]]\n"
                     "\tLogicalVolumePath [LogicalVolumePath...]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'v':
            if ( opt_v > 1) {
               fprintf ( stderr, "%s -- v option already given two times\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return LVM_EINVALID_CMD_LINE;
      }
   }

   CMD_MINUS_CHK;

   if ( argc == optind) {
      fprintf ( stderr, "%s -- please enter a logical volume path\n\n", cmd);
      return LVM_ELVDISPLAY_LV_MISSING;
   }

   LVM_LOCK ( 1);
   LVM_CHECK_IOP;

   for ( ;optind < argc; optind++) {
      lv_name = argv[optind];
      if ( opt_v > 1) printf ( "%s -- checking logical volume name \"%s\"\n",
                               cmd, lv_name);
      if ( ( ret = lv_check_name ( lv_name)) < 0) {
         fprintf ( stderr, "%s -- invalid logical volume name \"%s\"\n\n",
                   cmd, lv_name);
         continue;
      }
   
      vg_name = vg_name_of_lv ( lv_name);

      if ( opt_v > 1) printf ( "%s -- checking logical volume \"%s\" "
                               "existence\n", cmd, lv_name);
      if ( opt_D > 0) ret = lv_check_exist ( lv_name);
      else            ret = lvm_tab_lv_check_exist ( lv_name);
      if ( ret != TRUE) {
         fprintf ( stderr, "%s -- logical volume \"%s\" doesn't exist\n\n",
                           cmd, lv_name);
         continue;
      }
   
      if ( opt_v > 1) printf ( "%s -- checking logical volume activity "
                               "of \"%s\"\n", cmd, lv_name);
      if ( lv_check_active ( vg_name, lv_name) < 0 && opt_D == 0) {
         fprintf ( stderr, "%s -- logical volume \"%s\" is not active; "
                           "try -D\n\n", cmd, lv_name);
         continue;
      }
   
      if ( opt_v > 1) printf ( "%s -- getting logical volume data for \"%s\" "
                               "from ", cmd, lv_name);
      /* from disk(s) */
      if ( opt_D > 0) {
         if ( opt_v > 1) printf ( "disk (lvmtab)\n");
         if ( ( ret = lv_read_with_pe ( vg_name, lv_name, &lv)) < 0) {
            fprintf ( stderr, "%s -- ERROR %d: no VALID logical volume "
                              "\"%s\"\n\n", cmd, ret, lv_name);
            continue;
         }
         if ( opt_v > 1) printf ( "%s -- checking logical volume consistency\n",
                                  cmd);
      /* from kernel */
      } else {
         if ( opt_v > 1) printf ( "kernel\n");
         if ( ( ret = lv_status_byname ( vg_name, lv_name, &lv)) < 0) {
            if ( ret == -LVM_ELV_STATUS_INTERNAL_OPEN ||
                 ret == -ENXIO) {
               fprintf ( stderr, "%s -- logical volume \"%s\" "
                                 "doesn't exist\n\n",
                                 cmd, lv_name);
               continue;
            } else {
               fprintf ( stderr, "%s -- ERROR %d getting logical"
                                 " volume info of \"%s\"\n\n",
                         cmd, ret, lv_name);
               continue;
            }
         }
      }
   
      if ( lv == NULL) {
         fprintf ( stderr, "%s -- logical volume \"%s\" doesn't exist\n\n",
                           cmd, lv_name);
         continue;
      }

      if ( ( ret = lv_check_consistency ( lv)) < 0) {
         fprintf ( stderr, "%s -- ERROR %d: logical volume \"%s\" "
                           "is inconsistent\n\n", cmd, ret, lv_name);
         continue;
      }

      lv_show ( lv);
      if ( opt_v > 0) {
         printf ( "\n");
         lv_show_current_pe_text ( lv);
      }
      printf ( "\n");
   }

   LVM_UNLOCK ( 1);

   return 0;
}
