/*
 * tools/lvrename.c
 *
 * Copyright (C) 1997 - 2000  Heinz Mauelshagen, Germany
 *
 * April-June,September 1998
 * January,February,October 1999
 * February 2000
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with LVM; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    27/06/1998 - changed lvm_tab_* calling convention
 *    04/09/1998 - corrected some messages
 *    26/01/1999 - made volume group directory prefix a preprocessor option
 *    28/01/1999 - fixed command line bug
 *               - fixed logical volume path too long in command line
 *    16/02/1999 - changed to bew lv_create_node()
 *    06/10/1999 - implemented support for long options
 *    15/02/2000 - use lvm_error()
 *
 */

#include <lvm_user.h>

char *cmd = NULL;

#ifdef DEBUG
   int opt_d = 0;
#endif


int main ( int argc, char** argv) {
   int c = 0;
   int l = 0;
   int opt_A = 1;
   int opt_A_set = 0;
   int opt_v = 0;
   int par_count = 0;
   int ret = 0;
   char *lv_name_old = NULL;
   char *lv_name_new = NULL;
   char *prefix = NULL;
   char vg_name[NAME_LEN] = { 0, };
   char lv_name_old_buf[NAME_LEN] = { 0, };
   char lv_name_new_buf[NAME_LEN] = { 0, };
   vg_t *vg = NULL;
   struct stat stat_buf;

#ifdef DEBUG
   char *options = "A:dh?v";
#else
   char *options = "A:h?v";
#endif
   struct option long_options[] = {
      { "autobackup", required_argument, NULL, 'A'},
#ifdef DEBUG
      { "debug",      no_argument,       NULL, 'd'},
#endif
      { "help",       no_argument,       NULL, 'h'},
      { "verbose",    no_argument,       NULL, 'v'},
      { NULL, 0, NULL, 0}
   };

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt_long ( argc, argv, options,
                               long_options, NULL)) != EOF) {
      switch ( c) {
         case 'A':
	    opt_A_set++;
            if ( opt_A > 1) {
               fprintf ( stderr, "%s -- A option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            if ( strcmp ( optarg, "y") == 0);
            else if ( strcmp ( optarg, "n") == 0) opt_A = 0;
            else {
               fprintf ( stderr, "%s -- invalid option argument \"%s\"\n\n",
                                 cmd, optarg);
               return LVM_EINVALID_CMD_LINE;
            }
            break;

#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_d++;
            break;
#endif

         case 'h':
         case '?':
            printf ( "\n%s (IOP %d)\n\n%s -- Logical Volume Rename\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
                     "\t[-A/--autobackup y/n]\n"
#ifdef DEBUG
                     "\t[-d/--debug]\n"
#endif
                     "\t[-h/-?/--help]\n"
                     "\t[-v/--verbose]\n"
                     "\tOldLogicalVolumePath NewLogicalVolumePath /\n"
                     "\tVolumeGroupName OldLogicalVolumeName "
                     "NewLogicalVolumeName\n\n",
                     lvm_version, LVM_LIB_IOP_VERSION,  cmd, cmd);
            return 0;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return LVM_EINVALID_CMD_LINE;
      }
   }

   CMD_MINUS_CHK;
   CMD_CHECK_OPT_A_SET;


   /* check alternate command line syntax */
   par_count = argc - optind;
   if ( par_count != 2 && par_count != 3) {
      fprintf ( stderr, "%s -- invalid command line\n\n", cmd);
      return LVM_EINVALID_CMD_LINE;
   }

   if ( par_count == 2) {
      if ( strlen ( lv_name_old = argv[optind]) > NAME_LEN - 1) {
         fprintf ( stderr, "%s -- the old logical volume path is longer than "
                           "the maximum of %d!\n\n",
                           cmd, NAME_LEN - 1);
         return LVM_ELVRENAME_LV_NAME;
      }
      if ( strlen ( lv_name_new = argv[optind+1]) > NAME_LEN - 1) {
         fprintf ( stderr, "%s -- the old logical volume path is longer than "
                           "the maximum of %d!\n\n",
                           cmd, NAME_LEN - 1);
         return LVM_ELVRENAME_LV_NAME;
      }
   } else {
      if ( strncmp ( argv[optind], LVM_DIR_PREFIX,
                     strlen ( LVM_DIR_PREFIX)) == 0) prefix = "";
      else                                           prefix = LVM_DIR_PREFIX;

      if ( strlen ( prefix) + strlen ( argv[optind]) +
           strlen ( argv[optind+1] + 1) > NAME_LEN - 1) {
         fprintf ( stderr, "%s -- the old logical volume path is longer than "
                           "the maximum of %d!\n\n",
                           cmd, sizeof ( lv_name_old_buf));
         return LVM_ELVRENAME_LV_NAME;
      }
      sprintf ( lv_name_old_buf, "%s%s/%s%c",
                prefix, argv[optind], argv[optind+1], 0);
      if ( strlen ( prefix) + strlen ( argv[optind]) +
           strlen ( argv[optind+2] + 1) > NAME_LEN - 1) {
         fprintf ( stderr, "%s -- the new logical volume path is longer than "
                           "the maximum of %d!\n\n",
                           cmd, sizeof ( lv_name_new_buf));
         return LVM_ELVRENAME_LV_NAME;
      }
      sprintf ( lv_name_new_buf, "%s%s/%s%c",
                prefix, argv[optind], argv[optind+2], 0);
      lv_name_old = lv_name_old_buf;
      lv_name_new = lv_name_new_buf;
   }

   if ( opt_v > 0) printf ( "%s -- checking old logical volume name\n", cmd);
   if ( lv_check_name ( lv_name_old) < 0) {
      fprintf ( stderr, "%s -- invalid logical volume name \"%s\"\n\n",
                        cmd, lv_name_old);
      return LVM_ELVRENAME_LV_NAME;
   }

   strcpy ( vg_name, vg_name_of_lv ( lv_name_old));
   if ( opt_v > 0) printf ( "%s -- checking for existence of old "
                            "logical volume\n",
                            cmd);
   if ( lvm_tab_lv_check_exist ( lv_name_old) != TRUE) {
      fprintf ( stderr, "%s -- logical volume \"%s\" doesn't exist\n\n",
                        cmd, lv_name_old);
      return LVM_ELVRENAME_LV_CHECK_EXIST_OLD;
   }

   if ( opt_v > 0) printf ( "%s -- checking active status for existing "
                            "logical volume \"%s\"\n", cmd, lv_name_old);
   if ( lv_check_active ( vg_name, lv_name_old) == TRUE) {
      fprintf ( stderr, "%s -- can't rename active logical volume \"%s\"\n\n",
                        cmd, lv_name_old);
      return LVM_ELVRENAME_LV_CHECK_ACTIVE;
   }

   if ( opt_v > 0) printf ( "%s -- checking new logical volume name\n", cmd);
   if ( lv_check_name ( lv_name_new) < 0) {
      fprintf ( stderr, "%s -- invalid logical volume name \"%s\"\n\n",
                        cmd, lv_name_new);
      return LVM_ELVRENAME_LV_NAME;
   }

   if ( opt_v > 0) printf ( "%s -- checking for nonexistence of new "
                            "logical volume\n", cmd);
   if ( lvm_tab_lv_check_exist ( lv_name_new) == TRUE) {
      fprintf ( stderr, "%s -- logical volume \"%s\" already exists\n\n",
                        cmd, lv_name_new);
      return LVM_ELVRENAME_LV_CHECK_EXIST_NEW;
   }

   if ( lstat ( lv_name_new, &stat_buf) != -1) {
      lvm_show_filetype ( stat_buf.st_mode, lv_name_new);
      return LVM_ELVRENAME_LSTAT;
   }

   if ( lvm_tab_vg_check_exist ( vg_name, &vg) != TRUE) {
      fprintf ( stderr, "%s -- volume group \"%s\" doesn't exist\n\n",
                        cmd, vg_name);
      return LVM_ELVRENAME_VG_CHECK_EXIST;
   }

   if ( vg_check_active ( vg_name) != TRUE) {
      fprintf ( stderr, "%s -- volume group \"%s\" is inactive\n\n",
                        cmd, vg_name);
      return LVM_ELVRENAME_VG_CHECK_ACTIVE;
   }

   if ( strcmp ( vg_name, vg_name_of_lv ( lv_name_new)) != 0) {
      fprintf ( stderr, "%s -- volume group names are different\n\n", cmd);
      return LVM_ELVRENAME_VG_NAME_DIFFER;
   }


   LVM_CHECK_IOP;
   LVM_LOCK ( 0);

   if ( opt_v > 0) printf ( "%s -- reading data of volume group \"%s\"\n",
                            cmd, vg_name);
   if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name, &vg)) != 0) {
      fprintf ( stderr, "%s -- ERROR \"%s\" couldn't get data of volume "
                        "group \"%s\"\n\n",
                        cmd, lvm_error ( ret), vg_name);
      return LVM_ELVRENAME_VG_READ;
   }

   if ( ( l = lv_get_index_by_name ( vg, lv_name_old)) < 0) {
      fprintf ( stderr, "%s -- ERROR: couldn't get index of logical "
                        "volume \"%s\"\n\n", cmd, lv_name_old);
      return LVM_ELVRENAME_LV_GET_INDEX;
   }


   /* remove it in kernel */
   if ( opt_v > 0) printf ( "%s -- removing old logical volume VGDA "
                            "in kernel\n", cmd);
   if ( ( ret = lv_remove ( vg, vg->lv[l], lv_name_old)) < 0) {
      fprintf ( stderr, "%s -- ERROR \"%s\" removing VGDA for logical volume "
                        "\"%s\" from kernel\n\n",
                        cmd, lvm_error ( ret), lv_name_old);
      return LVM_ELVRENAME_LV_REMOVE_OLD;
   }

   /* change the logical volume name */
   strcpy ( vg->lv[l]->lv_name, lv_name_new);

   /* create it in kernel */
   if ( opt_v > 0) printf ( "%s -- creating new logical volume VGDA "
                            "in kernel\n", cmd);
   if ( ( ret = lv_create ( vg, vg->lv[l], lv_name_new)) < 0) {
      fprintf ( stderr, "%s -- ERROR \"%s\" creating VGDA for logical volume "
                        "\"%s\" in kernel\n\n",
                        cmd, lvm_error ( ret), lv_name_new);
      return LVM_ELVRENAME_LV_CREATE;
   }

   /* store it on disks */
   if ( opt_v > 0) printf ( "%s -- storing logical volume VGDA on disk(s)\n",
                            cmd);
   if ( ( ret = vg_write_with_pv_and_lv ( vg)) < 0) {
      fprintf ( stderr, "%s -- ERROR \"%s\" storing data of volume group \"%s\""
                        " on disks\n\n", 
                        cmd, lvm_error ( ret), vg_name);
      fprintf ( stderr, "%s -- removing new logical volume VGDA in kernel\n",
                cmd);
      if ( ( ret = lv_remove ( vg, vg->lv[l], lv_name_new)) < 0)
         fprintf ( stderr, "%s -- ERROR \"%s\" removing VGDA for logical "
                           "volume \"%s\" from kernel\n\n",
                           cmd, lvm_error ( ret), lv_name_new);
      return LVM_ELVRENAME_VG_WRITE;
   }

   unlink ( lv_name_old);

   if ( opt_v > 0) printf ( "%s -- creating device special file\n", cmd);
   if ( ( ret = lv_create_node ( vg->lv[l])) < 0) {
      if ( ret == -LVM_ELV_CREATE_MODE_MKNOD) {
         fprintf ( stderr, "%s -- logical volume node \"%s\" already "
                           "exists\n\n",
                           cmd, lv_name_new);
      } else {
         fprintf ( stderr, "%s -- ERROR \"%s\" creating logical volume "
                           "node \"%s\"\n\n",
                           cmd, lvm_error ( ret), lv_name_new);
      }
      return LVM_ELVRENAME_LV_CREATE_NODE;
   }


   if ( opt_v > 0) printf ( "%s -- changing lvmtab\n", cmd);
   if ( vg_cfgbackup ( vg_name, LVMTAB_DIR, opt_v, vg) == 0 &&
        opt_A > 0) {
      printf ( "%s -- doing automatic backup of volume group \"%s\"\n",
               cmd, vg_name);
      vg_cfgbackup ( vg_name, VG_BACKUP_DIR, opt_v, vg);
   } else {
      printf ( "%s -- WARNING: you don't have an automatic backup of \"%s\"\n",
               cmd, vg_name);
   }

   lvm_interrupt ();
   LVM_UNLOCK ( 0);


   printf ( "%s -- logical volume \"%s\" successfully renamed to \"%s\"\n\n",
            cmd, lv_name_old, lv_name_new);
   return 0;
}
