/* Copyright (c) 1987, 1988, 1991  Stanley T. Shebs. */
/* Copyright 1988 by Chris D. Peterson, MIT. */
/* Copyright (c) 1995 Michael J. Peters */
/* Many improvements by Tim Moore, University of Utah. */
/* This program may be used, copied, modified, and redistributed freely */
/* for noncommercial purposes, so long as this notice remains intact.  */

/* X11R5 Intrinsics and DrawingArea functions for the Motif */
/* version of XConq. */

#include <signal.h>

#include "config.h"
#include "misc.h"
#include "dir.h"
#include "period.h"
#include "side.h"
#include "unit.h"
#include "map.h"
#include "X11.h"
#include "global.h"

/* various bitmap definitions. */

#include "xconqlogo.icon"

#define mask_width 16
#define mask_height 16
static char mask_bits[] = {
   0xe0, 0x03,  0xd8, 0x0f,  0xb4, 0x19,  0x8a, 0x21,
   0x86, 0x61,  0x85, 0x41,  0x83, 0xc1,  0xff, 0xff,
   0xff, 0xff,  0x83, 0xc1,  0x82, 0xa1,  0x86, 0x61,
   0x84, 0x51,  0x98, 0x2d,  0xf0, 0x1b,  0xc0, 0x07};

#define curs_width 16
#define curs_height 16
static char curs_bits[] = {
   0xe0, 0x03,  0x98, 0x0c,  0x84, 0x10,  0x82, 0x20,
   0x82, 0x20,  0x81, 0x40,  0x81, 0x40,  0xff, 0x7f,
   0x81, 0x40,  0x81, 0x40,  0x82, 0x20,  0x82, 0x20,
   0x84, 0x10,  0x98, 0x0c,  0xe0, 0x03,  0x00, 0x00};

#define bomb1_width 32
#define bomb1_height 32
#define BH 32
static char bomb1_bits[] = {
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0xc0,  0x01, 0x00,  0x00, 0xf0,  0x07, 0x00,
   0x00, 0xf0,  0x07, 0x00,  0x00, 0xf8,  0x0f, 0x00,
   0x00, 0xf8,  0x0f, 0x00,  0x00, 0xfc,  0x0f, 0x00,
   0x00, 0xfe,  0x1f, 0x00,  0x00, 0xfe,  0x3f, 0x00,
   0x00, 0xfe,  0x3f, 0x00,  0x00, 0xfc,  0x1f, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00};

#define bomb2_width 32
#define bomb2_height 32
static char bomb2_bits[] = {
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0xf8,  0x01, 0x00,
   0x00, 0xf8,  0x07, 0x00,  0x00, 0xfc,  0x0f, 0x00,
   0x00, 0xfc,  0x1f, 0x00,  0x00, 0xfe,  0x1f, 0x00,
   0x00, 0xfe,  0x3f, 0x00,  0x00, 0xfc,  0x1f, 0x00,
   0x00, 0xfc,  0x0f, 0x00,  0x00, 0xf0,  0x07, 0x00,
   0x00, 0xf0,  0x03, 0x00,  0x00, 0xe0,  0x03, 0x00,
   0x00, 0xf0,  0x07, 0x00,  0x00, 0xf0,  0x07, 0x00,
   0x00, 0xf0,  0x0f, 0x00,  0x00, 0xf0,  0x0f, 0x00,
   0x00, 0xf8,  0x0f, 0x00,  0x00, 0xf8,  0x1f, 0x00,
   0x00, 0xf8,  0x1f, 0x00,  0x00, 0xfc,  0x1f, 0x00,
   0x00, 0xff,  0x7f, 0x00,  0xc0, 0xff,  0xff, 0x03,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00};

#define bomb3_width 32
#define bomb3_height 32
static char bomb3_bits[] = {
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0xfe, 0xe2,  0xa3, 0x3f,
   0x00, 0xfc,  0x1f, 0x00,  0x3c, 0xff,  0x7f, 0x7c,
   0x80, 0xff,  0xff, 0x00,  0xc0, 0xff,  0xff, 0x01,
   0xc0, 0xff,  0xff, 0x01,  0xe0, 0xff,  0xff, 0x03,
   0xe0, 0xff,  0xff, 0x03,  0xe0, 0xff,  0xff, 0x03,
   0xe0, 0xff,  0xff, 0x01,  0xe0, 0xff,  0xff, 0x49,
   0x82, 0xff,  0xff, 0x34,  0x14, 0xfe,  0x3f, 0x42,
   0xe2, 0xff,  0x9f, 0x34,  0x40, 0xfe,  0x3f, 0x41,
   0xbe, 0xfd,  0xdf, 0x1e,  0x00, 0xf8,  0x1f, 0x01,
   0xfe, 0xfd,  0xdf, 0x7f,  0x00, 0xfc,  0x1f, 0x00,
   0x00, 0xfc,  0x1f, 0x00,  0x00, 0xfc,  0x3f, 0x00,
   0x00, 0xfe,  0x1f, 0x00,  0x00, 0xfe,  0x3f, 0x00,
   0x00, 0xff,  0x3f, 0x00,  0xc0, 0xff,  0xff, 0x00,
   0xfc, 0xff,  0xff, 0x3f,  0xfe, 0xff,  0xff, 0x7f,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00};

#define bomb4_width 32
#define bomb4_height 32
static char bomb4_bits[] = {
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0xc0,  0x07, 0x00,
   0x00, 0x78,  0x10, 0x00,  0x00, 0x0f,  0x46, 0x00,
   0x80, 0x61,  0x81, 0x00,  0xc0, 0x1c,  0x00, 0x01,
   0x40, 0x02,  0x00, 0x00,  0x20, 0x01,  0x00, 0x00,
   0x20, 0x01,  0x00, 0x00,  0x20, 0x00,  0x00, 0x00,
   0x20, 0x00,  0x00, 0x00,  0x40, 0x00,  0x00, 0x00,
   0x80, 0x00,  0x00, 0x00,  0x00, 0x02,  0x00, 0x00,
   0x00, 0x02,  0x00, 0x00,  0x00, 0x04,  0x00, 0x00,
   0x00, 0x04,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x04,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x04,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x04,  0x00, 0x00,  0x00, 0x02,  0x00, 0x00,
   0x00, 0x01,  0x00, 0x00,  0xc0, 0x00,  0x00, 0x00,
   0x2c, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00,
   0x00, 0x00,  0x00, 0x00,  0x00, 0x00,  0x00, 0x00};

static char *fallbacks[] = {
  "*background:		Khaki",
  "*foreground:		#FFFFFFFFFFFF",
  "*FontList:		-misc-fixed-medium-r-normal--15-140-75-75-c-90-iso8859-1",
  "*Font:		-misc-fixed-medium-r-normal--15-140-75-75-c-90-iso8859-1",
  "*labelFontList:	-misc-fixed-medium-r-normal--15-140-75-75-c-90-iso8859-1",
  (char *)NULL
};

static XtResource SideRes[] = {
  {"iconFont", "IconFont", XtRFontStruct,
     sizeof(XFontStruct), XtOffsetOf(AppData, iconfont), XtRString, ICONFONT},
  {"unitFont", "UnitFont", XtRFontStruct,
     sizeof(XFontStruct), XtOffsetOf(AppData, unitfont), XtRString, UNITFONT},

  {"hexDisplayMode", "HexDisplayMode", "HexDisplayMode",
     sizeof(HexDisplayMode), XtOffsetOf(AppData, showmode), XtRString, "BothIcons"},

  {"ownColor", XtCBackground, XtRPixel,
     sizeof(Pixel), XtOffsetOf(AppData, owncolor), XtRString, "black"},
  {"alternateColor", XtCBackground, XtRPixel,
     sizeof(Pixel), XtOffsetOf(AppData, altcolor), XtRString, "blue"},
  {"differentColor", XtCBackground, XtRPixel,
     sizeof(Pixel), XtOffsetOf(AppData, diffcolor), XtRString, "maroon"},
  {"borderColor", XtCForeground, XtRPixel,
     sizeof(Pixel), XtOffsetOf(AppData, bdcolor), XtRString, "blue"},
  {"grayColor", XtCForeground, XtRPixel,
     sizeof(Pixel), XtOffsetOf(AppData, graycolor), XtRString, "light gray"},
  {"map.enemyColor", XtCForeground, XtRPixel,
     sizeof(Pixel), XtOffsetOf(AppData, enemycolor), XtRString, "red"},
  {"map.neutralColor", XtCForeground, XtRPixel,
     sizeof(Pixel), XtOffsetOf(AppData, neutcolor), XtRString, "light gray"},
  {"map.goodColor", XtCForeground, XtRPixel,
     sizeof(Pixel), XtOffsetOf(AppData, goodcolor), XtRString, "green"},
  {"map.badColor", XtCForeground, XtRPixel,
     sizeof(Pixel), XtOffsetOf(AppData, badcolor), XtRString, "red"},
};

char *routine_executing = "Program Start";

static XtAppContext app;
static XEvent event;
static XtIntervalId timer;
static bool rootcursor;		/* true if using parent window's cursor */
char *hexdispmodes[] = {
    "FullHex", "BorderHex", "TerrIcons", "BothIcons", NULL };

/* Initilize Intrinsics */

void init_Xt() {

    XtSetLanguageProc (NULL, NULL, NULL);
    XtToolkitInitialize();
    app = XtCreateApplicationContext();
    XtAppSetFallbackResources(app, fallbacks);
}

/* Display mode type converter */

Boolean CvtHexDisplayMode(Display *display, XrmValue *args, Cardinal *num_args, 
                       XrmValuePtr fromVal, XrmValuePtr toVal, XtPointer *closure_ret) {

    static HexDisplayMode i;

    for (i=0; hexdispmodes[i] != NULL; i++) {
        if (0==strcmp(fromVal->addr, hexdispmodes[i]))
            break;
    }
    if (hexdispmodes[i]!=NULL) {
        toVal->size = sizeof(HexDisplayMode);
        *(HexDisplayMode*)(toVal->addr) = (i);
        *closure_ret = (char *)True;
        return True;
    } else {
        XtDisplayStringConversionWarning(display, (char *)fromVal->addr,
            "HexDisplayMode");
        *closure_ret = False;
        return False;
    }
}

int open_display(Side *side, int argc, char *argv[]) {

    side->display = XtOpenDisplay(app, side->host, PROGRAMNAME, PROGRAMCLASS, NULL, 0, &argc, argv);
    if (side->display == NULL) {
        fprintf(stderr,"Error opening display %s\n", side->host);
        exit(1);
    }
    side->screen_num = DefaultScreen(side->display);

    if (Debug) {
        XSynchronize(side->display, True);
        printf("Synching the X server.\n");
    }

    side->top = XtVaAppCreateShell(PROGRAMNAME, PROGRAMCLASS, 
        applicationShellWidgetClass, side->display, 
        NULL);

    XtAppSetTypeConverter(app, XtRString, "HexDisplayMode", 
        (XtTypeConverter)CvtHexDisplayMode,
        NULL, 0, XtCacheAll, NULL);

    XtVaGetApplicationResources(side->top, &(side->appdata), SideRes, 
        XtNumber(SideRes), NULL, NULL, NULL);
                
    return 1;
}

Pixel white_color(Side *side) {

    return WhitePixel(side->display, XDefaultScreen(side->display));
}

Pixel black_color(Side *side) {

    return BlackPixel(side->display, XDefaultScreen(side->display));
}

void set_StaticGray(Side *side) {

    Pixel fg, bg;

    printf("Using StaticGray visual class with depth of 1\n");

    fg = black_color(side);
    bg = white_color(side);
    side->monochrome = TRUE;

    side->appdata.bgcolor = bg;
    side->appdata.owncolor = bg;
    side->appdata.altcolor = bg;
    side->appdata.diffcolor = bg; 

    side->appdata.fgcolor = fg;
    side->appdata.bdcolor = fg;
    side->appdata.graycolor = fg;
    side->appdata.enemycolor = fg;
    side->appdata.neutcolor = fg;
    side->appdata.goodcolor = fg;
    side->appdata.badcolor = fg;
}

/* Get a color set up and warn if not getting what was asked for. */

Pixel request_color(Side *side, char *colorname) {

    XrmValue    from;
    XrmValue    to;

    to.addr = (XtPointer)malloc(sizeof(Pixel));
    to.size = sizeof(Pixel);
    if (Debug) printf("Allocating %s\n", colorname);
    from.addr = colorname;
    from.size = strlen(colorname);
    if (XtConvertAndStore(side->top, XtRString, &from, XtRPixel, &to) == False) {
        XtDisplayStringConversionWarning(side->display, (char *)from.addr,
            XtRPixel);
        return white_color(side);
    }
    return *(Pixel*)to.addr;
}

void set_terrain_colors(Side *side) {

  int   t;

  if (!side->monochrome)
    for_all_terrain_types(t) {
        side->hexcolor[t] = request_color(side, ttypes[t].color);
    }
}

void change_bgcolor(Side *side, Pixel bgcolor) {

    if (side->monochrome)
        return;

    side->appdata.bgcolor = bgcolor;
    XtVaSetValues(side->map_area,
        XtNbackground, bgcolor,
        NULL);

    show_map(side);
    show_world(side);
}

/* Acquire a set of colors.  There are too many to make it feasible to */
/* customize them via .Xdefaults, so we don't even try.  If there aren't */
/* enough colors, drop into monochrome mode.  This doesn't take the window */
/* manager into account - it may have grabbed some color space. */

void init_colors(Side *side) {

    XVisualInfo visual_info;
    
    /* Visual classes: */
    char *visual_class[] = {
        "StaticGray",	/* 0 */
        "GrayScale",	/* 1 */
        "StaticColor",	/* 2 */
        "PseudoColor",	/* 3 */
        "TrueColor",	/* 4 */
        "DirectColor"	/* 5 */
    };
    int desired_class = 5;

    side->monochrome = FALSE;
    side->default_depth = DefaultDepth(side->display, side->screen_num);

    if (side->default_depth == 1) {
        set_StaticGray(side);
        return;
    }

    while (!XMatchVisualInfo(side->display, side->screen_num, side->default_depth,
        desired_class--, &visual_info)) {}
    desired_class++;

    if (desired_class < StaticColor) {
        printf("Could not obtain desired color visual\n");
        set_StaticGray(side);
        return;
    }

    printf("Display %s using %s visual class with depth of %d\n", 
        side->host, visual_class[desired_class], side->default_depth);

    set_terrain_colors(side);
}

/* The side color here reflects ally/neutral/enemy status. */

Pixel side_color(Side *side, Side *side2) {

    if (side == side2 || allied_side(side, side2)) return side->appdata.altcolor;
    if (enemy_side(side, side2)) return side->appdata.enemycolor;
    return side->appdata.neutcolor;
}

/* Since XCreatePixmapCursor() takes XColors and not pixel values we */
/* have to look the colors associated with the foreground and */
/* background pixel values up in the color table and pass them to */
/* XCreatePixmapCursor(). */

Cursor make_cursor(Display *dpy, Pixmap curs, Pixmap mask, 
    Pixel foreground, Pixel background, unsigned int x, unsigned int y) {

    XColor defs[2];

    defs[0].pixel = foreground;
    defs[1].pixel = background;
    XQueryColors(dpy, DefaultColormap(dpy, DefaultScreen(dpy)), defs, 2);
    return  XCreatePixmapCursor(dpy, curs, mask, &defs[0], &defs[1], x, y);
}

/* Try to load a bitmap of the given name, looking in both the current dir */
/* and the library dir. */

Pixmap load_bitmap(Side *side, char *name) {

    Pixmap pixmap;
    int status, hotx, hoty;
    unsigned int w, h;

    make_pathname(NULL, name, "b", spbuf);
    status = XReadBitmapFile(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), spbuf, &w, &h, &pixmap, &hotx, &hoty);
    if (status == BitmapSuccess) return pixmap;

    make_pathname(xconqlib, name, "b", spbuf);
    status = XReadBitmapFile(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), spbuf, &w, &h, &pixmap, &hotx, &hoty);
    if (status == BitmapSuccess) return pixmap;

    fprintf(stderr, "Bitmap name \"%s\" not found here or in \"%s\"!\n",
        name, xconqlib);
    return (-1);
}

/* Do misc setup thingies. */

/* Do the rigmarole to convert all those short arrays into X-approved */
/* Bitmaps.  Note that this has to be for *each* display separately (!) */
/* Also get the cursor shape.  If the hardware can't hack the desired */
/* cursor size, warn about it and just use the root window's cursor. */
/* 0x0 cursor specs seem to be don't cares - machine can handle any size */
/* X11 also needs gazillions of GCs, since we've got so many fonts and */
/* colors and bitmaps. */

init_misc(Side *side) {

    char *name;

    Pixel mask;
    XGCValues values;
    int u;
    unsigned int w, h;
    Pixmap mmask, mcurs;

    /* 
     * Get the name from the display unless the game is being restored
     * from a save file.
     */

    if (!midturnrestore &&
        (name = XGetDefault(side->display, PROGRAMNAME, "SideName")) != NULL)
        side->name = name;
    side->flashtimer = -1;

    mask = GCForeground | GCBackground;
    values.foreground = side->appdata.fgcolor;
    values.background = side->appdata.bgcolor;

    side->icongc = XCreateGC(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), mask, &values);
    side->unittextgc = XCreateGC(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), mask, &values);
    side->worldgc = XCreateGC(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), mask, &values);              

    side->flashgc = XCreateGC(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), mask, &values);
    XSetFunction(side->display, side->flashgc, GXinvert);

    side->unitgc = XCreateGC(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), mask, &values);
    mask = GCFillStyle | GCGraphicsExposures;
    values.fill_style = FillSolid;
    values.graphics_exposures = False;
    XChangeGC(side->display, side->unitgc, mask, &values);

    mask = GCForeground | GCBackground;
    values.foreground = side->appdata.bgcolor;
    values.background = side->appdata.fgcolor;

    side->invicongc = XCreateGC(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), mask, &values);

    XSetFont(side->display, side->icongc, side->appdata.iconfont->fid);
    XSetFont(side->display, side->unittextgc, side->appdata.unitfont->fid);
    XSetFont(side->display, side->invicongc, side->appdata.iconfont->fid);

    side->hw = side->appdata.iconfont->max_bounds.width;
    side->hh = side->appdata.iconfont->max_bounds.ascent +
               side->appdata.iconfont->max_bounds.descent;
    side->hch = side->hh - (int)(side->appdata.iconfont->max_bounds.width/4);
    if (Debug) printf(" icon: hw = %d; hh = %d; hch = %d\n", side->hw, side->hh, side->hch);
    side->uw = side->appdata.unitfont->max_bounds.width;
    side->uh = side->appdata.unitfont->max_bounds.ascent + 
               side->appdata.unitfont->max_bounds.descent;
    if (Debug) printf(" unit: uw = %d; uh = %d\n", side->uw, side->uh);
    side->mask_offx = (side->hw - side->uw)/2;
    side->mask_offy = (side->hh - side->uh)/2;

    if (Debug) printf("done fonts ...\n");

    side->logostipple = XCreatePixmapFromBitmapData(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), 
        xconqlogo_bits, xconqlogo_width, xconqlogo_height, 
        side->appdata.fgcolor, side->appdata.bgcolor, side->default_depth);
 
    mmask = XCreateBitmapFromData(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), mask_bits, mask_width, mask_height);
    mcurs = XCreateBitmapFromData(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), curs_bits, curs_width, curs_height);

    side->bombpics[0] = XCreateBitmapFromData(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), bomb1_bits, bomb1_width, bomb1_height);
    side->bombpics[1] = XCreateBitmapFromData(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), bomb2_bits, bomb2_width, bomb2_height);
    side->bombpics[2] = XCreateBitmapFromData(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), bomb3_bits, bomb3_width, bomb3_height);
    side->bombpics[3] = XCreateBitmapFromData(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), bomb4_bits, bomb4_width, bomb4_height);

    for_all_unit_types(u) {
        if (utypes[u].bitmapname && (strlen(utypes[u].bitmapname) > 0)) {
            side->unitslist[u].mask = load_bitmap(side, utypes[u].bitmapname);
            side->unitslist[u].pic = XCreatePixmap(side->display, 
                RootWindowOfScreen(XtScreen(side->top)), side->uw, side->uh, 
                side->default_depth);
            XCopyPlane(side->display, side->unitslist[u].mask,
                side->unitslist[u].pic, side->unitgc, 0, 0, side->uw, side->uh, 
                0, 0, 1);
        }
        else
            utypes[u].bitmapname = NULL;
    }

    if (Debug) printf("Bitmaps stored ...\n");

    rootcursor = FALSE;
    XQueryBestCursor(side->display, 
        RootWindowOfScreen(XtScreen(side->top)), curs_width, curs_height, &w, &h);
    if (Debug) printf("Best cursor size is %dx%d\n", w, h);
    if (w >= curs_width && h >= curs_height) {
        side->curs =
          make_cursor(side->display, mcurs, mmask, white_color(side),
                      black_color(side), 7, 7);

    } else {
        fprintf(stderr, "Warning: Can't have %dx%d cursors on \"%s\"!\n",
                curs_width, curs_height, side->host);
        fprintf(stderr, "Using default cursor...\n");
        rootcursor = TRUE;
    }
    if (Debug) printf("Cursor stored ...\n");
}

/* Do little things necesary to make it all go, in this case managing all */
/* the manager widgets. */

void fixup_windows(Side *side) {

    XtManageChild (side->left);
    XtManageChild (side->right);
    XtManageChild (side->main);
    XtRealizeWidget (side->top);

    if (!rootcursor) XDefineCursor(side->display, 
        XtWindow(side->map_area), side->curs);
}

/* Shut a single display down, but only if there's one to operate on. */
/* Hit the display slot, for safety. */

void close_display(Side *side) {

    XtCloseDisplay(side->display);
    side->display = NULL;
}

/* Trivial abstraction - sometimes other routines like to ensure all output */
/* actually on the screen. */

void flush_output(Side *side) {

    if (active_display(side)) XFlush(side->display);
}

void flush_events() {  

    while (XtAppPending(app)) {
        XtAppNextEvent (app, &event);
        XtDispatchEvent (&event);
    }
}

void process_events() {

    XtAppNextEvent (app, &event);
    XtDispatchEvent (&event);
}

/* Function to print debugging info when a segmentation violation or */
/* bus error occurs. */

#ifndef DEBUG

void program_crash(int sig, void *scp) {

  int i;
  static bool already_been_here = FALSE;

  close_displays();
  printf("Fatal error encountered. Signal %d\n", sig);
  printf("Routine executing %s\n", routine_executing);
  printf("Procedure stack (incomplete): \n");
  for (i = 0; i <= procedure_stack_ptr; i++)
    printf(" %s\n", procedure_executing[i]);
  if (already_been_here) exit(1);
  else {
    already_been_here = TRUE;
    write_savefile("emergency.save.xconq");
    if (sig != 2)
      abort();
    else exit(1);
  }
}

#endif


void hang_up(int sig, void *scp) {

  static bool already_been_here = FALSE;

  close_displays();
  if (already_been_here) exit(1);
  else {
    already_been_here = TRUE;
    write_savefile(SAVEFILE2);
    exit(1);
  }
}

/* Handlers for X catastrophes attempt to do a save first. */
/* Could be more friendly and check success of save, but can't do anything */
/* about it anyway... */

static int handle_x_error (Display *disp, XErrorEvent *evt) {

    static int num_errors = 0;
    char buf[BUFSIZE];

    XGetErrorText(disp, evt->error_code, buf, BUFSIZE);
    printf("\nX error on display %s: %s\n", DisplayString(disp), buf);
    if (++num_errors >= 10) {
        printf("\nX error: trying emergency save!\n");
        write_savefile(SAVEFILE);
        exit(1);
    }
}

static int handle_xio_error (Display *disp) {

    printf("\nX IO error on display %s: trying emergency save!\n",
         DisplayString(disp));
    write_savefile(SAVEFILE);
    exit(1);
}

/* Ignore ^C if humans in the game, do it otherwise, including when the */
/* last human player turns into a machine (this is called by option cmd). */
/* Attempts to be more clever seem to be bad news. */

void init_sighandlers() {

    if (numhumans > 0 && !Debug) {
        signal(SIGINT, SIG_IGN);
    }
    signal(SIGHUP, hang_up);
#ifndef DEBUG
    signal(SIGBUS, program_crash);
    signal(SIGSEGV, program_crash);
    signal(SIGFPE, program_crash);
    signal(SIGILL, program_crash);
    signal(SIGSYS, program_crash);
    signal(SIGINT, program_crash);
    signal(SIGQUIT, program_crash);
    signal(SIGTERM, program_crash);
#endif
    (void)XSetIOErrorHandler(handle_xio_error);
    (void)XSetErrorHandler(handle_x_error);
}

/* To get the traditional digital clock display, we need fixed-format digit */
/* printing.  There will *never* be an option for analog display... */

static void update_clocks(XtPointer client_data, XtIntervalId *id) {

    Side *side;

    global.elapsedtime += CLOCK_INTERVAL;
    sprintf(spbuf, "%02d:%02d:%02d",
            global.elapsedtime/360000,		/* hours */
            (global.elapsedtime/6000)%60,	/* minutes */
            (global.elapsedtime/100)%60);	/* seconds */

    for_all_sides(side) {

        if (active_display(side) && XtIsRealized(side->turnclock))
            show_turn_clock(side, spbuf);

        if (!side->lost && side->more_units) {
            side->timetaken += CLOCK_INTERVAL;
            if (!side->timedout && global.giventime && side->timetaken > global.giventime) {
                side->timedout = TRUE;
                time_up(side);
            }
            sprintf(side->timetakenbuf, "%02d:%02d:%02d",
                side->timetaken/360000,		/* hours */
                (side->timetaken/6000)%60,	/* minutes */
                (side->timetaken/100)%60);	/* seconds */
            if (active_display(side) && XtIsRealized(side->sideclock))
                show_side_clock(side, side->timetakenbuf);
        }
    }

    timer = XtAppAddTimeOut(app, (CLOCK_INTERVAL*10), update_clocks, NULL);
}

void stop_timer() {

    XtRemoveTimeOut(timer);    
}

void start_timer() {

    timer = XtAppAddTimeOut(app, (CLOCK_INTERVAL*10), update_clocks, NULL);
}

/* Beep the beeper! */

void beep(Side *side) {

    XBell(side->display, DefaultScreen(side->display));
}

/* General window clearing. */

void clear_window(Side *side, Window win) {

    XClearArea(side->display, win, 0,0, 0,0, False);
}

void clear_map_pix(Side *side) {

    XSetForeground(side->display, side->icongc, side->appdata.bgcolor);
    XFillRectangle(side->display, side->map_pix, side->icongc,
        0, 0, side->mw, side->mh);
}

/* This interfaces higher-level drawing decisions to the rendition of */
/* individual pieces of display. */

void draw_terrain_row(Side *side, int sx, int sy, char *buf, int len, Pixel color) {

    int sy_adj;

    sy_adj = sy + side->appdata.iconfont->max_bounds.ascent;
    XSetForeground(side->display, side->icongc, color);
    XDrawString(side->display, XtWindow(side->map_area), side->icongc, sx, sy_adj, buf, len);
    XDrawString(side->display, side->map_pix, side->icongc, sx, sy_adj, buf, len);
}

void draw_hex_icon(Side *side, int sx, int sy, char t, Pixel color) {

    char buf[2];
    int sy_adj;

    buf[0] = t;
    buf[1] = '\0';
    sy_adj = sy + side->appdata.iconfont->max_bounds.ascent;
    XSetForeground(side->display, side->icongc, color);
    XDrawString(side->display, XtWindow(side->map_area), side->icongc, sx, sy_adj, buf, 1);
    XDrawString(side->display, side->map_pix, side->icongc, sx, sy_adj, buf, 1);
}

/* Splash a unit image (either bitmap or font char) onto some window. */

void draw_unit_icon(Side *side, int sx, int sy, int u, Pixel color) {

    int sx_adj;
    int sy_adj;
    char buf[2];

    sx_adj = sx + side->mask_offx;

    if (utypes[u].bitmapname != NULL ) {
      sy_adj = sy + side->mask_offy;
      XSetForeground(side->display, side->unitgc, color);
      XSetClipMask(side->display, side->unitgc, side->unitslist[u].mask);
      XSetClipOrigin(side->display, side->unitgc, sx_adj, sy_adj);
      XFillRectangle(side->display, XtWindow(side->map_area), side->unitgc, sx_adj, sy_adj, side->uw, side->uh);
      XFillRectangle(side->display, side->map_pix, side->unitgc, sx_adj, sy_adj, side->uw, side->uh);
    }
    else {

        XSetForeground(side->display, side->unittextgc, color);
        buf[0] = utypes[u].uchar;
        buf[1] = '\0';
        sy_adj = sy + side->appdata.iconfont->max_bounds.ascent;
        XDrawString(side->display, XtWindow(side->map_area), side->unittextgc, sx_adj, sy_adj, buf, 1);
        XDrawString(side->display, side->map_pix, side->unittextgc, sx_adj, sy_adj, buf, 1);
    }
}

/* Draw the number of an unfriendly side (never called for own units). */

void draw_side_number(Side *side, int sx, int sy, int n, Pixel color) {

    int sy_adj;
    char buf[2];

    buf[0] = n%22;
    buf[0] += (buf[0]<10)?'0':('A' - 10);
    buf[1] = '\0';

    if (n >= 0 && n!=MAXSIDES) {
        XSetForeground(side->display, side->icongc, color);
        sy_adj = sy + side->appdata.iconfont->max_bounds.ascent;
        XDrawString(side->display, XtWindow(side->map_area), side->icongc, sx, sy_adj, buf, 1);
        XDrawString(side->display, side->map_pix, side->icongc, sx, sy_adj, buf, 1);
    }
}

/* Flash the player's screen in an unmistakable way. */

void invert_whole_map(Side *side) {

    /* GC needs to be set for inverted drawing */
    XDrawRectangle(side->display, XtWindow(side->map_area), side->flashgc, 0, 0, side->mw, side->mh);
}

/* Draw just one of the mushroom cloud shapes. */

void draw_mushroom(Side *side, int x, int y, int i) {

    int sx, sy;
    Pixel color;

    color = ((side->monochrome || i == 3) ? side->appdata.fgcolor : side->appdata.bgcolor);
    xform(side, unwrap(side, x, y), y, &sx, &sy);
    XSetForeground(side->display, side->unitgc, color);
    XSetClipMask(side->display, side->unitgc, side->bombpics[i]);
    XSetClipOrigin(side->display, side->unitgc, sx-BH/4, sy-BH/2);
    XFillRectangle(side->display, XtWindow(side->map_area), side->unitgc, sx-BH/4, sy-BH/2, BH, BH);
}

void draw_blast_icon(Side *side, int sx, int sy, char type, Pixel color) {

    int sy_adj;
    char buf[2];

    buf[0] = type;
    buf[1] = '\0';
    XSetForeground(side->display, side->icongc, color);
    sy_adj = sy + side->appdata.iconfont->max_bounds.ascent;
    XDrawString(side->display, XtWindow(side->map_area), side->icongc, sx, sy_adj, buf, 1);
}

static void flash_nap(XtPointer client_data, XtIntervalId *id) {

    Side *side = (Side *)client_data;

    side->flashtimer = -1;
    XDrawLine(side->display, XtWindow(side->map_area), side->flashgc,
            side->flashx1, side->flashy1, side->flashx2, side->flashy2);
    XDrawLine(side->display, XtWindow(side->map_area), side->flashgc,
            side->flashx1, side->flashy2, side->flashx2, side->flashy1);
}
 
/* Flash a pair of lines up, slow enough to draw the eye, but not so slow */
/* as to get in the way. */

void flash_position(Side *side, int sx, int sy, int tm) {

    if (tm > 0 && side->flashtimer < 0) {
        side->flashx1 = sx - 50 + side->hw/2;  side->flashy1 = sy + 50 + side->hch/2;
        side->flashx2 = sx + 50 + side->hw/2;  side->flashy2 = sy - 50 + side->hch/2;
        XDrawLine(side->display, XtWindow(side->map_area), side->flashgc,
            side->flashx1, side->flashy1, side->flashx2, side->flashy2);
        XDrawLine(side->display, XtWindow(side->map_area), side->flashgc,
            side->flashx1, side->flashy2, side->flashx2, side->flashy1);

        side->flashtimer = (long)XtAppAddTimeOut(app, tm, flash_nap, (XtPointer)side);
    }
}

/* The "cursor icon" is just a pair of special chars - nothing to do with */
/* X's notion of cursors. */

void draw_cursor_icon(Side *side, int sx, int sy) {

    int sy_adj;

    sy_adj = sy + side->appdata.iconfont->max_bounds.ascent;
    XDrawString(side->display, XtWindow(side->map_area), side->invicongc, sx, sy_adj, "[", 1);
    XSetForeground(side->display, side->icongc, side->appdata.fgcolor);
    XDrawString(side->display, XtWindow(side->map_area), side->icongc, sx, sy_adj, "]", 1);
}

/* Draw a single horizontal constant-color bar on the world map.  If part */
/* would not be drawn because of the map's obliqueness, cut it in two and */
/* wrap one of the pieces around. */

void draw_bar(Side *side, int x, int y, int len, Pixel color) {

    int sx1, sx2, sy, sww;

    w_xform(side, x, y, &sx1, &sy);
    w_xform(side, x + len, y, &sx2, &sy);
    sww = side->mm * world.width;
    XSetFillStyle(side->display, side->worldgc, FillSolid);
    XSetForeground(side->display, side->worldgc, color);
    if (sx1 < sww && sx2 >= sww) {
        XFillRectangle(side->display, XtWindow(side->world), side->worldgc,
                       sx1, sy, (unsigned int) sww - sx1, (unsigned int) side->mm);
        XFillRectangle(side->display, XtWindow(side->world), side->worldgc,
                       0, sy, (unsigned int) sx2 - sww, (unsigned int) side->mm);
    } else {
        sx1 %= sww;
        sx2 %= sww;
        XFillRectangle(side->display, XtWindow(side->world), side->worldgc,
                       sx1, sy, (unsigned int) sx2 - sx1, (unsigned int) side->mm);
    }
}

/* Invert the outline box on the world map.  This is a little tricky, */
/* because we want the lines to run through the middle of the world's */
/* hexes, and because the line drawn should not overlap (or the overlaps */
/* will be doubly inverted and look strange). */

void invert_box(Side *side, int vcx, int vcy) {

    int x1, y1, x2, y2, sx1, sy1, sx2, sy2, mm2;

    mm2 = side->mm/2;
    x1 = vcx - world.w2 + world.h2/2;  y1 = vcy - world.h2;
    x2 = vcx + world.w2 - world.h2/2;  y2 = vcy + world.h2;
    w_xform(side, x1, y1, &sx1, &sy1);
    w_xform(side, x2, y2, &sx2, &sy2);
    sx1 += mm2;  sy1 -= mm2;  sx2 += mm2;  sy2 += mm2;
    XSetFunction(side->display, side->worldgc, GXinvert);
    /* is this next call really necessary? */
    XSetLineAttributes(side->display, side->worldgc, 1, LineSolid, CapButt, JoinMiter);
    XDrawLine(side->display, XtWindow(side->world), side->worldgc, sx1, sy1, sx2, sy1);
    XDrawLine(side->display, XtWindow(side->world), side->worldgc, sx2, sy1-1, sx2, sy2+1);
    XDrawLine(side->display, XtWindow(side->world), side->worldgc, sx2, sy2, sx1, sy2);
    XDrawLine(side->display, XtWindow(side->world), side->worldgc, sx1, sy2+1, sx1, sy1-1);
    XSetFunction(side->display, side->worldgc, GXcopy);
}


void add_work_proc(Boolean (*work_proc)(XtPointer)) {

    XtAppAddWorkProc(app, work_proc, NULL);
}
